/*
 *
 *  Copyright (c) 2010 Christoph Keller <gri@nospam@not-censored.com>
 *
 *  This file is part of Web2SMS.
 *
 *  Web2SMS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Web2SMS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Web2SMS. If not, see <http://www.gnu.org/licenses/>
 *
 */

// Local includes
#include "messagewidget.hpp"
#include "providers/providerinterface.hpp"
#include "providers/contacts.h"
#include "providers/accountsmodel.hpp"
#include "ui_messagewidget.h"

// Global includes
#include <QtGui/QProgressDialog>
#include <QtGui/QMessageBox>

#ifdef Q_WS_MAEMO_5
# include <QtMaemo5/QMaemo5ValueButton>
# include <QtMaemo5/QMaemo5ListPickSelector>
#else // Q_WS_MAEMO_5
# include <QtGui/QComboBox>
#endif // Q_WS_MAEMO_5

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class MessageWidgetPrivate : public QObject
{
  Q_OBJECT

public:
  MessageWidgetPrivate()
  {}

  MessageWidget* self;

  // Properties
  Ui_MessageWidget ui;

#ifdef Q_WS_MAEMO_5
  QMaemo5ValueButton* chooseProviderButton;
  QMaemo5ListPickSelector selector;
#else // Q_WS_MAEMO_5
  QComboBox* chooseProviderComboBox;
#endif // Q_WS_MAEMO_5

  QPointer<ProviderInterface> provider;

  AccountsModel accountsModel;

  // Functions
  void init();

public slots:
  // Provider callbacks
  void onLoginReply(bool success);
  void onLogoutReply(bool success);
  void onError(const QString& message);
  void onFreeSmsCountChanged(int count);
  void onSendMessageReply(bool success, int messageId);

  // Ui callbacks
  void onChooseContactButtonClicked();
  void onSendButtonClicked();
  void onProviderSelectionChanged();
};

//////////////////////////////////////////////////////////////////////////

void MessageWidgetPrivate::init()
{
  // Setup the ui
  ui.setupUi(self);

  // Connect signals and slots
  connect(ui.chooseContactButton, SIGNAL(clicked()), this, SLOT(onChooseContactButtonClicked()));
  connect(ui.sendButton         , SIGNAL(clicked()), this, SLOT(onSendButtonClicked()));

#ifdef Q_WS_MAEMO_5
  // Setup the value button
  chooseProviderButton = new QMaemo5ValueButton(tr("Provider"), self);
  ui.providerPlaceholderLayout->addWidget(chooseProviderButton);

  // Setup the pick selector
  connect(&selector, SIGNAL(selected(QString)), this, SLOT(onProviderSelectionChanged()));
  selector.setModel(&accountsModel);

  // Set the pick selector
  chooseProviderButton->setPickSelector(&selector);
#else // Q_WS_MAEMO_5
  // Setup the combobox
  chooseProviderComboBox = new QComboBox(self);
  ui.providerPlaceholderLayout->addWidget(chooseProviderComboBox);

  // Setup the seleciton changes
  connect(chooseProviderComboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(onProviderSelectionChanged()));
  chooseProviderComboBox->setModel(&accountsModel);
  chooseProviderComboBox->setCurrentIndex(0);
#endif // Q_WS_MAEMO_5
}

//////////////////////////////////////////////////////////////////////////

void MessageWidgetPrivate::onLoginReply(bool success)
{
  // Enable or disable the send button
  ui.sendButton->setEnabled(success);

  //self->parentWidget()->setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);

  if ( !success )
    QMessageBox::critical(self, self->windowTitle(), tr("Login failed. Please check your account data."));
}

//////////////////////////////////////////////////////////////////////////

void MessageWidgetPrivate::onLogoutReply(bool success)
{
  Q_UNUSED(success);

  // Always disable the send button when logging out
  ui.sendButton->setDisabled(true);
}

//////////////////////////////////////////////////////////////////////////

void MessageWidgetPrivate::onError(const QString& message)
{
  QMessageBox::critical(self, self->windowTitle(), message);
}

//////////////////////////////////////////////////////////////////////////

void MessageWidgetPrivate::onFreeSmsCountChanged(int count)
{
  // Hide tha label when we have no clue how many free sms are left
  ui.statusLabel->setVisible( count != -1 );

  // Update the label text
  if ( count != -1 )
    ui.statusLabel->setText( tr("You have %n free SMS left", "", count) );
}

//////////////////////////////////////////////////////////////////////////

void MessageWidgetPrivate::onSendMessageReply(bool success, int messageId)
{
  Q_UNUSED(messageId);

  if ( !success )
    return;

  // Get the message text
  QString message = ui.messageText->toPlainText();

  // List all contacts and add the message to sent
  foreach(const ContactInfo& contact, ui.contactEdit->contacts())
  {
    QString name = contact.name();
    QString number = contact.number();
    QString uid = contact.uniqueId();

    // Really add the messages
    add_to_sent_sms(&name, &number, &uid, &message);
  }
}

//////////////////////////////////////////////////////////////////////////

void MessageWidgetPrivate::onChooseContactButtonClicked()
{
  QString dialogTitle = tr("Choose a contact");
  ContactInfoList contacts;

  // Let the user choose a number
  if ( get_contact_for_sms(&dialogTitle, &contacts) )
  {
    // Get the current text
    foreach(ContactInfo contact, contacts)
    {
      // Try to convert the number to an international format
      QString number = contact.number();
      if ( !make_number_international(number) )
      {
        QMessageBox::critical(self, tr("Wrong number format"),
            tr("%1's phone number \"%2\" could not be converted to an international format.").arg(contact.name()).arg(number));
      
        continue;
      }

      // Set the contacts' number back
      contact.setNumber(number);

      // Add the contact
      ui.contactEdit->addContact(contact);
    }
  }
}

//////////////////////////////////////////////////////////////////////////

void MessageWidgetPrivate::onSendButtonClicked()
{
  QProgressDialog* progressDialog = new QProgressDialog(self);
  progressDialog->setAttribute(Qt::WA_DeleteOnClose);
  progressDialog->setWindowTitle( tr("Sending SMS ...") );
  progressDialog->setRange(0, 0);
  progressDialog->show();

  connect(provider.data(), SIGNAL(sendMessageReply(bool, int)), this, SLOT(onSendMessageReply(bool, int)));
  connect(provider.data(), SIGNAL(sendMessageReply(bool, int)), progressDialog, SLOT(accept()));

  provider->sendMessage( ui.contactEdit->contacts(), ui.messageText->toPlainText() );
}

//////////////////////////////////////////////////////////////////////////

void MessageWidgetPrivate::onProviderSelectionChanged()
{
#ifdef Q_WS_MAEMO_5
  int index = selector.currentIndex();
#else // Q_WS_MAEMO_5
  int index = chooseProviderComboBox->currentIndex();
#endif // Q_WS_MAEMO_5

  // Get the current index
  QModelIndex currentIndex = accountsModel.index(index, 0);
  if ( !currentIndex.isValid() )
    return;

  // Get the account alias
  QString accountAlias = currentIndex.data(AccountsModel::AccountAliasRole).toString();
  if ( accountAlias.isEmpty() )
    return;

  // Give the button a shiny icon
#ifdef Q_WS_MAEMO_5
  chooseProviderButton->setIcon( qvariant_cast<QIcon>(currentIndex.data(Qt::DecorationRole)) );
#endif // Q_WS_MAEMO_5

  // Change the current provider
  AccountsPtr accounts = accountsModel.accounts();
  if ( accounts )
    self->setProvider( accounts->providerForAlias(accountAlias) );
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

MessageWidget::MessageWidget(QWidget* parent /* = 0 */, Qt::WindowFlags f /* = 0 */)
: QWidget(parent, f)
, d(new MessageWidgetPrivate)
{
  d->self = this;
  d->init();
}

//////////////////////////////////////////////////////////////////////////

MessageWidget::~MessageWidget()
{
  delete d;
}

//////////////////////////////////////////////////////////////////////////

void MessageWidget::setProvider(ProviderInterface* provider)
{
  // Something changed?
  if ( d->provider == provider )
    return;

  // Disconnect the old provider
  if ( d->provider )
  {
    disconnect(d->provider, SIGNAL(error(QString)), d, SLOT(onError(QString)));
    disconnect(d->provider, SIGNAL(loginReply(bool)), d, SLOT(onLoginReply(bool)));
    disconnect(d->provider, SIGNAL(logoutReply(bool)), d, SLOT(onLogoutReply(bool)));
    disconnect(d->provider, SIGNAL(freeSmsCountChanged(int)), d, SLOT(onFreeSmsCountChanged(int)));

    // Logout
    if ( d->provider->isLoggedIn() )
      d->provider->logout();
  }

  // Take the new provider
  d->provider = provider;

  // Connect the new provider
  if ( d->provider )
  {
    d->onFreeSmsCountChanged(-1);

    // Connect signals
    connect(d->provider, SIGNAL(error(QString)), d, SLOT(onError(QString)));
    connect(d->provider, SIGNAL(loginReply(bool)), d, SLOT(onLoginReply(bool)));
    connect(d->provider, SIGNAL(logoutReply(bool)), d, SLOT(onLogoutReply(bool)));
    connect(d->provider, SIGNAL(freeSmsCountChanged(int)), d, SLOT(onFreeSmsCountChanged(int)));

    // Show a dialog that we're logging out first
    if ( !d->provider->isLoggedIn() )
    {
      QProgressDialog* progressDialog = new QProgressDialog(this);
      progressDialog->setAttribute(Qt::WA_DeleteOnClose);
      progressDialog->setWindowTitle( tr("Logging in ...") );
      progressDialog->setRange(0, 0);
      progressDialog->show();

      // Close the information dialog and this window
      connect(provider, SIGNAL(loginReply(bool)), progressDialog, SLOT(accept()));

      // Login
      d->provider->login();
    }
  }
}

//////////////////////////////////////////////////////////////////////////

ProviderInterface* MessageWidget::provider() const
{
  return d->provider;
}

//////////////////////////////////////////////////////////////////////////

void MessageWidget::setAccounts(const AccountsPtr& accounts)
{
  d->accountsModel.setAccounts(accounts);

#ifdef Q_WS_MAEMO_5
  d->selector.setCurrentIndex(0);
#else // Q_WS_MAEMO_5
  d->chooseProviderComboBox->setCurrentIndex(0);
#endif // Q_WS_MAEMO_5
}

//////////////////////////////////////////////////////////////////////////

AccountsPtr MessageWidget::accounts() const
{
  return d->accountsModel.accounts();
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

#include "messagewidget.moc"
