/*
 *
 *  Copyright (c) 2010 Christoph Keller <gri@nospam@not-censored.com>
 *
 *  This file is part of Web2SMS.
 *
 *  Web2SMS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Web2SMS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Web2SMS. If not, see <http://www.gnu.org/licenses/>
 *
 */

// Local includes
#include "accountsdialog.hpp"
#include "createaccountdialog.hpp"
#include "accountsmodel.hpp"
#include "ui_accountsdialog.h"

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class AccountsDialogPrivate : public QObject
{
  Q_OBJECT

public:
  AccountsDialogPrivate()
  {}

  AccountsDialog* self;

  // Properties
  Ui_AccountsDialog ui;

  AccountsPtr accounts;
  AccountsModel accountsModel;

  // Functions
  void init();

  QString getSelectedAlias() const;

public slots:
  // Buttons
  void onNewButtonClicked();
  void onEditButtonClicked();
  void onDeleteButtonClicked();
  // List
  void onCurrentChanged(const QModelIndex& index);
};

//////////////////////////////////////////////////////////////////////////

void AccountsDialogPrivate::init()
{
  // Setup the ui
  ui.setupUi(self);

  // Connect signals and slots
  connect(ui.newButton, SIGNAL(clicked()), this, SLOT(onNewButtonClicked()));
  connect(ui.editButton, SIGNAL(clicked()), this, SLOT(onEditButtonClicked()));
  connect(ui.deleteButton, SIGNAL(clicked()), this, SLOT(onDeleteButtonClicked()));

  // Set the model
  ui.listView->setModel(&accountsModel);
  connect(ui.listView->selectionModel(), SIGNAL(currentChanged(QModelIndex,QModelIndex)), this, SLOT(onCurrentChanged(QModelIndex)));
}

//////////////////////////////////////////////////////////////////////////

QString AccountsDialogPrivate::getSelectedAlias() const
{
  // Get the current index
  QModelIndex index = ui.listView->currentIndex();

  // Nothing selected?
  if ( !index.isValid() )
    return QString();

  // Get the account configuration of the item
  return index.data(AccountsModel::AccountAliasRole).toString();
}

//////////////////////////////////////////////////////////////////////////

void AccountsDialogPrivate::onNewButtonClicked()
{
  // Show a dialog where the user sets the wanted alias settings
  CreateAccountDialog dialog(self);
  dialog.setAlias( tr("New Alias") );
  dialog.setProviderInfoList( accounts->providers() );

  // Dialog was successful?
  if ( dialog.exec() != QDialog::Accepted )
    return;

  // Create a temporary config which will be added
  AccountConfig config(dialog.alias(), dialog.providerInfo().providerId);
  accounts->addAccount(config);

  // Get the provider interface
  ProviderInterfacePtr ptr = accounts->providerForAlias(dialog.alias());
  if ( ptr )
    ptr->showAccountSettingsDialog(self);
}

//////////////////////////////////////////////////////////////////////////

void AccountsDialogPrivate::onEditButtonClicked()
{
  // Get the account configuration of the item
  QString alias = getSelectedAlias();

  if ( alias.isEmpty() )
    return;

  // Get a provider pointer
  ProviderInterfacePtr ptr = accounts->providerForAlias(alias);
  if ( ptr )
    ptr->showAccountSettingsDialog(self);
}

//////////////////////////////////////////////////////////////////////////

void AccountsDialogPrivate::onDeleteButtonClicked()
{
  // Get the selected configuration
  QString alias = getSelectedAlias();

  if ( alias.isEmpty() )
    return;

  // Remove the account
  accounts->removeAccount(alias);
}

//////////////////////////////////////////////////////////////////////////

void AccountsDialogPrivate::onCurrentChanged(const QModelIndex& index)
{
  // Enable or disable the edit and delete buttons
  ui.editButton->setEnabled( index.isValid() );
  ui.deleteButton->setEnabled( index.isValid() );
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

AccountsDialog::AccountsDialog(QWidget* parent /* = 0 */)
: QDialog(parent)
, d(new AccountsDialogPrivate)
{
  d->self = this;
  d->init();
}

//////////////////////////////////////////////////////////////////////////

AccountsDialog::~AccountsDialog()
{
  delete d;
}

//////////////////////////////////////////////////////////////////////////

void AccountsDialog::setAccounts(const AccountsPtr& accounts)
{
  // Something changed?
  if ( d->accounts == accounts )
    return;

  // Take the accounts
  d->accounts = accounts;
  d->accountsModel.setAccounts(accounts);
}

//////////////////////////////////////////////////////////////////////////

AccountsPtr AccountsDialog::accounts() const
{
  return d->accounts;
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

#include "accountsdialog.moc"
