/*
 *
 *  Copyright (c) 2010 Christoph Keller <gri@nospam@not-censored.com>
 *
 *  This file is part of Web2SMS.
 *
 *  Web2SMS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Web2SMS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Web2SMS. If not, see <http://www.gnu.org/licenses/>
 *
 */

// Local includes
#include "messagewidget.hpp"
#include "providers/providerinterface.hpp"
#include "contacts.h"
#include "ui_messagewidget.h"

// Global includes
#include <QtGui/QProgressDialog>
#include <QtGui/QMessageBox>

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class MessageWidgetPrivate : public QObject
{
  Q_OBJECT

public:
  MessageWidgetPrivate()
  : provider(NULL)
  {}

  MessageWidget* self;

  // Properties
  Ui_MessageWidget ui;

  ProviderInterface* provider;

  // Functions
  void init();

public slots:
  // Provider callbacks
  void onLoginReply(bool success);
  void onLogoutReply(bool success);
  //void onError(const QString& message);
  void onFreeSmsCountChanged(int count);

  // Ui callbacks
  void onChooseContactButtonClicked();
  void onSendButtonClicked();
};

//////////////////////////////////////////////////////////////////////////

void MessageWidgetPrivate::init()
{
  // Setup the ui
  ui.setupUi(self);

  // Connect signals and slots
  connect(ui.chooseContactButton, SIGNAL(clicked()), this, SLOT(onChooseContactButtonClicked()));
  connect(ui.sendButton         , SIGNAL(clicked()), this, SLOT(onSendButtonClicked()));
}

//////////////////////////////////////////////////////////////////////////

void MessageWidgetPrivate::onLoginReply(bool success)
{
  // Enable or disable the send button
  ui.sendButton->setEnabled(success);
}

//////////////////////////////////////////////////////////////////////////

void MessageWidgetPrivate::onLogoutReply(bool success)
{
  Q_UNUSED(success);

  // Always disable the send button when logging out
  ui.sendButton->setDisabled(true);
}

//////////////////////////////////////////////////////////////////////////

/*void MessageWidgetPrivate::onError(const QString& message)
{
  Q_UNUSED(message);
}*/

//////////////////////////////////////////////////////////////////////////

void MessageWidgetPrivate::onFreeSmsCountChanged(int count)
{
  // Hide tha label when we have no clue how many free sms are left
  ui.statusLabel->setVisible( count != -1 );

  // Update the label text
  if ( count != -1 )
    ui.statusLabel->setText( tr("You have %n free SMS left", "", count) );
}

//////////////////////////////////////////////////////////////////////////

void MessageWidgetPrivate::onChooseContactButtonClicked()
{
  QString dialogTitle = tr("Choose a contact");
  QString name, number, uid;

  // Let the user choose a number
  if ( get_contact_for_sms(&dialogTitle, &name, &number, &uid) )
  {
    // Try to get the number in a international format
    if ( !make_number_international(&number) )
    {
      QMessageBox::critical(self, tr("Wrong number format"),
          tr("%1's phone number \"%2\" could not be converted to an international format.").arg(name).arg(number));

      return; // There's no reason to continue
    }

    // Get the current text
    QString currentText = ui.contactEdit->text();

    // Append some separation if there's already some text
    if ( !currentText.isEmpty() )
      currentText += ", ";

    // Append the selected name and number to the field
    currentText += QString("%1 <%2>").arg(name).arg(number);

    // Set the text back to the control
    ui.contactEdit->setText(currentText);
  }
}

//////////////////////////////////////////////////////////////////////////

void MessageWidgetPrivate::onSendButtonClicked()
{
  QProgressDialog* progressDialog = new QProgressDialog(self);
  progressDialog->setAttribute(Qt::WA_DeleteOnClose);
  progressDialog->setWindowTitle( tr("Sending SMS ...") );
  progressDialog->setRange(0, 0);
  progressDialog->show();

  connect(provider, SIGNAL(sendMessageReply(bool, int)), progressDialog, SLOT(close()));

  provider->sendMessage( ui.contactEdit->text(), ui.messageText->toPlainText() );
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

MessageWidget::MessageWidget(QWidget* parent /* = 0 */)
: QWidget(parent)
, d(new MessageWidgetPrivate)
{
  d->self = this;
  d->init();
}

//////////////////////////////////////////////////////////////////////////

MessageWidget::~MessageWidget()
{
  delete d;
}

//////////////////////////////////////////////////////////////////////////

void MessageWidget::setProvider(ProviderInterface* provider)
{
  // Something changed?
  if ( d->provider == provider )
    return;

  // Disconnect the old provider
  if ( d->provider )
  {
    disconnect(d->provider, SIGNAL(loginReply(bool)), d, SLOT(onLoginReply(bool)));
    disconnect(d->provider, SIGNAL(logoutReply(bool)), d, SLOT(onLogoutReply(bool)));

    // Set a null pixmap
    d->ui.providerImage->setPixmap( QPixmap() );
  }

  // Take the new provider
  d->provider = provider;

  // Connect the new provider
  if ( d->provider )
  {
    // Connect signals
    connect(d->provider, SIGNAL(loginReply(bool)), d, SLOT(onLoginReply(bool)));
    connect(d->provider, SIGNAL(logoutReply(bool)), d, SLOT(onLogoutReply(bool)));
    connect(d->provider, SIGNAL(freeSmsCountChanged(int)), d, SLOT(onFreeSmsCountChanged(int)));

    // Change the ui to the new provider
    d->ui.providerImage->setPixmap( d->provider->pixmap() );
  }
}

//////////////////////////////////////////////////////////////////////////

ProviderInterface* MessageWidget::provider() const
{
  return d->provider;
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

#include "messagewidget.moc"
