/*
 *  Copyright 2010 Ruediger Gad
 *
 *  This file is part of vumeter.
 *
 *  vumeter is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  vumeter is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with vumeter.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "mainwindow.h"
#include "ui_mainwindow.h"

#include "pa_utils.h"

#include <QString>

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);

    setDbScale();

    ui->currentVolume->setNum(0);
    ui->maximumVolume->setNum(0);

    context_create();
    sources = get_source_names();
    ui->sourcesComboBox->addItems(*sources);
    connect(ui->sourcesComboBox, SIGNAL(activated(QString)), this, SLOT(sourceChanged(QString)));

    qDebug("Recording from: %s", sources->at(0).toLocal8Bit().constData());
    connect_to_vu_meter_stream(sources->at(0).toLocal8Bit().constData(), this);

    connect(this, SIGNAL(updateVuMeterSignal(float)), this, SLOT(updateValues(float)), Qt::QueuedConnection);
    connect(ui->maxLabel, SIGNAL(linkActivated(QString)), this, SLOT(resetMaxValue()));

    connect(ui->radioButtonDb, SIGNAL(clicked()), this, SLOT(setDbScale()));
    connect(ui->radioButtonLinear, SIGNAL(clicked()), this, SLOT(setLinearScale()));
}

MainWindow::~MainWindow()
{
    delete ui;
    context_stop();
}

void MainWindow::resetMaxValue(){
    maxVol = 0;
    ui->maximumVolume->setNum(maxVol);
}

void MainWindow::setLinearScale(){
    ui->vuMeterWidget->setDbUnit(false);
    resetMaxValue();
}

void MainWindow::setDbScale(){
    ui->vuMeterWidget->setDbUnit(true);
    resetMaxValue();
}

void MainWindow::sourceChanged(QString source){
    qDebug("New source selected: %s", source.toLocal8Bit().constData());
    disconnect_stream();

    resetMaxValue();
    updateValues(0.);

    connect_to_vu_meter_stream(source.toLocal8Bit().constData(), this);
}

void MainWindow::updateValues(float value){
    int vol = 0;

    if(ui->radioButtonDb->isChecked()){
        vol = 100 + pa_sw_volume_to_dB(pa_sw_volume_from_linear((double)value));
    }else if(ui->radioButtonLinear->isChecked()){
        vol = 100 * value;
    }

    if(vol < 0){
        vol = 0;
    }

    ui->vuMeterWidget->updateValue(vol);
    ui->currentVolume->setNum(vol);
    if(maxVol < vol){
        maxVol = vol;
        ui->maximumVolume->setNum(vol);
    }
}

void MainWindow::updateVuMeter(const float *data){
    float vol = *data;

    emit updateVuMeterSignal(vol);
}


