/*
 *  Copyright 2010 Ruediger Gad
 *
 *  This file is part of vumeter.
 *
 *  vumeter is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  vumeter is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with vumeter.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "volumemeterwidget.h"

#include <QPen>
#include <QRect>

VolumeMeterWidget::VolumeMeterWidget (QWidget *parent,
                                      const QColor &col1, int width1,
                                      const QColor &col2, int width2)
    : QGLWidget(QGLFormat(QGL::SampleBuffers), parent)
    , background(NULL)
    , currentValue(0)
    , defaultColor(Qt::darkBlue)
    , triggerIndicatorColor(Qt::darkYellow)
    , thickPen(col1, width1)
    , thinPen(col2, width2)
    , triggerValueHigh(0)
    , triggerValueLow(0)
{
    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    connect(this, SIGNAL(redraw()), this, SLOT(update()));
    connect(this, SIGNAL(clicked()), parent, SLOT(toggleFullScreen()));
}

VolumeMeterWidget::~VolumeMeterWidget ()
{
    delete background;
}

void VolumeMeterWidget::drawBackground (QPainter *painter)
{
    painter->drawImage(QRect(0, 0, w, h),
                       *background,
                       QRect(0, 0, w, h));
}

void VolumeMeterWidget::initBackground ()
{
    delete background;
    background = new QImage(size(), QImage::Format_ARGB32_Premultiplied);
}

void VolumeMeterWidget::initPainter (QPainter *painter)
{
    painter->setRenderHint(QPainter::Antialiasing, true);
}

void VolumeMeterWidget::mouseReleaseEvent (QMouseEvent */*event*/)
{
    emit clicked();
}

void VolumeMeterWidget::paintGL ()
{
    QPainter painter(this);

    h = height();
    w = width();

    initPainter(& painter);

    if (!background)
    {
        initBackground();
        prepareBackground();
    }

    drawBackground(& painter);

    if (triggerValueHigh > 0)
        drawTriggerIndicator(& painter, triggerValueHigh);
    if (triggerValueLow > 0)
        drawTriggerIndicator(& painter, triggerValueLow);

    draw(& painter);

    painter.end();
}

void VolumeMeterWidget::prepareBackground ()
{
}

void VolumeMeterWidget::resizeEvent (QResizeEvent */*event*/)
{
    delete background;
    background = NULL;
}

void VolumeMeterWidget::setColors (QColor lower, QColor mid, QColor upper)
{
    lowerColor = lower;
    midColor = mid;
    upperColor = upper;
}

void VolumeMeterWidget::setColorThresholds (int lower, int upper)
{
    lowerThreshold = lower;
    upperThreshold = upper;
}

void VolumeMeterWidget::setScale (int maxValue)
{
    maximumValue = maxValue;
    currentValue = 0;

    delete background;
    background = NULL;

    emit redraw();
}

void VolumeMeterWidget::setTriggerValues (int valueHigh, int valueLow)
{
    triggerValueHigh = valueHigh;
    triggerValueLow = valueLow;
    emit redraw();
}

void VolumeMeterWidget::setUseColoredScale (bool b)
{
    useColoredScale = b;
    emit redraw();
}

void VolumeMeterWidget::updateValue (int value)
{
    currentValue = value;
    emit redraw();
}
