#include "mainwindow.h"
#include "ui_mainwindow.h"

#include "pa_utils.h"

#include <QString>

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);

    setDbScale();
    ui->volumeBar->setValue(maxVol);
    ui->currentVolume->setNum(0);
    ui->maximumVolume->setNum(0);

    context_create();
    sources = get_source_names();
    ui->sourcesComboBox->addItems(*sources);
    connect(ui->sourcesComboBox, SIGNAL(activated(QString)), this, SLOT(sourceChanged(QString)));

    qDebug("Recording from: %s", sources->at(0).toLocal8Bit().constData());
    connect_to_vu_meter_stream(sources->at(0).toLocal8Bit().constData(), this);

    connect(this, SIGNAL(updateVuMeterSignal(float)), this, SLOT(updateValues(float)), Qt::QueuedConnection);
    connect(ui->maxLabel, SIGNAL(linkActivated(QString)), this, SLOT(resetMaxValue()));

    connect(ui->dBRadioButton, SIGNAL(clicked()), this, SLOT(setDbScale()));
    connect(ui->linearRadioButton, SIGNAL(clicked()), this, SLOT(setLinearScale()));
}

MainWindow::~MainWindow()
{
    delete ui;
    context_stop();
}

void MainWindow::resetMaxValue(){
    if(ui->linearRadioButton->isChecked()){
        maxVol = 0;
    }else if(ui->dBRadioButton->isChecked()){
        maxVol = MIN_DB;
    }

    ui->maximumVolume->setNum(maxVol);
}

void MainWindow::setDbScale(){
    ui->volumeBar->setMinimum(-70);
    ui->volumeBar->setMaximum(5);

    maxVol = MIN_DB;
    ui->maximumVolume->setNum(maxVol);
}

void MainWindow::setLinearScale(){
    ui->volumeBar->setMinimum(0);
    ui->volumeBar->setMaximum(100);

    maxVol = 0;
    ui->maximumVolume->setNum(maxVol);
}

void MainWindow::sourceChanged(QString source){
    qDebug("New source selected: %s", source.toLocal8Bit().constData());
    disconnect_stream();
    connect_to_vu_meter_stream(source.toLocal8Bit().constData(), this);
}

void MainWindow::updateValues(float value){
    int vol = 0;

    if(ui->linearRadioButton->isChecked()){
        vol = 100 * value;
    }else if(ui->dBRadioButton->isChecked()){
        vol = (int) pa_sw_volume_to_dB(pa_sw_volume_from_linear((double)value));
    }

    if(vol < MIN_DB){
        return;
    }

    ui->volumeBar->setValue(vol);
    ui->currentVolume->setNum(vol);
    if(maxVol < vol){
        maxVol = vol;
        ui->maximumVolume->setNum(vol);
    }
}

void MainWindow::updateVuMeter(const float *data){
    float vol = *data;

    if(vol >= 0 && vol < 1){
        emit updateVuMeterSignal(vol);
    }
}


