/* maemo vpnc-gui
 * 
 * Copyright (c) 2007 Michael "ScriptKiller" Arndt
 * http://scriptkiller.de/
 * <scriptkiller@gmx.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *
 */

/* IMPORTS */
#include <string.h>
#include "hildon.h"
#include <gtk/gtk.h>
#include <glib/glist.h>

#include "profiles_gui.h"
#include "profile.h"
#include "profiles_backend.h"
#include "profiles_gui_gconf.h"
#include "profiles_gui_vpnc.h"

/* (local) VARIABLES */
GtkWidget *name_entry;
static const gchar *profile_name;
static int profile_type;
static GtkWidget *vpnc_button;
static GtkWidget *openvpn_button;
static GtkListStore *box_store;
static GtkComboBox *box;

// TODO: combo_box_get_active_text is just for "text"-boxes!! get rid of it

/* LOCAL FUNCTIONS */

/**
 * (Re-)create the combo-box showing all profiles
 */
static void profiles_gui_recreate_list() {

	GtkTreeIter iter;

	gtk_list_store_clear(box_store);
	GList *l=profiles_backend_list_profiles();
	for(GList *li = l; li!= NULL; li = g_list_next(li)) {
		struct profile *p = li->data;
		gtk_list_store_append(box_store, &iter);
		gtk_list_store_set(box_store, &iter,
				0 /* column no */, p->name,
				-1);
		puts(p->name);
	}
}

/**
 * Callback for the rename-profile dialog
 * @param window dialog's window
 * @param response user-input
 * @param data unused
 */
static void profiles_rename_cb(GtkDialog *window,
		gint response,
		gpointer data) {


	gchar *reason;

	struct profile *pr=profiles_gui_get_selected_profile();

	// TODO: this is redundant code - consider refactoring
	profile_name=gtk_entry_get_text(GTK_ENTRY(name_entry));

	if(response == GTK_RESPONSE_OK) {


		if(strlen(profile_name) < 1) {

			GtkWidget *error=gtk_message_dialog_new(NULL,
					GTK_DIALOG_MODAL,
					GTK_MESSAGE_ERROR,
					GTK_BUTTONS_OK,
					"Please enter a name!");
			gtk_dialog_run(GTK_DIALOG(error));
			gtk_widget_destroy(error);

		}
		else if((reason=profiles_backend_profile_name_invalid(profile_name))!=NULL) {
			gchar *prepend="This name is invalid:\n";
			int length=strlen(prepend)+strlen(reason)+1;
			gchar *errormsg=g_malloc(length);
			g_strlcpy(errormsg, prepend, length);
			g_strlcat(errormsg, reason, length);
			GtkWidget *error=gtk_message_dialog_new(NULL,
					GTK_DIALOG_MODAL,
					GTK_MESSAGE_ERROR,
					GTK_BUTTONS_OK,
					errormsg);
			gtk_dialog_run(GTK_DIALOG(error));
			gtk_widget_destroy(error);

		}
		else if(profiles_backend_profile_name_exists(profile_name)) {
			GtkWidget *error=gtk_message_dialog_new(NULL,
					GTK_DIALOG_MODAL,
					GTK_MESSAGE_ERROR,
					GTK_BUTTONS_OK,
					"This name already exists!");
			gtk_dialog_run(GTK_DIALOG(error));
			gtk_widget_destroy(error);

		}
		else {
			/* Change the name */
			if(profiles_backend_rename_profile(pr, profile_name)) {
				gtk_widget_destroy(GTK_WIDGET(window));
				profiles_gui_recreate_list();
				
				GtkWidget *msg = gtk_message_dialog_new(NULL,
						GTK_DIALOG_MODAL,
						GTK_MESSAGE_INFO,
						GTK_BUTTONS_OK,
						"Profile renamed, please re-configure\nauto-connect if needed!");
				gtk_dialog_run(GTK_DIALOG(msg));
				gtk_widget_destroy(msg);



				/* select edited profile */
				profiles_gui_select_profile(pr->name);

				
			}
			else {

				GtkWidget *msg = gtk_message_dialog_new(NULL,
						GTK_DIALOG_MODAL,
						GTK_MESSAGE_ERROR,
						GTK_BUTTONS_OK,
						"Rename failed!");
				gtk_dialog_run(GTK_DIALOG(msg));
				gtk_widget_destroy(msg);

			}

		}

	}
	else {

		gtk_widget_destroy(GTK_WIDGET(window));
	}

}

/**
 * Callback for the new-profile dialog
 * @param window dialog's window
 * @param response user-input
 * @param data unused
 */
static void profiles_new_cb(GtkDialog *window,
    gint response,
    gpointer data) {

	gchar *reason;
	
  profile_name=gtk_entry_get_text(GTK_ENTRY(name_entry));

  if(response == GTK_RESPONSE_OK) {


    if(strlen(profile_name) < 1) {

      GtkWidget *error=gtk_message_dialog_new(NULL,
	  GTK_DIALOG_MODAL,
	  GTK_MESSAGE_ERROR,
	  GTK_BUTTONS_OK,
	  "Please enter a name!");
      gtk_dialog_run(GTK_DIALOG(error));
      gtk_widget_destroy(error);

    }
    else if((reason=profiles_backend_profile_name_invalid(profile_name))!=NULL) {
	  gchar *prepend="This name is invalid:\n";
	  int length=strlen(prepend)+strlen(reason)+1;
  	  gchar *errormsg=g_malloc(length);
	  g_strlcpy(errormsg, prepend, length);
	  g_strlcat(errormsg, reason, length);
	  GtkWidget *error=gtk_message_dialog_new(NULL,
	  GTK_DIALOG_MODAL,
	  GTK_MESSAGE_ERROR,
	  GTK_BUTTONS_OK,
	  errormsg);
      gtk_dialog_run(GTK_DIALOG(error));
      gtk_widget_destroy(error);
      
    }
    else if(profiles_backend_profile_name_exists(profile_name)){
      GtkWidget *error=gtk_message_dialog_new(NULL,
	  GTK_DIALOG_MODAL,
	  GTK_MESSAGE_ERROR,
	  GTK_BUTTONS_OK,
	  "This name already exists!");
      gtk_dialog_run(GTK_DIALOG(error));
      gtk_widget_destroy(error);

    }
    else {

      if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(vpnc_button))) {
	profile_type=PROFILE_TYPE_VPNC;
      }
      else if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(openvpn_button))) {
	profile_type=PROFILE_TYPE_OPENVPN;
      }
      else {
	puts("no type selected!");
      }


      gchar *profile_name_copy=g_strdup(profile_name);
      gtk_widget_destroy(GTK_WIDGET(window));

      
      if(profile_type==PROFILE_TYPE_VPNC) {
	profiles_gui_vpnc_new(profile_name_copy);
      } else if(profile_type==PROFILE_TYPE_OPENVPN) {

	GtkWidget *error=gtk_message_dialog_new(NULL,
	    GTK_DIALOG_MODAL,
	    GTK_MESSAGE_ERROR,
	    GTK_BUTTONS_OK,
	    "OpenVPN currently unimplemented, coming soon!");
	gtk_dialog_run(GTK_DIALOG(error));
	gtk_widget_destroy(error);


	//profiles_gui_openvpn_new();
      } else {
	      printf("Implement profile type %d in %s:%d!\n", profile_type, __FILE__,__LINE__);
      }
      
      g_free(profile_name_copy);
    }
  }
  else {
    gtk_widget_destroy(GTK_WIDGET(window));
  }

}



/* EXPORTS */

/**
 * Create and save a new profile or change an existing profile.
 * Shows an error dialog if creation fails.
 * 
 * Called by an external module in most cases.
 * 
 * Memory of specified profile must not be freed after successfully
 * calling this function!
 * 
 * @param p the profile
 * @return TRUE if successful
 */
gboolean profiles_gui_save_profile(struct profile *p) {

	if(!profiles_backend_save_profile(p)) {
		GtkWidget *msg = gtk_message_dialog_new(NULL,
				GTK_DIALOG_MODAL,
				GTK_MESSAGE_ERROR,
				GTK_BUTTONS_OK,
				"Creation failed!");
		gtk_dialog_run(GTK_DIALOG(msg));
		gtk_widget_destroy(msg);

		return FALSE;
	}
	profiles_gui_recreate_list();
	
	/* select edited profile */
	profiles_gui_select_profile(p->name);
	
	return TRUE;
}

/**
 * Get the combo-box widget.
 * @return a pointer to the combo-box-widget showing all profiles
 */
GtkComboBox *profiles_gui_get_combo_box() {
	return box;
}

/**
 * Select a profile in the GUI by name.
 * @param profile a pointer to a string containing the profile name
 */
void profiles_gui_select_profile(const gchar *profile) {
	gint idx=0;

	/* This should work as the order of profiles
	 * is retained, looks dangerous though!
	 */
	GList *l=profiles_backend_list_profiles();
	for(GList *li=l; li!=NULL; li=g_list_next(li)) {
		struct profile *p = li->data;
		if(strcmp(p->name, profile)==0) {
			gtk_combo_box_set_active(box, idx);
			break;
		}
		idx++;
	}
}

/**
 * De(-activate) profile-choose widget.
 * @param active TRUE to activate
 */
void profiles_gui_set_profile_chooser_active(gboolean active) {
	gtk_widget_set_sensitive(GTK_WIDGET(box), active);
}

/**
 * Get currently selected profile.
 * @return a pointer to the profile or NULL if none selected
 */
struct profile *profiles_gui_get_selected_profile() {
	
	gchar *text=gtk_combo_box_get_active_text(GTK_COMBO_BOX(box));

	/* save as last-used */
	profiles_gui_gconf_set_last_used_profile(text);

	
	// TODO: free text?!
	GList *l=profiles_backend_list_profiles();
	struct profile *pr=NULL;
	
	if(text==NULL) {
		puts("No profile selected");
		return NULL;
		
	}
	
	for(GList *li = l; li!= NULL; li = g_list_next(li)) {
		struct profile *p = li->data;
		if(strcmp(p->name, text)==0) {
			pr=p;
			break;
		}
	}
	
	if(pr==NULL) {
		puts("This definitly shouldn't happen ...!");
	}

	return pr;


}


/**
 * Initialize the profiles gui. This function must be called before
 * calling any other functions!
 */
void profiles_gui_init() {

	// TODO: document that mess!
	box_store=gtk_list_store_new(1, G_TYPE_STRING);
	box=GTK_COMBO_BOX(gtk_combo_box_new_with_model(GTK_TREE_MODEL(box_store)));
	
	GtkCellRenderer *c=gtk_cell_renderer_text_new();
	gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (box), c, FALSE);
	gtk_cell_layout_set_attributes (
			GTK_CELL_LAYOUT (box),
			c, "text", 0,
			NULL
			);

	profiles_backend_init();
	profiles_gui_recreate_list();
	
	profiles_gui_gconf_init();
	gchar *last_used=profiles_gui_gconf_get_last_used_profile();
	if(last_used!=NULL) {
		profiles_gui_select_profile(last_used);
		g_free(last_used);
	}
}



/**
 * Show dialog to edit currently selected profile, show an error
 * dialog if no profile is selected or other problems occur.
 */
void profiles_gui_edit_profile() {

	struct profile *pr=profiles_gui_get_selected_profile();

	if(pr==NULL) {
		GtkWidget *window=gtk_message_dialog_new(NULL,
				GTK_DIALOG_MODAL,
				GTK_MESSAGE_INFO,
				GTK_BUTTONS_OK,
				"No Profile selected!");
		gtk_dialog_run(GTK_DIALOG (window));
		gtk_widget_destroy(window);
		return;
	}

	
	if(pr->type == PROFILE_TYPE_VPNC) {
		profiles_gui_vpnc_edit(pr);
	}
	else {
		printf("Implement profile type %d in %s:%d!\n", pr->type, __FILE__, __LINE__);
	}

	/* select edited profile */
	profiles_gui_select_profile(pr->name);
}

/**
 * Show dialog to rename currently selected profile, show an error
 * dialog if no profile is selected or other problems occur.
 */
void profiles_gui_rename_profile() {

	struct profile *pr=profiles_gui_get_selected_profile();

	if(pr==NULL) {
		GtkWidget *window=gtk_message_dialog_new(NULL,
				GTK_DIALOG_MODAL,
				GTK_MESSAGE_INFO,
				GTK_BUTTONS_OK,
				"No Profile selected!");
		gtk_dialog_run(GTK_DIALOG (window));
		gtk_widget_destroy(window);
		return;
	}

	GtkWidget *window;
	window=gtk_dialog_new_with_buttons("Rename Profile",
			NULL,
			GTK_DIALOG_MODAL | GTK_DIALOG_NO_SEPARATOR,
			"Cancel",
			GTK_RESPONSE_CANCEL,
			"OK",
			GTK_RESPONSE_OK,
			NULL);

	name_entry=gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(name_entry), pr->name);

	GtkWidget *table=gtk_table_new(10, 2, FALSE);
	gtk_table_attach_defaults(GTK_TABLE(table),
			gtk_label_new("Name"),
			0, 1, 0, 1);
	gtk_table_attach_defaults(GTK_TABLE(table),
			name_entry,
			1, 2, 0, 1);

	gtk_box_pack_end(GTK_BOX(GTK_DIALOG(window)->vbox), table, TRUE, TRUE, 0);


	g_signal_connect(G_OBJECT(window),
			"response",
			G_CALLBACK(profiles_rename_cb), NULL);

	gtk_widget_show_all(window);

}

/**
 * Show dialog to create a new profile.
 */
void profiles_gui_new_profile() {


  GtkWidget *window;
  window=gtk_dialog_new_with_buttons("New Profile",
      				NULL,
				GTK_DIALOG_MODAL | GTK_DIALOG_NO_SEPARATOR,
				"Cancel",
				GTK_RESPONSE_CANCEL,
				"Next",
				GTK_RESPONSE_OK,
				NULL);

  name_entry=gtk_entry_new();
  vpnc_button=gtk_radio_button_new_with_label(NULL, "Cisco");
  GSList *button_group=gtk_radio_button_get_group(GTK_RADIO_BUTTON(vpnc_button));
  openvpn_button=gtk_radio_button_new_with_label(button_group, "OpenVPN");
  GtkWidget *type_buttongroup_vbox=gtk_vbox_new(FALSE, 10);

  gtk_box_pack_start(GTK_BOX(type_buttongroup_vbox), vpnc_button, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(type_buttongroup_vbox), openvpn_button, FALSE, FALSE, 0);

  GtkWidget *table=gtk_table_new(10, 2, FALSE);
  gtk_table_attach_defaults(GTK_TABLE(table),
                            gtk_label_new("Name"),
                            0, 1, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(table),
                            name_entry,
                            1, 2, 0, 1);


  gtk_table_attach_defaults(GTK_TABLE(table),
                            gtk_label_new("Type"),
                            0, 1, 1, 2);

  gtk_table_attach_defaults(GTK_TABLE(table),
                            type_buttongroup_vbox,
                            1, 2, 1, 2);


  gtk_box_pack_end(GTK_BOX(GTK_DIALOG(window)->vbox), table, TRUE, TRUE, 0);


  g_signal_connect(G_OBJECT(window),
                   "response",
                   G_CALLBACK(profiles_new_cb), NULL);

  gtk_widget_show_all(window);

}

/**
 * Show dialog to delete currently selected profile. Show an error
 * dialog if no profile is selected or anything else goes wrong.
 */
void profiles_gui_delete_profile() {


	struct profile *p=profiles_gui_get_selected_profile();
	
	GtkWidget *window;

	if(p==NULL) {
		window=gtk_message_dialog_new(NULL,
				GTK_DIALOG_MODAL,
				GTK_MESSAGE_INFO,
				GTK_BUTTONS_OK,
				"No Profile selected!");
		gtk_dialog_run (GTK_DIALOG (window));
		gtk_widget_destroy(window);
	}
	else {

		window = gtk_message_dialog_new(NULL,
				GTK_DIALOG_MODAL,
				GTK_MESSAGE_QUESTION,
				GTK_BUTTONS_YES_NO,
				"Really delete profile?");
		if (gtk_dialog_run (GTK_DIALOG (window)) == GTK_RESPONSE_YES) {
#ifdef DEBUG
			g_printf("Deleting ...\n");
#endif
			// TODO: delete by profile ptr!
			if(profiles_backend_delete_profile_by_name(p->name)) {
				GtkWidget *msg = gtk_message_dialog_new(NULL,
						GTK_DIALOG_MODAL,
						GTK_MESSAGE_INFO,
						GTK_BUTTONS_OK,
						"Profile deleted!");
				gtk_dialog_run(GTK_DIALOG(msg));
				gtk_widget_destroy(msg);

				/* update display */
				profiles_gui_recreate_list();

			}
			else {
				GtkWidget *msg = gtk_message_dialog_new(NULL,
						GTK_DIALOG_MODAL,
						GTK_MESSAGE_ERROR,
						GTK_BUTTONS_OK,
						"Deletion failed!");
				gtk_dialog_run(GTK_DIALOG(msg));
				gtk_widget_destroy(msg);


			}
		}
		gtk_widget_destroy(window);

	}

}
