/* maemo vpnc-gui
 * 
 * Copyright (c) 2007 Michael "ScriptKiller" Arndt
 * http://scriptkiller.de/
 * <scriptkiller@gmx.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *
 */

/* See also: http://maemo.org/development/documentation/how-tos/4-x/maemo_connectivity_guide.html
 */

/* IMPORTS */
#include <glib.h>
#include <conic.h>

#include "autoconnect_libconic.h"

/* (local) VARIABLES */
static ConIcConnection *connection;

/** function to call when we are connected to a connection */
static void (*connected_cb)(const gchar *)=NULL;


/* LOCAL FUNCTIONS */

/**
 * Handle connection events.
 */
static void my_connection_handler(ConIcConnection *connection,
                                  ConIcConnectionEvent *event,
                                  gpointer user_data)
{
    ConIcConnectionStatus status = con_ic_connection_event_get_status(event);
    ConIcConnectionError error;

    const gchar *iap_id = con_ic_event_get_iap_id(CON_IC_EVENT(event));
    const gchar *bearer = con_ic_event_get_bearer_type(CON_IC_EVENT(event));
    
    switch(status) {

        case CON_IC_STATUS_CONNECTED:
            g_debug("Hey, we are connected to IAP %s with bearer %s!", iap_id, bearer);
	    /* call back */
	    if(connected_cb!=NULL)
		    connected_cb(iap_id);
            break;

        case CON_IC_STATUS_DISCONNECTING:
            g_debug("We are disconnecting...");
            break;

        case CON_IC_STATUS_DISCONNECTED:
            g_debug("And we are disconnected. Let's see what went wrong...");
            error = con_ic_connection_event_get_error(event);

            switch(error) {

                case CON_IC_CONNECTION_ERROR_NONE:
                    g_debug("Libconic thinks there was nothing wrong.");
                    break;

                case CON_IC_CONNECTION_ERROR_INVALID_IAP:
                    g_debug("Invalid (non-existing?) IAP was requested.");
                    break;

                case CON_IC_CONNECTION_ERROR_CONNECTION_FAILED:
                    g_debug("Connection just failed.");
                    break;

                case CON_IC_CONNECTION_ERROR_USER_CANCELED:
                    g_debug("User canceled the connection attempt");
                    break;
            }
            break;

        default:
            g_debug("Unknown connection status received");
    }
}


/* EXPORTS */

void autoconnect_libconic_set_connected_cb(void (*callback)(const gchar *)) {
	connected_cb=callback;
}

/**
 * Initialize autonnection libconic. This must be called before
 * any functions.
 */
void autoconnect_libconic_init() {


	/* Create connection object */
	connection = con_ic_connection_new();

	/* Connect signal to receive connection events */
	g_signal_connect(G_OBJECT(connection), "connection-event", G_CALLBACK(my_connection_handler), NULL);

	/* Set automatic events, needed to listen to events that
	 * were not triggered by us.
	 */
	// TODO: automatic-event on 4.X?!
	g_object_set(G_OBJECT(connection), "automatic-connection-events", TRUE, NULL);
	g_object_set(G_OBJECT(connection), "automatic-event", TRUE, NULL);
		

	/* Request connection and check for the result */
	//if(!con_ic_connection_connect(connection, CON_IC_CONNECT_FLAG_NONE))
	//	g_warning("Request for connection failed");
}

/**
 * Create a list of all available connections on the device.
 * Results must be freed.
 * 
 * @return a g_list of g_alloced strings
 */
GList *autoconnect_libconic_list_available_connections() {

    /* ConIcConnection object named "connection" has already been created */
    GSList *saved_iaps = con_ic_connection_get_all_iaps(connection);
   
    GList *ret=NULL;
 
    g_debug("The following connections have been saved by the user:");
    for (GSList *iter = saved_iaps; iter != NULL; iter = g_slist_next(iter)) {
        
        /* Get IAP object and print some information about it */
        ConIcIap *iap = (ConIcIap *)iter->data;
        g_debug("Connection %s called '%s' using bearer %s",
                con_ic_iap_get_id(iap), con_ic_iap_get_name(iap),
                con_ic_iap_get_bearer_type(iap));
        
	ret=g_list_append(ret, g_strdup(con_ic_iap_get_name(iap)));

        /* We unref the IAP object as we are not going to use it anymore */
        g_object_unref(iap);
    }

    g_slist_free(saved_iaps);

    return ret;

}
