/* maemo vpnc-gui
 * 
 * Copyright (c) 2007 Michael "ScriptKiller" Arndt
 * http://scriptkiller.de/
 * <scriptkiller@gmx.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *
 */

/* IMPORTS */
#include "hildon.h"
#include <gtk/gtk.h>
#include <string.h>
#include <strings.h>

#include "profile.h"
#include "profiles_backend.h"
#include "autoconnect_gui.h"
#include "autoconnect_gconf.h"
#include "autoconnect_libconic.h"

static GtkWidget *window;
static GtkWidget *connection_view;
static GtkTreeModel *connection_model;

#define NO_PROFILE "/none/"

/** function to call when we are connected to a connection */
static void (*autoconnect_cb)(const gchar *)=0x0;

/**
 * Called when a connection has been established
 * @param conn_name name of the connection
 */
static void autoconnect_gui_connected_cb(const gchar *conn_name) {
	printf("autoconnect GUI: %s\n", conn_name);

	gchar *profile=
		autoconnect_gconf_get_profile_for_connection(conn_name);

	if(profile==NULL)
		return;
	
	printf("autoconnect GUI, profile: %s\n", profile);
	
	/* call back */
	if(autoconnect_cb!=NULL)
		autoconnect_cb(profile);

	g_free(profile);
}

static void autoconnect_gui_update_connectionlist() {
	
	/* clear */
	// TODO: free?
        gtk_list_store_clear(GTK_LIST_STORE(connection_model));
	

	GList *connections=autoconnect_libconic_list_available_connections();

	for(GList *li=connections; li!=NULL; li=g_list_next(li)) {

		/* get profile for connection */
		gchar *prof=
			autoconnect_gconf_get_profile_for_connection(li->data);
		if(prof==NULL) {
			prof=g_strdup("");
		}
			
		
		GtkTreeIter iter;
		gtk_list_store_append(GTK_LIST_STORE(connection_model), &iter);
		gtk_list_store_set(GTK_LIST_STORE(connection_model), &iter,
			0, li->data,
			1, prof,
			-1);
	}
	/* don't free data, just the list */
	g_list_free(connections);

}

/**
 * Show a list of available connection and let the user
 * configure a profile for each one.
 */
static void autoconnect_gui_edit_preferences_edit() {
	
	/* get selected connection */
	GtkTreeSelection *sel=
		gtk_tree_view_get_selection(GTK_TREE_VIEW(connection_view));
	
	if(sel==NULL) {
		GtkWidget *msg=gtk_message_dialog_new(NULL,
				GTK_DIALOG_MODAL,
				GTK_MESSAGE_INFO,
				GTK_BUTTONS_OK,
				"No connection selected!");
		gtk_dialog_run(GTK_DIALOG(msg));
		gtk_widget_destroy(msg);
		return;
	}
	GList *paths=gtk_tree_selection_get_selected_rows(sel, NULL);
	if(paths==NULL)
		return;
	
	/* don't bother with multiple selections */
	GtkTreePath *path=paths->data;
	

        GtkTreeIter i;
	
	GValue value={0, };
	
	/* set iterator to selection */
        gtk_tree_model_get_iter(connection_model, &i, path);
	gtk_tree_model_get_value(connection_model, &i,
			0 /* column no */, &value);

	const gchar *connection_name=g_value_get_string(&value);
	printf("selected: %s\n", connection_name);
	gchar *profile=autoconnect_gconf_get_profile_for_connection(connection_name);
	printf("profile: %s\n", profile);

        printf("selected: %s\n", connection_name);
	
	
	/* free selection */
	g_list_foreach(paths, (GFunc)gtk_tree_path_free, NULL);
	g_list_free(paths);
	

	
	GtkListStore *box_model;
	GtkWidget *box_view;

	/* create model as a one-column list with string-values */
	box_model=gtk_list_store_new(1, G_TYPE_STRING);

	/* create view as combo-box */
	box_view=gtk_combo_box_new_with_model(GTK_TREE_MODEL(box_model));

	/* create cell renderer for box */
	GtkCellRenderer *renderer=gtk_cell_renderer_text_new();
	gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(box_view), renderer, FALSE);
	gtk_cell_layout_set_attributes(
			GTK_CELL_LAYOUT(box_view),
			renderer, "text", 0,
			NULL
			);

	/* seems like widget has to be shown before anything can be added
	 * to model?!
	 */
	gtk_widget_show(GTK_WIDGET(box_view));
	GtkTreeIter iter;
	
	gtk_list_store_append(GTK_LIST_STORE(box_model), &iter);
	gtk_list_store_set(GTK_LIST_STORE(box_model), &iter,
			0 /* column no */, NO_PROFILE,
			-1);

	/* index of currently selected profile */
	gint selected_idx=0; /* start with 0 - none */
	gint idx=1; /* real entries begin with 1 */
	
	GList *l=profiles_backend_list_profiles();
	for(GList *li = l; li!= NULL; li = g_list_next(li)) {
                struct profile *p = li->data;
                gtk_list_store_append(box_model, &iter);
                gtk_list_store_set(box_model, &iter,
                                0 /* column no */, p->name,
                                -1);
		if((profile != NULL) && (strcmp(p->name, profile)==0))
			selected_idx=idx;
		idx++;
        }

	/* preselect current entry */
	gtk_combo_box_set_active(GTK_COMBO_BOX(box_view), selected_idx);
	
	GtkWidget *selector=gtk_dialog_new_with_buttons
		("Select Profile",
		 NULL,
		 GTK_DIALOG_MODAL | GTK_DIALOG_NO_SEPARATOR,
		 "Cancel",
		 GTK_RESPONSE_CANCEL,
		 "OK",
		 GTK_RESPONSE_OK,
		 NULL);


	gtk_box_pack_end(GTK_BOX(GTK_DIALOG(selector)->vbox),
			 box_view, TRUE, TRUE, 0);

	if(gtk_dialog_run (GTK_DIALOG (selector)) == GTK_RESPONSE_OK) {

		// TODO: hacky and ugly ....
		gchar *profile_name=gtk_combo_box_get_active_text(GTK_COMBO_BOX(box_view));
		printf("selected profile: %s\n", profile_name);

		if(strcmp(profile_name, NO_PROFILE)==0) {
			g_free(profile_name);
			profile_name=NULL;
		}
		
		/* save changes */
		if(!autoconnect_gconf_set_profile_for_connection(
					connection_name,
					profile_name)) {
			GtkWidget *error=gtk_message_dialog_new(NULL,
					GTK_DIALOG_MODAL,
					GTK_MESSAGE_ERROR,
					GTK_BUTTONS_OK,
					"Could not set value in gconf!");
			gtk_dialog_run(GTK_DIALOG(error));
			gtk_widget_destroy(error);

		}

		if(profile_name!=NULL)
			g_free(profile_name);


		/* update widget */
		autoconnect_gui_update_connectionlist();
		
	}

	/* frees connection_name */
	g_value_unset(&value);
	

	if(profile!=NULL) {
		g_free(profile);
	}
	
	gtk_widget_destroy(selector);
}


static void autoconnect_gui_edit_preferences_response(GtkDialog * window,
						      gint response,
		      				      gpointer data)
{
	if(response == GTK_RESPONSE_OK)
		autoconnect_gui_edit_preferences_edit();
	else
		gtk_widget_destroy(GTK_WIDGET(window));

}

/* EXPORTS */
void autoconnect_gui_set_autoconnect_cb(void (*callback)(const gchar *)) {
	autoconnect_cb=callback;
}


/**
 * Initialize the autoconnect gui. Must be called before
 * any other function.
 */
void autoconnect_gui_init() {

	/* init gconf backend */
	autoconnect_gconf_init();

	/* init libconic backend */
	autoconnect_libconic_init();

	/* register connected callback */	
	autoconnect_libconic_set_connected_cb(autoconnect_gui_connected_cb);
}


/**
 * Show a dialog to edit autoconnect preferences.
 */
void autoconnect_gui_edit_preferences() {

	window = gtk_dialog_new_with_buttons("Autoconnect",
			NULL,
			GTK_DIALOG_MODAL | GTK_DIALOG_NO_SEPARATOR,
			"Cancel",
			GTK_RESPONSE_CANCEL,
			"Edit",
			GTK_RESPONSE_OK,
			NULL);



	/*************** MODEL ***************/
	
	/* model consists of 2x STRING */
	connection_model=GTK_TREE_MODEL(gtk_list_store_new(2, G_TYPE_STRING,
			G_TYPE_STRING));


	/* insert connections */
	autoconnect_gui_update_connectionlist();
	

	/************* VIEW ****************/	
	connection_view=gtk_tree_view_new();
	gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(connection_view), TRUE);
	gtk_tree_view_set_model(GTK_TREE_VIEW(connection_view), connection_model);

	/* fist column, displays name of connection */
	GtkTreeViewColumn *col;
	GtkCellRenderer *renderer;

        col= gtk_tree_view_column_new();
	gtk_tree_view_column_set_title(col, "Connection");

	gtk_tree_view_append_column(GTK_TREE_VIEW(connection_view), col);

	renderer = gtk_cell_renderer_text_new();

	gtk_tree_view_column_pack_start(col, renderer, TRUE);

	gtk_tree_view_column_add_attribute(col, renderer, "text", 0 /* view column no */);


	/* second column, displays state of connection
	 * (bound or unbound to profile)
	 */
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title(col, "Profile");

	gtk_tree_view_append_column(GTK_TREE_VIEW(connection_view), col);

	renderer = gtk_cell_renderer_text_new();

	gtk_tree_view_column_pack_start(col, renderer, TRUE);

	gtk_tree_view_column_add_attribute(col, renderer, "text", 1 /* view column no */);

	

	/* signals */
	g_signal_connect(G_OBJECT(window),
			"response",
			G_CALLBACK(autoconnect_gui_edit_preferences_response),
			NULL);


	/* pack */
	GtkWidget *conn_scroll = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(conn_scroll),
			GTK_POLICY_NEVER, 
			GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(conn_scroll),
			GTK_SHADOW_IN);

	gtk_container_add(GTK_CONTAINER(conn_scroll), connection_view);

	
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(window)->vbox),
			   conn_scroll, TRUE, TRUE, 0);

	gtk_widget_show_all(window);
}
