#include "mainwindow.h"
#include "ui_mainwindow.h"

#include <QVBoxLayout>
#include <QDesktopWidget>
#include "settingswindow.h"
#include "simswindow.h"
#include "smswindow.h"
#include "aboutwindow.h"
#include "commonstatic.h"
#include "widgets/counterwidget.h"

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);    

    SetUpdatedLabel(QDateTime());
    m_NoSettings = false;
    m_Counters = NULL;
    m_Timer = NULL;
    m_Vodafone = new VodafoneWidget("", "");
    m_Closing = false;
    m_WaitIcon = new QMovie(":/images/wait.gif");
    m_WaitIcon->setScaledSize(QSize(40, 40));

    m_MainMenu = NULL;
#ifdef Q_OS_SYMBIAN
    QMenuBar *menu = menuBar();
    ui->lblPuntiD->setText(tr("P. Vodafone One"));
    ui->lblScadenzaD->setText(tr("P. in Scadenza"));
#else
    QMenu *menu = menuBar()->addMenu(tr("Vodafone 190"));
    m_MainMenu = menu;
#endif
    m_SettingsAction = new QAction(tr("Impostazioni"), this);
    connect(m_SettingsAction, SIGNAL(triggered()), this, SLOT(SettingsSlot()));
    m_AboutAction = new QAction(tr("About"), this);
    connect(m_AboutAction, SIGNAL(triggered()), this, SLOT(AboutSlot()));
#ifndef Q_OS_SYMBIAN
    m_QuitAction = new QAction(tr("Esci"), this);
    connect(m_QuitAction, SIGNAL(triggered()), this, SLOT(QuitSlot()));
#endif
    m_UpdateAction = new QAction(tr("Aggiorna"), this);
    connect(m_UpdateAction, SIGNAL(triggered()), this, SLOT(UpdateSlot()));

    m_SelectSimAction = new QAction(tr("Cambia SIM"), this);
    connect(m_SelectSimAction, SIGNAL(triggered()), this, SLOT(SelectSIMSlot()));
    m_SelectSimAction->setEnabled(false);

    menu->addAction(m_UpdateAction);
    menu->addAction(m_SettingsAction);
    menu->addAction(m_SelectSimAction);
    menu->addAction(m_AboutAction);
#ifndef Q_OS_SYMBIAN
    menu->addAction(m_QuitAction);
#endif
    connect(ui->btnSMS, SIGNAL(clicked()), this, SLOT(SmsSlot()));

    m_Settings = new Settings(this);

    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(OrientationChanged()));
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5AutoOrientation, true);
    ui->scrollArea->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    ui->scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
#endif

    if (ui->frmContatori->layout() == NULL){
        QVBoxLayout* layout = new QVBoxLayout(ui->frmContatori);
        layout->setContentsMargins(9, 3, 6, 6);
        layout->setSpacing(3);
        layout->setAlignment(Qt::AlignTop);
        ui->frmContatori->setLayout(layout);
    }

    if (!m_Settings->Load())
        m_NoSettings = true;
    else if (m_Settings->ShowLastData){
        if (m_Settings->LastUpdated.isValid())
            SetUpdatedLabel(m_Settings->LastUpdated);
        ui->lblCreditoResiduo->setText(QString("%1 %2").arg(m_Settings->LastCredito, 4, 'f', 2).arg(m_Settings->LastValuta).trimmed());
        ui->lblPunti->setText(QString("%1").arg(m_Settings->LastSaldoPunti->Punti));
        ui->lblScadenza->setText(QString("%1 %2").arg(m_Settings->LastSaldoPunti->PuntiInScadenza).arg(m_Settings->LastSaldoPunti->Scadenza.toString("dd/MM/yyyy")).trimmed());

        m_Counters = new QList<Counter*>(*m_Settings->LastCounters);
        RefreshCounterWidgets();
    }

    //Apply stylesheet:
    if (m_NoSettings || m_Settings->StyleSheetName == "Default")
        CommonStatic::SetDefaultStylesheet();
    else if (m_Settings->StyleSheetName == "System")
        qApp->setStyleSheet("");
    else{
        QString filename = QString("%1/%2.css").arg(CommonStatic::GetStylesheetFolder())
                                               .arg(m_Settings->StyleSheetName);
        CommonStatic::SetStylesheet(filename);
    }
}

MainWindow::~MainWindow()
{
    delete m_Vodafone;
    delete ui;
}

void MainWindow::Init()
{
    ui->btnSMS->setEnabled(false);
    m_SettingsAction->setEnabled(false);
    m_UpdateAction->setEnabled(false);

    m_Timer = new QTimer();
    m_Timer->setInterval(500);
    m_Timer->setSingleShot(true);
    connect(m_Timer, SIGNAL(timeout()), this, SLOT(TimerSlot()));
    m_Timer->start();
}

void MainWindow::closeEvent(QCloseEvent *)
{
    m_Closing = true;
    m_Vodafone->AbortRequest();
    m_Settings->Save();
    qApp->quit();
}

void MainWindow::TimerSlot()
{
    delete m_Timer;
    m_Timer = NULL;

    if (m_NoSettings)
        SettingsSlot();

    m_Vodafone->SetCredentials(m_Settings->Username, m_Settings->Password);
    m_Vodafone->DebugXml = m_Settings->LogXml;

    if (!m_Settings->Username.isEmpty() && !m_Settings->Password.isEmpty())
        Update();
    else
        m_SettingsAction->setEnabled(true);
}

void MainWindow::SettingsSlot()
{
    QString user = m_Settings->Username;
    QString pwd = m_Settings->Password;
    QString style = m_Settings->StyleSheetName;
    bool std = m_Settings->ShowStandardCounters;

    SettingsWindow* frm = new SettingsWindow(this, m_Settings);
#if defined(Q_OS_SYMBIAN)
    frm->showMaximized();
#endif
    if (frm->exec() == QDialog::Accepted){
        m_Vodafone->DebugXml = m_Settings->LogXml;
        if (!m_NoSettings &&
            (m_Settings->Username != user || m_Settings->Password != pwd || m_Settings->ShowStandardCounters != std)){
            m_Vodafone->SetCredentials(m_Settings->Username, m_Settings->Password);
            Update();
        }        

        //Recreate counterWidgets on style change (with default stylesheet progressbare
        //are not correclty update)
        if (m_Settings->StyleSheetName != style)
            RefreshCounterWidgets();

        for (int i=0; i<ui->frmContatori->layout()->count(); i++){
            QLayoutItem *child = ui->frmContatori->layout()->itemAt(i);
            CounterWidget* w = (CounterWidget*)child->widget();
            w->SetPercentageVisible(m_Settings->ShowPercentage);
        }
    }
    delete frm;
    m_NoSettings = false;
}

void MainWindow::AboutSlot()
{
    AboutWindow* frm = new AboutWindow(this);
#if defined(Q_OS_SYMBIAN)
    frm->showMaximized();
#else
    frm->show();
#endif
    frm->Init();
    frm->exec();

    delete frm;
}

void MainWindow::QuitSlot()
{
    this->close();
}

void MainWindow::UpdateSlot()
{
    Update();
}

void MainWindow::SelectSIMSlot()
{
    SimsWindow* frm = new SimsWindow(this, new QStringList(m_Vodafone->GetSims()));
#if defined(Q_OS_SYMBIAN)
    frm->showMaximized();
#endif
    if (frm->exec() == QDialog::Accepted){
        QString sim = frm->GetSelectedSim();
        if (!sim.isEmpty() && sim != m_Vodafone->GetCurrentSim()){
            if (m_Vodafone->SetCurrentSim(sim))
                Update();
        }
    }
    delete frm;
}

void MainWindow::SmsSlot()
{
    SmsWindow* frm = new SmsWindow(this, m_Vodafone, m_Settings);
#if defined(Q_OS_SYMBIAN)
    frm->showMaximized();
#endif
    frm->exec();
    delete frm;
}

bool counterToAscending(const Counter* s1 , const Counter* s2)
{
    return s1->Type < s2->Type;
}

void MainWindow::Update()
{
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
#else
    ui->lblLogo->setMovie(m_WaitIcon);
    m_WaitIcon->start();
#endif

    ui->btnSMS->setEnabled(false);
    m_SettingsAction->setEnabled(false);
    m_UpdateAction->setEnabled(false);
    m_SelectSimAction->setEnabled(false);

    QString valuta;
    if (m_Vodafone->IsLoggedIn() || m_Vodafone->Login()){
        if (m_Closing)
            return;

        ui->btnSMS->setEnabled(true);

        if (m_Vodafone->GetSims().count() > 1){
#ifdef Q_WS_MAEMO_5
            m_MainMenu->setTitle(m_Vodafone->GetCurrentSim());
#else
            this->setWindowTitle(m_Vodafone->GetCurrentSim());
#endif
        }
        float credito = 0;
        if (m_Vodafone->GetCurrentSimType() == VodafoneWidget::SimRicaricabile){
            ui->lblCreditoResiduoD->setText(tr("Credito Residuo"));
            credito = m_Vodafone->GetCredito(&valuta);
            if (valuta == "E")
                valuta = QChar(8364);
            if (m_Closing)
                return;
        }else{
            ui->lblCreditoResiduoD->setText(tr("Traffico"));
            InfoConto* conto = new InfoConto(this);
            m_Vodafone->GetInfoConto(conto);
            credito = conto->TrafficoNonFatturato;
            valuta = QChar(8364);
        }

        SaldoPunti* punti = new SaldoPunti();
        m_Vodafone->GetSaldoPunti(punti);
        if (m_Closing)
            return;

        ui->lblCreditoResiduo->setText(QString("%1 %2").arg(credito, 4, 'f', 2).arg(valuta).trimmed());
        ui->lblPunti->setText(QString("%1").arg(punti->Punti));
        ui->lblScadenza->setText(QString("%1 %2").arg(punti->PuntiInScadenza).arg(punti->Scadenza.toString("dd/MM/yyyy")).trimmed());

        m_Settings->LastCredito = credito;
        m_Settings->LastValuta = valuta;
        m_Settings->LastSaldoPunti = punti;

        m_Counters = m_Vodafone->GetListaContatori(m_Settings->ShowStandardCounters);
        if (m_Closing)
            return;

        QLayoutItem *child;
        while ((child = ui->frmContatori->layout()->takeAt(0)) != 0) {
            delete child->widget();
            delete child;
        }

        qSort(m_Counters->begin() , m_Counters->end(), counterToAscending );
        foreach(Counter* c, *m_Counters){
            bool res = m_Vodafone->GetDettaglioContatore(c);
            if (m_Closing)
                return;
            if (res){
                CounterWidget* w = new CounterWidget(this, c, m_Settings);
                ui->frmContatori->layout()->addWidget(w);
            }
        }
        m_Settings->LastCounters = m_Counters;
        m_Settings->LastUpdated = QDateTime::currentDateTime();
        SetUpdatedLabel(m_Settings->LastUpdated);
    }else{
        if (!m_Closing)
            CommonStatic::ErrorMessage(this, tr("Login fallito"));
    }

    m_SelectSimAction->setEnabled(m_Vodafone->GetSims().count() > 1);
    m_SettingsAction->setEnabled(true);
    m_UpdateAction->setEnabled(true);

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
#else
    ui->lblLogo->setMovie(NULL);
    ui->lblLogo->setPixmap(QPixmap(":/images/vodafone_logo.png"));
#endif
}

void MainWindow::RefreshCounterWidgets()
{
    QLayoutItem *child;
    while ((child = ui->frmContatori->layout()->takeAt(0)) != 0) {
        delete child->widget();
        delete child;
    }

    if (m_Counters){
        foreach(Counter* c, *m_Counters){
            CounterWidget* w = new CounterWidget(this, c, m_Settings);
            ui->frmContatori->layout()->addWidget(w);
        }
    }
}

void MainWindow::OrientationChanged()
{
    if (ui->frmContatori->layout() != NULL){
        RefreshCounterWidgets();
        SetUpdatedLabel(m_Settings->LastUpdated);
    }
}

void MainWindow::SetUpdatedLabel(QDateTime date)
{
#if defined(Q_OS_SYMBIAN)
    ui->lblUpdated->setVisible(false);
    ui->lblUpdatedP->setVisible(true);
#else
    if (CommonStatic::IsPortrait()){
        ui->lblUpdated->setVisible(false);
        ui->lblUpdatedP->setVisible(true);
    }else{
        ui->lblUpdatedP->setVisible(false);
        ui->lblUpdated->setVisible(true);
    }
#endif
    if (!date.isNull()){
        ui->lblUpdatedP->setText(QString(tr("Aggiornato: %1").arg(date.toString("dd/MM/yyyy hh:mm"))));
        ui->lblUpdated->setText(QString(tr("Aggiornato: %1").arg(date.toString("dd/MM/yyyy hh:mm"))));
    }
}

