#include "contactswindow.h"
#include "ui_contactswindow.h"

#ifdef Q_WS_MAEMO_5
    #include <QContactManager>
    #include <QContactName>
    #include <QContactPhoneNumber>
    #include <QContactThumbnail>
#endif
#include <QPushButton>
#include <QSettings>
#include <QFile>
#include "widgets/contactwidget.h"
#include "commonstatic.h"

ContactsWindow::ContactsWindow(QWidget *parent) :
    QDialog(parent),
    ui(new Ui::ContactsWindow)
{
    ui->setupUi(this);

    m_Timer = NULL;

#ifdef Q_OS_SYMBIAN
    QAction* softKey = new QAction( tr("Seleziona"), this );
    softKey->setSoftKeyRole( QAction::PositiveSoftKey );
    connect(softKey, SIGNAL(triggered()), this, SLOT(accept()));
    addAction(softKey);

    softKey = new QAction( tr("Indietro"), this );
    softKey->setSoftKeyRole( QAction::NegativeSoftKey );
    connect(softKey, SIGNAL(triggered()), this, SLOT(reject()));
    addAction(softKey);
#endif

#ifdef Q_WS_MAEMO_5
    ui->scrollArea->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    ui->scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
#endif

    if (ui->scrollAreaWidgetContents->layout() == NULL){
        QVBoxLayout* layout = new QVBoxLayout(ui->scrollAreaWidgetContents);
        ui->scrollAreaWidgetContents->setLayout(layout);
        layout->setAlignment(Qt::AlignTop);
        layout->setSpacing(0);
        layout->setMargin(3);
    }

    ui->txtCerca->grabKeyboard();
    connect(ui->txtCerca, SIGNAL(textChanged(QString)), this, SLOT(CercaSlot()));
    connect(ui->btnClear, SIGNAL(clicked()), this, SLOT(ClearSlot()));
}

ContactsWindow::~ContactsWindow()
{
    delete ui;
}

bool toAscending(const Contact* s1 , const Contact* s2)
{
    return s1->Name < s2->Name;
}

void ContactsWindow::FillContactsList()
{
    m_Contacts.clear();

//#if defined(Q_WS_MAEMO_5) || defined(Q_OS_SYMBIAN)
#ifdef Q_WS_MAEMO_5
    // Symbian and Maemo implementation (read phonebook from phone)
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
#endif

    QContactManager* man = NULL;
    QStringList availableManagers = QContactManager::availableManagers();

    QList<QContact> contacts;
    while (!availableManagers.isEmpty()) {
        man = new QContactManager(availableManagers.first());
        availableManagers.removeFirst();

        contacts = man->contacts();
        if (!contacts.isEmpty()) {
            availableManagers.clear();
            break;
        } else {
            delete man;
            man = NULL;
        }
    }

    if (!man)
        man = new QContactManager();

    if (!contacts.isEmpty()){
        foreach (QContact c, contacts){
            if (!c.isEmpty() && c.type() == QContactType::TypeContact){
                QList<QContactDetail> phns = c.details(QContactPhoneNumber::DefinitionName);
                foreach (QContactDetail detail, phns){
                    QContactPhoneNumber* phn = (QContactPhoneNumber*)(&detail);
                    if (phn->subTypes().contains(QContactPhoneNumber::SubTypeMobile)) {
                        Contact* contact = new Contact(this);
                        contact->Name = man->synthesizedContactDisplayLabel(c);
                        contact->MobileNumber = phn->number();

                        QContactThumbnail thn = c.detail(QContactThumbnail::DefinitionName);
                        if (!thn.isEmpty())
                            contact->Thumbnail = new QImage(thn.thumbnail());
                        m_Contacts.append(contact);
                    }
                }
            }
        }
    }

    delete man;
    man = NULL;
#else
    // Windows implementation (read a phonebook from file)
    QString book = QString("%1/contacts.conf").arg(QCoreApplication::applicationDirPath());
    if (!QFile::exists(book))
        return;

    QSettings* contacts = new QSettings(book, QSettings::IniFormat);

    contacts->beginGroup("Contacts");

    int max = contacts->beginReadArray("items");
    for (int i = 0; i < max; ++i) {
        contacts->setArrayIndex(i);

        Contact* contact = new Contact(this);
        contact->Name = contacts->value("name").toString();
        contact->MobileNumber = contacts->value("number").toString();

        m_Contacts.append(contact);
    }

    contacts->endGroup();
#endif

    qSort(m_Contacts.begin() , m_Contacts.end(), toAscending );
    QPixmap* cImage = new QPixmap(":/images/contacts.png");
    int i = 0;
    foreach (Contact* c, m_Contacts)
        AddContactToList(c, i++, cImage);

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
#endif
}

void ContactsWindow::AddContactToList(Contact* c, int index, QPixmap* defaultImage)
{    
    ContactWidget* w = new ContactWidget(this, c, defaultImage);
    w->setObjectName(QString("%1").arg(index));
    connect(w, SIGNAL(clicked()), this, SLOT(SelectContactSlot()));
    ui->scrollAreaWidgetContents->layout()->addWidget(w);
}

QString ContactsWindow::GetSelectedNumber()
{
    QString res = m_Selected.trimmed();
    if (res.startsWith("+39"))
        res.remove(0, 3);
    return res;
}

void ContactsWindow::SelectContactSlot()
{
    ContactWidget* w = (ContactWidget*)sender();
    int index = w->objectName().toInt();

    Contact* c = m_Contacts.at(index);
    m_Selected = c->MobileNumber;

    accept();
}

void ContactsWindow::Init()
{
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
#endif
    m_Timer = new QTimer();
    m_Timer->setInterval(300);
    m_Timer->setSingleShot(true);
    connect(m_Timer, SIGNAL(timeout()), this, SLOT(TimerSlot()));
    m_Timer->start();

    ui->txtCerca->setFocus();
}

void ContactsWindow::TimerSlot()
{
    delete m_Timer;
    m_Timer = NULL;

    FillContactsList();
}

void ContactsWindow::CercaSlot()
{
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
#endif
    ui->txtCerca->setFocus();

    QWidget* child;
    for (int i=0; i<ui->scrollAreaWidgetContents->layout()->count(); i++){
        child = ui->scrollAreaWidgetContents->layout()->itemAt(i)->widget();
        if (child){
            int index = child->objectName().toInt();
            Contact* c = m_Contacts.at(index);
            if (ui->txtCerca->text().isEmpty() || c->Name.contains(ui->txtCerca->text(), Qt::CaseInsensitive))
                child->setVisible(true);
            else
                child->setVisible(false);
        }
    }
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
#endif
}

void ContactsWindow::ClearSlot()
{
    ui->txtCerca->clear();
}
