#include "vodafonewidget.h"

VodafoneWidget::VodafoneWidget(QString username, QString password)
{    
    m_Aborting = false;
    m_CurrentReply = NULL;
    DebugXml = false;
    m_Username = username;
    m_Password = password;

    m_CookieJar = new QNetworkCookieJar (this);
    m_Nam = new QNetworkAccessManager(this);
    m_Nam->setCookieJar(m_CookieJar);
} // VodafoneWidget

VodafoneWidget::~VodafoneWidget()
{
    delete m_CookieJar;
    m_CookieJar = NULL;
    delete m_Nam;
    m_Nam = NULL;
} // ~VodafoneWidget

void VodafoneWidget::SetCredentials(QString username, QString password, bool logout)
{
    if (logout)
        Logout();
    m_Username = username;
    m_Password = password;
} // SetCredentials

bool VodafoneWidget::IsLoggedIn()
{
    bool result = false;
    QNetworkRequest request = PrepareRequest("https://widget.vodafone.it/190/trilogy/jsp/utility/checkUser.jsp");
    QString string = SynchRequest(&request);
    if (DebugXml)
        qDebug() << "IsLoggedIn: " << string;

    QDomDocument doc("response");
    doc.setContent(string);

    QDomNode root = doc.firstChildElement("root");
    QDomNode logged = root.firstChildElement("logged-in");
    if (!logged.isNull() && logged.firstChild().nodeValue() == "true")
        result = true;

    QDomNode sessionID = root.firstChildElement("session-id");
    if (!sessionID.isNull())
        m_SessionID = sessionID.firstChild().nodeValue();

    return result;
} // IsLoggedIn

bool VodafoneWidget::Login()
{
    QUrl url("https://widget.vodafone.it/190/trilogy/jsp/login.do");
    url.addQueryItem("username", m_Username);
    url.addQueryItem("password", m_Password);
    url.addQueryItem("cu_notrace", "true");
    QNetworkRequest request = PrepareRequest(url);

    QUrl newUrl;
    QString string = SynchRequest(&request, &newUrl);
    //The login process redirects through more than one page:
    while (newUrl.isValid()){        
        if (newUrl.path().endsWith("home.do")) // This redirect in not needed to login
            break;
        request = PrepareRequest(newUrl);        
        string = SynchRequest(&request, &newUrl);
    }

    return IsLoggedIn();
} // Login

void VodafoneWidget::Logout()
{
    QNetworkRequest request = PrepareRequest("https://widget.vodafone.it/190/trilogy/jsp/logout.do");
    SynchRequest(&request);
} // Logout

QNetworkRequest VodafoneWidget::PrepareRequest(QUrl url)
{
    return PrepareRequest(url.toString());
} // PrepareRequest

QNetworkRequest VodafoneWidget::PrepareRequest(QString url)
{
    QUrl tUrl(url);
    QNetworkRequest request(tUrl);
    request.setRawHeader("User-Agent", "Vodafone_DW");
    return request;
} // PrepareRequest

QString VodafoneWidget::SynchRequest(QNetworkRequest* request, QUrl* redirectUrl)
{
    if (DebugXml)
        qDebug() << "SynchRequest: " << request->url().toString();

    QTime start;
    start.start();

    if (redirectUrl)
        redirectUrl->clear();
    QNetworkReply* reply = m_Nam->get(*request);
    m_CurrentReply = reply;
    reply->ignoreSslErrors();

    QEventLoop* loop = new QEventLoop();
    QObject::connect(m_Nam, SIGNAL(finished(QNetworkReply *)), loop, SLOT(quit()));
    loop->exec();
    QObject::disconnect(m_Nam, SIGNAL(finished(QNetworkReply *)), loop, SLOT(quit()));
    delete loop;
    if (m_Aborting){
        m_Aborting = false;
        m_CurrentReply = NULL;
        delete reply;
        return "";
    }

    if(reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt() == 302 && redirectUrl)
        *redirectUrl = reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl();
    QByteArray bytes = reply->readAll();
    QString result(bytes);

    m_CurrentReply = NULL;
    delete reply;

    if (DebugXml){
        QTime elapsed(0, 0, 0, 0);
        elapsed = elapsed.addMSecs(start.elapsed());
        qDebug() << "SynchRequest: Time " << elapsed.toString("hh:mm:ss.z");
    }
    return result;
} // SynchRequest

float VodafoneWidget::GetCredito(QString* valuta)
{
    float result = 0;
    QNetworkRequest request = PrepareRequest("https://widget.vodafone.it/190/fast/mx/CreditoResiduoPush.do?hpfdtpri=y");
    QString string = SynchRequest(&request);
    if (DebugXml)
        qDebug() << "GetCredito: " << string;

    QDomDocument doc("response");
    doc.setContent(string);

    QDomNode root = doc.firstChildElement("root");
    QDomNodeList list = root.childNodes();
    int found = 0;
    for (int i=0; i<list.count(); i++){
        QDomNode node = list.at(i);
        if (node.nodeName() == "e"){
            QDomNode name = node.attributes().namedItem("n");
            if (name.nodeValue() == "CREDITO_RESIDUO"){
                QDomNode val = node.attributes().namedItem("v");
                QString credito = val.nodeValue();
                result = Decrypt(&credito, &m_SessionID).toFloat();
                found++;
            }else if (name.nodeValue() == "VALUTA"){
                QDomNode val = node.attributes().namedItem("v");
                *valuta = val.nodeValue();
                found++;
            }
        }
        if (found >= 2)
            break;
    }

    return result;
} // GetCredito

QList<Counter*>* VodafoneWidget::GetListaContatori()
{
    QList<Counter*>* result = new QList<Counter*>();

    QNetworkRequest request = PrepareRequest("https://widget.vodafone.it/190/ebwe/mx/Desktopwidgetconto.do");
    QString string = SynchRequest(&request);
    if (DebugXml)
        qDebug() << "GetListaContatori: " << string;

    QDomDocument doc("response");
    doc.setContent(string);
    if (GetReturnCode(doc) != "0")
        return result;

    QDomNode root = doc.firstChildElement("root");
    QDomNodeList list = root.childNodes();
    for (int i=0; i<list.count(); i++){
        QDomNode node = list.at(i);
        if (node.nodeName() == "ProductItem"){
            Counter* c = new Counter();
            QDomNodeList clist = node.childNodes();
            int found = 0;
            for (int j=0; j<clist.count(); j++){
                QDomNode cnode = clist.at(j);
                if (cnode.nodeName() == "ProductType"){
                    c->Type = cnode.firstChild().nodeValue();
                    found++;
                }else if (cnode.nodeName() == "Classe"){
                    c->Class = cnode.firstChild().nodeValue();
                    found++;
                }else if (cnode.nodeName() == "ProductCode"){
                    c->Code = cnode.firstChild().nodeValue();
                    found++;
                }else if (cnode.nodeName() == "Descrizione"){
                    c->Description = cnode.firstChild().nodeValue();
                    found++;
                }
                if (found == 4)
                    break;
            }
            result->append(c);
        }
    }
    return result;
} // GetListaContatori

bool VodafoneWidget::GetDettaglioContatore(Counter* counter)
{
    QUrl url("https://widget.vodafone.it/190/ebwe/mx/DesktopwidgetContatoreDettagli.do");
    url.addQueryItem("ProductType", counter->Type);
    url.addQueryItem("Classe", counter->Class);
    url.addQueryItem("ProductCode", counter->Code);
    QNetworkRequest request = PrepareRequest(url);

    QString string = SynchRequest(&request);
    if (DebugXml)
        qDebug() << "GetDettaglioContatore: " << string;

    QDomDocument doc("response");
    doc.setContent(string);
    if (GetReturnCode(doc) != "0")
        return false;

    QDomNode root = doc.firstChildElement("root");
    QDomNodeList list = root.firstChildElement("SogliaItem").childNodes();
    int found = 0;
    for (int i=0; i<list.count(); i++){
        QDomNode node = list.at(i);
        if (node.nodeName() == "Soglia"){
            counter->Soglia = node.firstChild().nodeValue().toInt();
            found++;
        }else if (node.nodeName() == "DisplayType"){
            counter->SogliaDisplayType = node.firstChild().nodeValue().toInt();
            found++;
        }else if (node.nodeName() == "TrafficoUtilizzato"){
            counter->Utilizzato = node.firstChild().nodeValue().toInt();
            found++;
        }else if (node.nodeName() == "TrafficoResiduo"){
            counter->Residuo = node.firstChild().nodeValue().toInt();
            found++;
        }else if (node.nodeName() == "DimensioneScatto"){
            counter->DimensioneScatto = node.firstChild().nodeValue();
            found++;
        }else if (node.nodeName() == "UnitaScatto"){
            counter->Scatto = node.firstChild().nodeValue().toInt();
            found++;
        }else if (node.nodeName() == "Periodo"){
            counter->Periodo = QDate::fromString(node.firstChild().nodeValue(), "dd/MM/yyyy");
            found++;
        }else if (node.nodeName() == "DataUtile"){
            counter->DataUtile = QDate::fromString(node.firstChild().nodeValue(), "dd/MM/yyyy");
            found++;
        }

        if (found == 8)
            break;
    }

    return true;
} // GetDettaglioContatore

QString VodafoneWidget::GetReturnCode(QDomDocument doc)
{
    QString res;

    QDomNode root = doc.firstChildElement("root");
    QDomNode returnCode = root.firstChildElement("ReturnCode");
    if (!returnCode.isNull())
        res = returnCode.firstChild().nodeValue();
    return res;
} // GetReturnCode

QString VodafoneWidget::GetStatus(QDomDocument doc, QString* errorMessage)
{
    QString res;

    QDomNode root = doc.firstChildElement("root");
    QDomNodeList list = root.childNodes();
    int found = 0;
    for (int i=0; i<list.count(); i++){
        QDomNode node = list.at(i);
        if (node.nodeName() == "e"){
            QDomNode name = node.attributes().namedItem("n");
            if (name.nodeValue() == "STATUS"){
                res = node.attributes().namedItem("v").nodeValue();
                found++;
            }else if (name.nodeValue() == "RETURNMSG"){
                if (errorMessage && node.firstChild().isCDATASection())
                    *errorMessage = node.firstChild().toCDATASection().data();
                found++;
            }
        }
        if (found == 2)
            break;
    }
    return res;
} // GetStatus

bool VodafoneWidget::PrepareMessage(QString receiver, QString message, QByteArray* captchaImage, QString* errorMessage)
{
    QUrl url("https://widget.vodafone.it/190/fsms/precheck.do?channel=VODAFONE_DW");
    QNetworkRequest request = PrepareRequest(url);
    SynchRequest(&request);

    url = QUrl("https://widget.vodafone.it/190/fsms/prepare.do?channel=VODAFONE_DW");
    url.addQueryItem("receiverNumber", receiver);
    url.addQueryItem("message", message);
    request = PrepareRequest(url);

    QString string = SynchRequest(&request);
    if (DebugXml)
        qDebug() << "PrepareMessage: " << string;

    QDomDocument doc("response");
    doc.setContent(string);

    if (GetStatus(doc, errorMessage) != "1")
        return false;

    QDomNode root = doc.firstChildElement("root");
    QDomNodeList list = root.childNodes();
    for (int i=0; i<list.count(); i++){
        QDomNode node = list.at(i);
        if (node.nodeName() == "e"){
            QDomNode name = node.attributes().namedItem("n");
            if (name.nodeValue() == "CODEIMG"){
                if (node.firstChild().isCDATASection()){
                    QString data = node.firstChild().toCDATASection().data();
                    *captchaImage = QByteArray::fromBase64(data.toLatin1());
                    break;
                }
            }
        }
    }
    return true;
} // PrepareMessage

bool VodafoneWidget::SendMessage(QString receiver, QString message, QString captcha, QString* errorMessage)
{
    QUrl url("https://widget.vodafone.it/190/fsms/send.do?channel=VODAFONE_DW");
    url.addQueryItem("verifyCode", captcha);
    url.addQueryItem("receiverNumber", receiver);
    url.addQueryItem("message", message);
    QNetworkRequest request = PrepareRequest(url);

    QString string = SynchRequest(&request);
    if (DebugXml)
        qDebug() << "SendMessage: " << string;

    QDomDocument doc("response");
    doc.setContent(string);

    if (GetStatus(doc, errorMessage) != "1")
        return false;

    return true;
} // SendMessage

bool VodafoneWidget::GetSaldoPunti(SaldoPunti* saldo)
{
    QUrl url("https://widget.vodafone.it/190/jone/mx/SaldoPuntiPush.do?hpfdtpri=y");
    QNetworkRequest request = PrepareRequest(url);

    QString string = SynchRequest(&request);
    if (DebugXml)
        qDebug() << "GetSaldoPunti: " << string;

    QDomDocument doc("response");
    doc.setContent(string);

    QDomNode root = doc.firstChildElement("root");
    QDomNodeList list = root.childNodes();
    int found = 0;
    for (int i=0; i<list.count(); i++){
        QDomNode node = list.at(i);
        if (node.nodeName() == "e"){
            QDomNode name = node.attributes().namedItem("n");
            if (name.nodeValue() == "SALDO_PUNTI"){
                QDomNode val = node.attributes().namedItem("v");
                QString punti = val.nodeValue();
                saldo->Punti = Decrypt(&punti, &m_SessionID).toFloat();
                found++;
            }else if (name.nodeValue() == "PUNTI_IN_SCADENZA"){
                QDomNode val = node.attributes().namedItem("v");
                QString punti = val.nodeValue();
                saldo->PuntiInScadenza = Decrypt(&punti, &m_SessionID).toFloat();
                found++;
            }else if (name.nodeValue() == "ANNO_ACCUMULO_PUNTI_IN_SCADENZA"){
                QDomNode val = node.attributes().namedItem("v");
                saldo->AnnoPuntiInScadenza = val.nodeValue().toInt();
                found++;
            }else if (name.nodeValue() == "DATA_AGGIORNAMENTO_SALDO"){
                QDomNode val = node.attributes().namedItem("v");
                QString data = val.nodeValue();
                data = data.left(data.indexOf("+"));
                saldo->Aggiornamento = QDate::fromString(data, "yyyy-MM-dd");
                found++;
            }else if (name.nodeValue() == "DATA_SCADENZA_PUNTI"){
                QDomNode val = node.attributes().namedItem("v");
                QString data = val.nodeValue();
                data = data.left(data.indexOf("+"));
                saldo->Scadenza = QDate::fromString(data, "yyyy-MM-dd");
                found++;
            }
        }
        if (found == 5)
            break;
    }

    return true;
} // GetSaldoPunti

QString VodafoneWidget::Decrypt(QString* crypted, QString* sessionID)
{
    QString cryptTable = " ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789\t!@#$%^&*()`\'-=[];,./?_+{}|:<>~";
    QString doubleQuote ="\"";
    QString lineFeed ="\n";
    int cryptLength = cryptTable.length() - 1;

    QString loc1;
    int loc2 = 0;
    QString loc3;
    int loc10 = 0;
    bool loc4 = false;
    QString result = "";
    QList<int> loc6;
    int loc7 = sessionID->length();
    int loc8 = crypted->length();

    loc10 = 0;
    while (loc10 < loc7) {
        loc6.append(cryptTable.indexOf(sessionID->at(loc10)));
        ++loc10;
    }

    int loc11 = 0;
    loc10 = 0;
    while (loc11 < loc8) {
        if (loc10 >= loc7)
            loc10 = 0;
        loc1 = crypted->at(loc11);
        if ((loc2 = cryptTable.indexOf(loc1)) != -1) {
            if (loc4) {
                if (loc2 != cryptLength) {
                    if (loc1 != "\'")
                        loc2 = loc2 + cryptLength;
                    else{
                        loc3 = doubleQuote;
                        loc2 = -1;
                    }
                } else {
                    loc3 = lineFeed;
                    loc2 = -1;
                }
                loc4 = false;
            } else {
                if (loc2 == cryptLength) {
                    loc4 = true;
                    loc10 = (loc10 - 1);
                    loc3 = "";
                    loc2 = -1;
                }
            }
        } else
            loc3 = loc1;

        if (loc2 != -1)
            loc3 = cryptTable.at(loc6[loc10] ^ loc2);
        result.append(loc3);
        ++loc11;
        ++loc10;
    }
    return result;
} // Decrypt

void VodafoneWidget::AbortRequest()
{
    if (m_CurrentReply){
        m_Aborting = true;
        m_CurrentReply->abort();
    }
} // AbortRequest
