#include "contactswindow.h"
#include "ui_contactswindow.h"
#ifdef Q_WS_MAEMO_5
    #include <QContactManager>
    #include <QContactName>
    #include <QContactPhoneNumber>
    #include <QContactThumbnail>    
#endif
#include <QPushButton>
#include <QSettings>
#include <QFile>
#include "widgets/contactwidget.h"
#include "commonstatic.h"

ContactsWindow::ContactsWindow(QWidget *parent) :
    QDialog(parent),
    ui(new Ui::ContactsWindow)
{
    ui->setupUi(this);

    m_Timer = NULL;

#ifdef Q_WS_MAEMO_5
    ui->scrollArea->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    ui->scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
#endif

    if (ui->scrollAreaWidgetContents->layout() == NULL){
        QVBoxLayout* layout = new QVBoxLayout(ui->scrollAreaWidgetContents);
        ui->scrollAreaWidgetContents->setLayout(layout);
        layout->setAlignment(Qt::AlignTop);
        layout->setSpacing(0);
        layout->setMargin(3);
    }

    ui->txtCerca->grabKeyboard();
    connect(ui->txtCerca, SIGNAL(textChanged(QString)), this, SLOT(CercaSlot()));
    connect(ui->btnClear, SIGNAL(clicked()), this, SLOT(ClearSlot()));
}

ContactsWindow::~ContactsWindow()
{
    delete ui;
}

bool toAscending(const Contact* s1 , const Contact* s2)
{
    return s1->Name < s2->Name;
}

void ContactsWindow::FillContactsList()
{
    m_Contacts.clear();

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);

    QContactManager* man = NULL;
    QStringList availableManagers = QContactManager::availableManagers();

    QList<QContactLocalId> contactIds;
    while (!availableManagers.isEmpty()) {
        man = new QContactManager(availableManagers.first());
        availableManagers.removeFirst();

        contactIds = man->contactIds();
        if (!contactIds.isEmpty()) {
            availableManagers.clear();
            break;
        } else {
            delete man;
            man = NULL;
        }
    }

    if (!man)
        man = new QContactManager();

    if (!contactIds.isEmpty()){
        QList<QContact> contacts;
        foreach (QContactLocalId id, contactIds){
            QContact c = man->contact(id);
            if (!c.isEmpty())
                contacts.append(c);
        }

        foreach (QContact c, contacts){
            if (!c.isEmpty() && c.type() == QContactType::TypeContact){
                QList<QContactDetail> phns = c.details(QContactPhoneNumber::DefinitionName);
                foreach (QContactDetail detail, phns){
                    QContactPhoneNumber phn = (QContactPhoneNumber)detail;
                    if (phn.subTypes().contains(QContactPhoneNumber::SubTypeMobile)) {
                        Contact* contact = new Contact(this);
                        contact->Name = man->synthesizedContactDisplayLabel(c);
                        contact->MobileNumber = phn.number();

                        QContactThumbnail thn = c.detail(QContactThumbnail::DefinitionName);
                        if (!thn.isEmpty())
                            contact->Thumbnail = new QImage(thn.thumbnail());
                        m_Contacts.append(contact);
                    }
                }
            }
        }
    }

    delete man;
    man = NULL;    
#else
    QString book = QString("%1/contacts.conf").arg(QCoreApplication::applicationDirPath());
    if (!QFile::exists(book))
        return;

    QSettings* contacts = new QSettings(book, QSettings::IniFormat);

    contacts->beginGroup("Contacts");

    int max = contacts->beginReadArray("items");
    for (int i = 0; i < max; ++i) {
        contacts->setArrayIndex(i);

        Contact* contact = new Contact(this);
        contact->Name = contacts->value("name").toString();
        contact->MobileNumber = contacts->value("number").toString();

        m_Contacts.append(contact);
    }

    contacts->endGroup();
#endif

    qSort(m_Contacts.begin() , m_Contacts.end(), toAscending );
    int i = 0;
    foreach (Contact* c, m_Contacts)
        AddContactToList(c, i++);

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
#endif
}

void ContactsWindow::AddContactToList(Contact* c, int index)
{    
    ContactWidget* w = new ContactWidget(this, c);
    w->setObjectName(QString("%1").arg(index));
    connect(w, SIGNAL(clicked()), this, SLOT(SelectContactSlot()));
    ui->scrollAreaWidgetContents->layout()->addWidget(w);
}

QString ContactsWindow::GetSelectedNumber()
{
    return m_Selected;
}

void ContactsWindow::SelectContactSlot()
{
    ContactWidget* w = (ContactWidget*)sender();
    int index = w->objectName().toInt();

    Contact* c = m_Contacts.at(index);
    m_Selected = c->MobileNumber;

    accept();
}

void ContactsWindow::Init()
{
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
#endif
    m_Timer = new QTimer();
    m_Timer->setInterval(300);
    m_Timer->setSingleShot(true);
    connect(m_Timer, SIGNAL(timeout()), this, SLOT(TimerSlot()));
    m_Timer->start();

    ui->txtCerca->setFocus();
}

void ContactsWindow::TimerSlot()
{
    delete m_Timer;
    m_Timer = NULL;

    FillContactsList();
}

void ContactsWindow::CercaSlot()
{
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
#endif
    ui->txtCerca->setFocus();

    QWidget* child;
    for (int i=0; i<ui->scrollAreaWidgetContents->layout()->count(); i++){
        child = ui->scrollAreaWidgetContents->layout()->itemAt(i)->widget();
        if (child){
            int index = child->objectName().toInt();
            Contact* c = m_Contacts.at(index);
            if (ui->txtCerca->text().isEmpty() || c->Name.contains(ui->txtCerca->text(), Qt::CaseInsensitive))
                child->setVisible(true);
            else
                child->setVisible(false);
        }
    }
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
#endif
}

void ContactsWindow::ClearSlot()
{
    ui->txtCerca->clear();
}
