/**
 *  Viito -- an image dictionary for sign language
 *
 *  Copyright (C) 2010  Aki Niemi <aki.niemi@iki.fi>
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <QtGui>
#include <QDomDocument>
#include <QFile>
#include <QListView>
#include <QStandardItemModel>
#include <QPixmapCache>
#include <QPixmap>
#include <QLabel>
#include <QDebug>

#include "vmainwindow.h"
#include "vimagewidget.h"

static QDomDocument readXmlDocument(const QString &fileName)
{
    QFile file(fileName);
    QDomDocument doc;

    QString error;
    int errorLine;
    int errorColumn;

    if (!file.open(QIODevice::ReadOnly)) {
	qDebug() << "Read failed:" << file.error();
	return doc;
    }

    if (!doc.setContent(&file, &error, &errorLine, &errorColumn)) {
	qDebug() << "Error parsing XML document:" << error
		 << "at line:" << errorLine
		 << "and column:" << errorColumn;
    }

    file.close();
    return doc;
}

static QPixmap pixmapFromFile(const QString &name)
{
    QPixmap pixmap;

    if (!QPixmapCache::find(name, pixmap)) {
	pixmap = QPixmap(QString("/opt/viito/data/%1").arg(name))
	    .scaled(800, 480, Qt::KeepAspectRatio, Qt::SmoothTransformation);
	QPixmapCache::insert(name, pixmap);
    }

    return pixmap;
}

VMainWindow::VMainWindow()
    : QMainWindow()
{
}

VMainWindow::VMainWindow(const QString &file)
    : QMainWindow()
{
    QDomDocument doc = readXmlDocument(file);
    setModel(doc);

    mTopicView = new QListView(this);
    mTopicView->setModel(mModel);

    mImageView = new QListView(this);
    mImageView->setAttribute(Qt::WA_Maemo5StackedWindow);
    mImageView->setWindowFlags(mImageView->windowFlags() | Qt::Window);

    mImageWidget = new VImageWidget(mImageView);

    connect(mTopicView, SIGNAL(clicked(const QModelIndex&)),
	    this, SLOT(topicClicked(const QModelIndex&)));

    connect(mImageView, SIGNAL(clicked(const QModelIndex&)),
	    this, SLOT(imageClicked(const QModelIndex&)));

    connect(mImageWidget, SIGNAL(next()),
	    this, SLOT(nextImage()));

    connect(mImageWidget, SIGNAL(previous()),
	    this, SLOT(previousImage()));

    setAttribute(Qt::WA_Maemo5StackedWindow);
    setCentralWidget(mTopicView);
}

void VMainWindow::setModel(const QDomDocument &doc)
{
    mModel = new QStandardItemModel;
    mSearchModel = new QStandardItemModel;

    QDomNodeList topics = doc.elementsByTagName("topic");

    for (int i = 0; i < topics.count(); i++) {

	QDomNode node = topics.at(i);

	QString name = node.attributes().namedItem("name").nodeValue();
	if (name.isEmpty())
	    continue;

	QDomNodeList images = node.childNodes();

	QStandardItem *topic = new QStandardItem(name);
	topic->setTextAlignment(Qt::AlignCenter);
	topic->setEditable(false);

	QStandardItemModel *imageModel = new QStandardItemModel;
	topic->setData(qVariantFromValue((void *)imageModel), VMainWindow::ModelRole);

	for (int j = 0; j < images.count(); j++) {

	    QDomNamedNodeMap imgattrs = images.at(j).attributes();
	    QStandardItem *image;

	    image = new QStandardItem(imgattrs.namedItem("name").nodeValue());

	    image->setData(imgattrs.namedItem("author").nodeValue(),
			   VMainWindow::AuthorRole);
	    image->setData(imgattrs.namedItem("file").nodeValue(),
			   VMainWindow::PathRole);

	    image->setTextAlignment(Qt::AlignCenter);
	    image->setEditable(false);

	    imageModel->appendRow(image);
	    mSearchModel->appendRow(image->clone());
	}
	mModel->appendRow(topic);
    }
    mModel->sort(0);
    mSearchModel->sort(0);
}

void VMainWindow::topicClicked(const QModelIndex &index)
{
    QVariant variant = index.data(ModelRole);
    QStandardItemModel *model = (QStandardItemModel *)variant.value<void *>();

    mImageView->setModel(model);
    mImageView->setWindowTitle(QString("%1 | %2")
			       .arg(windowTitle())
			       .arg(QString(index.data().value<QString>())));
    mImageView->show();
}

void VMainWindow::imageClicked(const QModelIndex &index)
{
    setImage(index);
    mImageWidget->show();
}

void VMainWindow::setImage(const QModelIndex &index, int offset)
{
    QModelIndex next;

    if (offset != 0) {
	next = index.sibling(index.row() + offset, 0);
	if (!next.isValid())
	    return;

	mImageView->scrollTo(next);
	mImageView->setCurrentIndex(next);
    } else {
	next = index;
    }

    QString author = QString(" %1").arg(next.data(AuthorRole).toString());
    QString title = QString("%1 | %2")
	.arg(mImageView->windowTitle())
	.arg(next.data().toString());
    QPixmap pixmap = pixmapFromFile(next.data(PathRole).toString());

    mImageWidget->setWindowTitle(title);
    mImageWidget->setPixmap(pixmap);
    mImageWidget->setAuthor(author);
}

void VMainWindow::nextImage()
{
    setImage(mImageView->currentIndex(), 1);
}

void VMainWindow::previousImage()
{
    setImage(mImageView->currentIndex(), -1);
}

void VMainWindow::about()
{
    qDebug() << "About.";
}
