///////////////////////////////////////////////////////////////////////////////
//            Copyright (C) 2004-2011 by The Allacrost Project
//            Copyright (C) 2012-2013 by Bertram (Valyria Tear)
//                         All Rights Reserved
//
// This code is licensed under the GNU GPL version 2. It is free software
// and you may modify it and/or redistribute it under the terms of this license.
// See http://www.gnu.org/copyleft/gpl.html for details.
///////////////////////////////////////////////////////////////////////////////

/** ****************************************************************************
*** \file    tileset_editor.h
*** \author  Bar�� Soner U�akl? blackkknight@hotmail.com
*** \author  Yohann Ferreira, yohann ferreira orange fr
*** \brief   Header file for editor's tileset editor dialog
*******************************************************************************/

#ifndef __TILESET_EDITOR_HEADER__
#define __TILESET_EDITOR_HEADER__

#include <QGLWidget>
#include <QDialog>
#include <QMessageBox>
#include <QMouseEvent>
#include <QAction>
#include <QLabel>
#include <QPushButton>
#include <QGridLayout>
#include <QFileDialog>

#include "tileset.h"

namespace vt_editor
{

/** ****************************************************************************
*** \brief OpenGL widget display of a tileset within the tileset editor
***
*** This class represents the display of an open tileset within the tileset editor.
*** It is through this class that the user interacts with the open tileset image,
*** viewing and modifying its properties.
***
*** \todo The tileset pointer is really better suited to be a local, but for some
*** reason when you change it to a local, the app will seg fault when you close
*** the tileset editor. I believe this is because the destructor of this class
*** destroys the video engine singleton, and this is done before the tileset
*** object is destructed (which contain StillImage objects, that need the video
*** engine to be destroyed properly). Fix this in the future by not creating a
*** video engine instance for this tileset display.
*** ***************************************************************************/
class TilesetDisplay : public QGLWidget
{
public:
    TilesetDisplay();
    ~TilesetDisplay();

    //! \brief The current tileset that is being edited.
    Tileset *tileset;

protected:
    void initializeGL();
    void paintGL();
    void resizeGL(int w, int h);

    //! \name Mouse Processing Functions
    //! \brief Functions to process mouse events on the map.
    //! \note Reimplemented from QScrollView.
    //! \param evt A pointer to the QMouseEvent generated by the mouse.
    //{@
    void mousePressEvent(QMouseEvent *evt);
    void mouseMoveEvent(QMouseEvent *evt);
    void mouseReleaseEvent(QMouseEvent *evt);
    //@}

private:
    //! \brief A red, translucent square the size of 1/4th of a tile
    vt_video::StillImage _red_square;

    //! Tells the last edited square coords, used when using mouse drag to paint
    //! areas quickly.
    int32 _last_x;
    int32 _last_y;

    //! Tells whether the user was adding or removing a collision square
    //! when starting to drag the mouse.
    //! This is useful to keep on painting/removing the collisions squares
    //! instead of inverting the previous values blindly.
    bool _is_adding_collision;

    //! The function actually updating the tile collision values
    void _UpdateTiles(QMouseEvent *evt);

    //! Tells the given collision tile value
    bool _GetTileCollisionValue(QMouseEvent *evt);
}; // class TilesetDisplay : public QGLWidget


/** ****************************************************************************
*** \brief The primary class for the tileset editor
***
*** This class contains an instance of a TilesetDisplay, push buttons used for
*** modifying the properties of the tileset
*** ***************************************************************************/
class TilesetEditor : public QDialog
{
    //! Macro needed to use Qt's slots and signals.
    Q_OBJECT

public:
    //! \name TilesetEditor constructor
    //! \brief A constructor for the TilesetEditor class.This class is used to modify the tileset
    //! \definition files through an interface.
    //! \param parent The widget from which this dialog was invoked.
    TilesetEditor(QWidget *parent);
    ~TilesetEditor();

private slots:
    //! \brief Creates a new tileset definition file by loading a tileset image
    void _NewFile();

    //! \brief Loads a TDF file
    void _OpenFile();

    //! \brief Saves the modified tileset definition file
    void _SaveFile();

private:
    //! A push button for creating a new tileset
    QPushButton *_new_pbut;

    //! A push button for opening an existing tileset
    QPushButton *_open_pbut;

    //! A push button for saving the current tileset
    QPushButton *_save_pbut;

    //! A push button for exiting out of the tileset editor
    QPushButton *_exit_pbut;

    //! A layout to manage all the labels, spinboxes, and listviews.
    QGridLayout *_dia_layout;

    //! \brief The tileset display, also containing the tileset data itself
    TilesetDisplay *_tset_display;
}; // class TilesetEditor : public QDialog

} // namespace vt_editor

#endif
// __TILESET_EDITOR_HEADER__

