/**************************************************************************

    URPO

    Unix Remote Printing Operation
    Copyright (c) Arto Hyvättinen 2010

    This file is part of URPO.

    URPO is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    URPO is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.


**************************************************************************/

#include "printwidget.h"


#include <QLineEdit>
#include <QComboBox>
#include <QPushButton>
#include <QLabel>
#include <QProgressBar>
#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QFileDialog>
#include <QSpinBox>


PrintWidget::PrintWidget(QWidget *parent) :
    QWidget(parent)
{

    filenameEdit = new QLineEdit();
    browseButton = new QPushButton( tr("Browse"));
    connect(browseButton, SIGNAL(clicked()), this, SLOT(browseFile()));

    printerCombo = new QComboBox();
    printerCombo->setEnabled(false);
    printerCombo->setEditable(true);

    rangeEdit = new QLineEdit();
    // Page ranges in format 1-3,5,8-10
    // Valid characters: 0123456789 , +
    QRegExpValidator* rangeSensor = new QRegExpValidator( QRegExp("[0-9\\-,]+"), this );
    rangeEdit->setValidator( rangeSensor);

    // Copies: Spin 0..99
    copiesSpin = new QSpinBox();
    copiesSpin->setRange(1,99);
    copiesSpin->setValue(1);

    // Pages per sheet
    persheetCombo = new QComboBox();
    persheetCombo->insertItem(0,"1",1);
    persheetCombo->insertItem(1,"2",2);
    persheetCombo->insertItem(2,"4",4);

    printButton = new QPushButton( tr("Print"));
    printButton->setEnabled(false);
    connect( printButton, SIGNAL(clicked()), this, SLOT(doPrint()));

    statusLabel = new QLabel();
    progressBar = new QProgressBar();
    progressBar->setRange(0,0);
    cancelButton = new QPushButton(tr("Cancel"));
    connect(cancelButton,SIGNAL(clicked()),this,SIGNAL(cancel()));

    reconnectButton = new QPushButton( tr("Reconnect"));
    connect(reconnectButton, SIGNAL(clicked()), this, SIGNAL(reconnect()));


    QVBoxLayout* mainLayout = new QVBoxLayout();
    QHBoxLayout* fileLayout = new QHBoxLayout();
    fileLayout->addWidget( new QLabel( tr ("File") ));
    fileLayout->addWidget(filenameEdit);
    fileLayout->addWidget(browseButton);
    mainLayout->addLayout(fileLayout);

    QHBoxLayout* printerLayout = new QHBoxLayout();
    printerLayout->addWidget(new QLabel( tr("Printer")));
    printerLayout->addWidget(printerCombo);
    printerLayout->addStretch();
    mainLayout->addLayout(printerLayout);

    QHBoxLayout* rangeLayout = new QHBoxLayout();
    rangeLayout->addWidget(new QLabel(tr("Page range")));
    rangeLayout->addWidget(rangeEdit);
    mainLayout->addLayout(rangeLayout);

    QHBoxLayout* sheetLayout = new QHBoxLayout();
    sheetLayout->addWidget( new QLabel(tr("Pages per sheet")) );
    sheetLayout->addWidget(persheetCombo);
    sheetLayout->addStretch();
    sheetLayout->addWidget(new QLabel(tr("Copies")));
    sheetLayout->addWidget(copiesSpin);
    mainLayout->addLayout(sheetLayout);

    mainLayout->addStretch();

    mainLayout->addWidget(statusLabel);
    QHBoxLayout* barLayout = new QHBoxLayout;
    barLayout->addWidget(progressBar);
    barLayout->addStretch();
    barLayout->addWidget(cancelButton);
    barLayout->addWidget(reconnectButton);
    barLayout->addWidget(printButton);
    mainLayout->addLayout(barLayout);

    setLayout(mainLayout);

    progressBar->setVisible(false);

}

void PrintWidget::setStatus(QString message, bool busy)
{
    statusLabel->setText(message);
    if(busy)
    {
        // Busy: show progress bar, enable Cancel, disable others.
        progressBar->setVisible(true);
        cancelButton->setEnabled(true);
        reconnectButton->setEnabled(false);
    }
    else
    {
        progressBar->setVisible(false);
        reconnectButton->setEnabled(true);
        cancelButton->setEnabled(false);
    }
}

void PrintWidget::setPrinters(QStringList printers)
{
    printerCombo->clear();
    if(printers.isEmpty())
    {
        printerCombo->setEnabled(false);
    }
    else
    {
        printerCombo->addItems(printers);
        printerCombo->setEnabled(true);
        printerCombo->setCurrentIndex(0);
    }
}

void PrintWidget::setReady(bool ready)
{
    if(ready)
    {
        setStatus( tr("Ready"), false);
        printButton->setEnabled(true);
    }
    else
        printButton->setEnabled(false);
}


void PrintWidget::browseFile()
{
    QString path = QFileDialog::getOpenFileName(this, tr("Print file"));
    if(!path.isNull())
        filenameEdit->setText(path);

}

void PrintWidget::doPrint()
{
    // Make cups lp options
    QString options = QString("-d %1 -o number-up=").arg(printerCombo->currentText());
    options.append(persheetCombo->currentText());
    if( !rangeEdit->text().isEmpty())
        options += " -o page-ranges=" + rangeEdit->text();
    if( copiesSpin->value() > 1)
        options += QString(" -n %1").arg(copiesSpin->value());
    emit print(filenameEdit->text(), options);
}
