/*
 *  txtmsgtomail (Textmessage to EMail) statusbar plugin
 *  Copyright (C) 2011 Nicolai Hess
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <rtcom-eventlogger/eventlogger.h>
#include <gconf/gconf-client.h>
#include <dbus/dbus.h>
#include <string.h>
#include "txtmsgtomail_sp.h"

#define TXT_MSG_TO_MAIL_STATUS_PLUGIN_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE(obj, TYPE_TXT_MSG_TO_MAIL_STATUS_PUGIN, TxtMsgToMailStatusPluginPrivate))

#define QTM_MODEST_PLUGIN_SERVICE "com.nokia.Qtm.Modest.Plugin"
#define QTM_MODEST_PLUGIN_PATH "/com/nokia/Qtm/Modest/Plugin"
#define QTM_MODEST_PLUGIN_INTERFACE "com.nokia.Qtm.Modest.Plugin"
#define QTM_MODEST_PLUGIN_SEND_EMAIL "SendEmail"

#define GC_ROOT "/apps/maemo/txtmsgtomail"
#define TXTMSG_TO_MAIL_KEY_ENABLED GC_ROOT "/enabled"
#define TXTMSG_TO_MAIL_KEY_SENDER GC_ROOT "/mail_sender"
#define TXTMSG_TO_MAIL_KEY_ACCOUNT GC_ROOT "/mail_account"
#define TXTMSG_TO_MAIL_KEY_RECEIVER GC_ROOT "/mail_receiver"
#define TXTMSG_SUBJECT_TEMPLATE_KEY GC_ROOT "/subject_template"
#define TXTMSG_TO_MAIL_KEY_DISABLE_SMS_FORWARDING GC_ROOT "/disable_sms_forwarding"
#define TXTMSG_TO_MAIL_KEY_ENABLE_CHAT_FORWARDING GC_ROOT "/enable_chat_forwarding"
#define TXTMSG_TO_MAIL_KEY_DISABLE_INCOMING_FORWARDING GC_ROOT "/disable_incoming_forwarding"
#define TXTMSG_TO_MAIL_KEY_ENABLE_OUTGOING_FORWARDING GC_ROOT "/enable_outgoing_forwarding"

struct _TxtMsgToMailStatusPluginPrivate
{
  gboolean enabled;
  gboolean disable_sms_forwarding;
  gboolean enable_chat_forwarding;
  gboolean disable_incoming_forwarding;
  gboolean enable_outgoing_forwarding;
  gchar* mail_sender;
  gchar* mail_account;
  gchar* mail_receiver;
  gchar* subject_template;
  guint gconf_notify_handler;
  RTComEl *eventlogger;
  DBusGConnection* session_bus;
};

HD_DEFINE_PLUGIN_MODULE(TxtMsgToMailStatusPlugin, txt_msg_to_mail_status_plugin, HD_TYPE_STATUS_MENU_ITEM);

static void
send_mail(const gchar* remote_uid, const gchar* message, TxtMsgToMailStatusPlugin* plugin)
{
  if(!plugin->priv->enabled ||
     plugin->priv->mail_sender == NULL ||
     plugin->priv->mail_account == NULL ||
     plugin->priv->mail_receiver == NULL)
  {
    return;
  }
  DBusGProxy* dbus_proxy = NULL;
  if(plugin->priv->session_bus)
  {
    dbus_proxy = dbus_g_proxy_new_for_name(plugin->priv->session_bus,
					   QTM_MODEST_PLUGIN_SERVICE,
					   QTM_MODEST_PLUGIN_PATH,
					   QTM_MODEST_PLUGIN_INTERFACE);

    GHashTable* sender_info = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_free);
    g_hash_table_insert(sender_info, "from", g_strdup(plugin->priv->mail_sender));

    GHashTable* receiver_info = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_free);
    g_hash_table_insert(receiver_info, "to",g_strdup(plugin->priv->mail_receiver));

    GHashTable* text_info = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_free);
    g_hash_table_insert(text_info,"plain-body", g_strdup(message));
    
    if(plugin->priv->subject_template == NULL || strlen(plugin->priv->subject_template)==0)
    {
      g_hash_table_insert(text_info, "subject", g_strdup(remote_uid));
    }
    else
    {
      gchar** tile = g_strsplit(plugin->priv->subject_template, "$(SENDER)", -1);
      gchar* subject = g_strjoinv(remote_uid, tile);
      g_hash_table_insert(text_info, "subject", subject);
      g_strfreev(tile);
    }

    GPtrArray* dummy1 = g_ptr_array_new();
    GPtrArray* dummy2 = g_ptr_array_new();
    GHashTable* dummy3 = g_hash_table_new(NULL, NULL);

    dbus_g_proxy_call_no_reply(dbus_proxy, 
			       QTM_MODEST_PLUGIN_SEND_EMAIL,
			       dbus_g_type_get_map("GHashTable", G_TYPE_STRING, G_TYPE_STRING), sender_info,
			       dbus_g_type_get_map("GHashTable", G_TYPE_STRING, G_TYPE_STRING), receiver_info,
			       dbus_g_type_get_map("GHashTable", G_TYPE_STRING, G_TYPE_STRING), text_info,
			       dbus_g_type_get_collection("GPtrArray",
							  (dbus_g_type_get_map("GHashTable", 
									       G_TYPE_STRING, 
									       G_TYPE_STRING))), dummy1,
			       dbus_g_type_get_collection("GPtrArray",
							  (dbus_g_type_get_map("GHashTable", 
									       G_TYPE_STRING, 
									       G_TYPE_STRING))), dummy2,
			       G_TYPE_UINT, 0,
			       dbus_g_type_get_map("GHashTable", G_TYPE_STRING, G_TYPE_STRING), dummy3,
			       G_TYPE_INVALID,
			       G_TYPE_INVALID);
    g_object_unref(dbus_proxy);
  }
}

static void
_new_event_cb(RTComEl* backend,
              gint event_id,
              const gchar* local_uid,
              const gchar* remote_uid,
              const gchar* remote_ebook_uid,
              const gchar* group_uid,
              const gchar* service,
              gpointer user_data)
{
  TxtMsgToMailStatusPlugin* plugin = TXT_MSG_TO_MAIL_STATUS_PLUGIN(user_data);

  if((!plugin->priv->disable_sms_forwarding && g_strcmp0(service, "RTCOM_EL_SERVICE_SMS") == 0) ||
     (plugin->priv->enable_chat_forwarding && g_strcmp0(service, "RTCOM_EL_SERVICE_CHAT") == 0))
  {
    RTComElQuery *query = rtcom_el_query_new(plugin->priv->eventlogger);
    rtcom_el_query_prepare(query, "id", event_id, RTCOM_EL_OP_EQUAL, NULL);
    RTComElIter* iter = NULL;
    iter = rtcom_el_get_events(plugin->priv->eventlogger, query);
    g_object_unref(query);
    
    gboolean it = (iter != NULL) && rtcom_el_iter_first(iter);
    if(it)
    {
      gchar* text = NULL;
      gint outgoing;
      rtcom_el_iter_get_values(iter, 
			       "free-text", &text, 
			       "outgoing", &outgoing,
			       NULL);
      if(text)
      {
	if((outgoing == 0 && !plugin->priv->disable_incoming_forwarding) ||
	   (outgoing == 1 && plugin->priv->enable_outgoing_forwarding))
	{
	  send_mail(remote_uid, text, plugin);
	}
      }
      g_free(text);
    }
    if(iter)
      g_object_unref(iter);
  }
}

static void
_gconf_settings_changed(GConfClient* client,
			guint cnxn_id,
			GConfEntry* entry,
			gpointer user_data)
{
  TxtMsgToMailStatusPlugin* plugin = TXT_MSG_TO_MAIL_STATUS_PLUGIN(user_data);
  if(g_strcmp0(gconf_entry_get_key(entry), TXTMSG_TO_MAIL_KEY_ENABLED) == 0)
  {
    GConfValue* value = gconf_entry_get_value(entry);
    plugin->priv->enabled = gconf_value_get_bool(value);
  }
  if(g_strcmp0(gconf_entry_get_key(entry), TXTMSG_TO_MAIL_KEY_DISABLE_SMS_FORWARDING) == 0)
  {
    GConfValue* value = gconf_entry_get_value(entry);
    plugin->priv->disable_sms_forwarding = gconf_value_get_bool(value);
  }
  if(g_strcmp0(gconf_entry_get_key(entry), TXTMSG_TO_MAIL_KEY_ENABLE_CHAT_FORWARDING) == 0)
  {
    GConfValue* value = gconf_entry_get_value(entry);
    plugin->priv->enable_chat_forwarding = gconf_value_get_bool(value);
  }
  if(g_strcmp0(gconf_entry_get_key(entry), TXTMSG_TO_MAIL_KEY_DISABLE_INCOMING_FORWARDING) == 0)
  {
    GConfValue* value = gconf_entry_get_value(entry);
    plugin->priv->disable_incoming_forwarding = gconf_value_get_bool(value);
  }
  if(g_strcmp0(gconf_entry_get_key(entry), TXTMSG_TO_MAIL_KEY_ENABLE_OUTGOING_FORWARDING) == 0)
  {
    GConfValue* value = gconf_entry_get_value(entry);
    plugin->priv->enable_outgoing_forwarding = gconf_value_get_bool(value);
  }
  if(g_strcmp0(gconf_entry_get_key(entry), TXTMSG_TO_MAIL_KEY_SENDER) == 0)
  {
    GConfValue* value = gconf_entry_get_value(entry);
    if(plugin->priv->mail_sender)
      g_free(plugin->priv->mail_sender);
    if(gconf_value_get_string(value)!=NULL)
      plugin->priv->mail_sender = g_strdup(gconf_value_get_string(value));
  }
  if(g_strcmp0(gconf_entry_get_key(entry), TXTMSG_TO_MAIL_KEY_RECEIVER) == 0)
  {
    GConfValue* value = gconf_entry_get_value(entry);
    if(plugin->priv->mail_receiver)
      g_free(plugin->priv->mail_receiver);
    if(gconf_value_get_string(value)!=NULL)
      plugin->priv->mail_receiver = g_strdup(gconf_value_get_string(value));
  }
  if(g_strcmp0(gconf_entry_get_key(entry), TXTMSG_SUBJECT_TEMPLATE_KEY) == 0)
  {
    GConfValue* value = gconf_entry_get_value(entry);
    if(plugin->priv->subject_template)
      g_free(plugin->priv->subject_template);
    if(gconf_value_get_string(value)!=NULL)
      plugin->priv->subject_template = g_strdup(gconf_value_get_string(value));
  }
  if(g_strcmp0(gconf_entry_get_key(entry), TXTMSG_TO_MAIL_KEY_ACCOUNT) == 0)
  {
    GConfValue* value = gconf_entry_get_value(entry);
    if(plugin->priv->mail_account)
      g_free(plugin->priv->mail_account);
    if(gconf_value_get_string(value)!=NULL)
      plugin->priv->mail_account = g_strdup(gconf_value_get_string(value));
  }
}

static void _register_gconf_changes(TxtMsgToMailStatusPlugin* plugin)
{
  GConfClient* client = NULL;
  GError* error = NULL;
  client = gconf_client_get_default();
  if(!GCONF_IS_CLIENT(client))
  {
    return;
  }
  gconf_client_add_dir(client,
		       GC_ROOT,
		       GCONF_CLIENT_PRELOAD_NONE,
		       &error);
  if(error != NULL)
  {
    g_warning("can not listen on gconf client changes %s\n", error->message);
    g_error_free(error);
    error = NULL;
    g_object_unref(client);
    return;
  }
  plugin->priv->gconf_notify_handler = 
    gconf_client_notify_add(client, GC_ROOT, _gconf_settings_changed, plugin, NULL, &error);
  if(error != NULL)
  {
    g_warning("can not add notify on gconf client %s\n", error->message);
    g_error_free(error);
    error = NULL;
    g_object_unref(client);
    return;
  }
  g_object_unref(client);
}

static void
_unregister_gconf_changes(TxtMsgToMailStatusPlugin* plugin)
{
  GConfClient* client = NULL;
  client = gconf_client_get_default();
  if(GCONF_IS_CLIENT(client))
  {
    gconf_client_notify_remove(client, 
			       plugin->priv->gconf_notify_handler);
    plugin->priv->gconf_notify_handler = 0;
    gconf_client_remove_dir(client, GC_ROOT,
			    NULL);
    g_object_unref(client);
  }
}

static void
_read_gconf_settings(TxtMsgToMailStatusPlugin* plugin)
{
  GConfClient* client = gconf_client_get_default();
  if(!GCONF_IS_CLIENT(client))
    return;
  plugin->priv->enabled = gconf_client_get_bool(client, TXTMSG_TO_MAIL_KEY_ENABLED, NULL);
  plugin->priv->disable_sms_forwarding = gconf_client_get_bool(client, TXTMSG_TO_MAIL_KEY_DISABLE_SMS_FORWARDING, NULL);
  plugin->priv->enable_chat_forwarding = gconf_client_get_bool(client, TXTMSG_TO_MAIL_KEY_ENABLE_CHAT_FORWARDING, NULL);
  plugin->priv->disable_incoming_forwarding = gconf_client_get_bool(client, TXTMSG_TO_MAIL_KEY_DISABLE_INCOMING_FORWARDING, NULL);
  plugin->priv->enable_outgoing_forwarding = gconf_client_get_bool(client, TXTMSG_TO_MAIL_KEY_ENABLE_OUTGOING_FORWARDING, NULL);
  if(plugin->priv->mail_sender)
  {
    g_free(plugin->priv->mail_sender);
  }
  if(plugin->priv->mail_receiver)
  {
    g_free(plugin->priv->mail_receiver);
  }
  if(plugin->priv->subject_template)
  {
    g_free(plugin->priv->subject_template);
  }
  if(plugin->priv->mail_account)
  {
    g_free(plugin->priv->mail_account);
  }
  plugin->priv->mail_sender = gconf_client_get_string(client, TXTMSG_TO_MAIL_KEY_SENDER, NULL);
  plugin->priv->mail_account = gconf_client_get_string(client, TXTMSG_TO_MAIL_KEY_ACCOUNT, NULL);
  plugin->priv->mail_receiver = gconf_client_get_string(client, TXTMSG_TO_MAIL_KEY_RECEIVER, NULL);
  plugin->priv->subject_template = gconf_client_get_string(client, TXTMSG_SUBJECT_TEMPLATE_KEY, NULL);
  g_object_unref(client);
}

static void
txt_msg_to_mail_status_plugin_finalize(GObject* object)
{
  TxtMsgToMailStatusPlugin* plugin = TXT_MSG_TO_MAIL_STATUS_PLUGIN(object);

  if(plugin->priv->gconf_notify_handler != 0)
    _unregister_gconf_changes(plugin);
  if(plugin->priv->mail_sender)
  {
    g_free(plugin->priv->mail_sender);
    plugin->priv->mail_sender = NULL;
  }
  if(plugin->priv->mail_account)
  {
    g_free(plugin->priv->mail_account);
    plugin->priv->mail_account = NULL;
  }
  if(plugin->priv->mail_receiver)
  {
    g_free(plugin->priv->mail_receiver);
    plugin->priv->mail_receiver = NULL;
  }
  if(plugin->priv->subject_template)
  {
    g_free(plugin->priv->subject_template);
    plugin->priv->subject_template = NULL;
  }
  if(plugin->priv->eventlogger)
    g_object_unref(plugin->priv->eventlogger);
  if(plugin->priv->session_bus)
    dbus_g_connection_unref(plugin->priv->session_bus);
  G_OBJECT_CLASS(txt_msg_to_mail_status_plugin_parent_class)->finalize(object);
}

static void
txt_msg_to_mail_status_plugin_class_finalize(TxtMsgToMailStatusPluginClass* klass)
{
  
}

static void
txt_msg_to_mail_status_plugin_init(TxtMsgToMailStatusPlugin* plugin)
{
  plugin->priv = TXT_MSG_TO_MAIL_STATUS_PLUGIN_GET_PRIVATE(plugin);
  plugin->priv->gconf_notify_handler = 0;
  plugin->priv->enabled = FALSE;
  plugin->priv->disable_sms_forwarding = FALSE;
  plugin->priv->enable_chat_forwarding = FALSE;
  plugin->priv->disable_incoming_forwarding = FALSE;
  plugin->priv->enable_outgoing_forwarding = FALSE;
  plugin->priv->mail_sender = NULL;
  plugin->priv->mail_account = NULL;
  plugin->priv->mail_receiver = NULL;
  plugin->priv->subject_template = NULL;
  plugin->priv->eventlogger = rtcom_el_new();
  plugin->priv->session_bus = hd_status_plugin_item_get_dbus_g_connection(HD_STATUS_PLUGIN_ITEM(&plugin->parent),
									  DBUS_BUS_SESSION, 
									  NULL);
  _register_gconf_changes(plugin);
  _read_gconf_settings(plugin);
  g_signal_connect(plugin->priv->eventlogger,
		   "new-event",
		   G_CALLBACK(_new_event_cb),
		   plugin);
}


static void
txt_msg_to_mail_status_plugin_class_init(TxtMsgToMailStatusPluginClass* klass)
{
  g_type_class_add_private(klass, sizeof(TxtMsgToMailStatusPluginPrivate));
  G_OBJECT_CLASS(klass)->finalize = (GObjectFinalizeFunc)txt_msg_to_mail_status_plugin_finalize;
}
