unit txkinscroller;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils
  {$ifdef LCLQT}
  ,qtwidgets, qt4
  {$endif}
  ,ExtCtrls, Controls, lclclasses;

type
  TTxKinScroller = class(TComponent)
  private
    FEnabled : boolean;
    FTargetControl : TWinControl;
    FkinTimer : TTimer;
    FkinMouseDownPoint : TPoint;
    FkinMouseCurrentPoint : TPoint;
    FkinMouseDownPointInertia : TPoint;
    FkinMouseCurrentPointInertia : TPoint;
    FkinOriginalScrollBarValue : TPoint;
    FkinCurrentScrollBarValue : TPoint;
    FkinMouseIsDown : boolean;
    FkinMouseIsMoving : boolean;
    FkinInertiaActive : boolean;
    FkinInertiaY : integer;
    FkinInertiaX : integer;
    FkinIgnoreRadius : integer;
    FDoInvalidate : boolean;
    procedure kinStopInertia;
    function GetTimerInterval:cardinal;
    procedure SetTimerInterval(value : cardinal);
    procedure SetTargetControl(value : TWinControl);
    procedure FkinTimerTimer(Sender: TObject);
  public
    constructor Create(AOwner: TComponent);
    destructor Destroy;
    procedure MouseDown(X, Y: Integer);
    procedure MouseMove(X, Y: Integer);
    procedure MouseUp(X, Y: Integer);
  published
    property TimerInterval : cardinal read GetTimerInterval write SetTimerInterval default 50;
    property TargetControl : TWinControl read FTargetControl write SetTargetControl;
    property Enabled : boolean read FEnabled write FEnabled default False;
    property MoveIgnoreRadius : integer read FkinIgnoreRadius write FkinIgnoreRadius default 20;
    property InertiaActive : boolean read FkinInertiaActive;
    property DoInvalidate : boolean read FDoInvalidate write FDoInvalidate default False;
  end;

implementation


constructor TTxKinScroller.Create(AOwner: TComponent);
begin
  inherited;
  FKinTimer := TTimer.Create(Self);
  FKinTimer.Enabled := False;
  FKinTimer.OnTimer:= @FkinTimerTimer;
  SetTimerInterval(50);
  FKinIgnoreRadius := 20;
  FEnabled := False;
end;

destructor TTxKinScroller.Destroy;
begin
  FKinTimer.Free;
  inherited Destroy;
end;

procedure TTxKinScroller.kinStopInertia;
begin
  FkinInertiaActive := False;
  FkinInertiaY := 0;
  FkinInertiaX := 0;
  FkinTimer.Enabled := False;
end;

function TTxKinScroller.GetTimerInterval:cardinal;
begin
  Result := FkinTimer.Interval;
end;

procedure TTxKinScroller.SetTimerInterval(value : cardinal);
begin
  FKinTimer.Interval := value;
end;

procedure TTxKinScroller.SetTargetControl(value : TWinControl);
begin
  if value = nil then
    Exit;
  FTargetControl := value;
  FEnabled := True;
end;

procedure TTxKinScroller.MouseDown(X, Y: Integer);
begin
  if not FEnabled then
    Exit;
  FkinMouseDownPoint.X := X;
  FkinMouseDownPoint.Y := Y;
  FkinMouseDownPointInertia.X := X;
  FkinMouseDownPointInertia.Y := Y;
  FkinOriginalScrollBarValue.Y := TQtAbstractScrollArea(FTargetControl.Handle).verticalScrollBar.getValue;
  FkinOriginalScrollBarValue.X := TQtAbstractScrollArea(FTargetControl.Handle).horizontalScrollBar.getValue;
  FkinMouseIsDown := True;
  if FDoInvalidate then
    FTargetControl.Invalidate;
  if FkinInertiaActive then
    kinStopInertia;
end;

procedure TTxKinScroller.MouseMove(X, Y: Integer);
var
  SArea : TQtAbstractScrollArea;
  Dist : TPoint;
  SAreaSize : TPoint;
  amountY, amountX : integer;
begin
  if not FEnabled then
    Exit;
  FkinMouseCurrentPoint.X := X;
  FkinMouseCurrentPoint.Y := Y;
  FkinMouseCurrentPointInertia.X := X;
  FkinMouseCurrentPointInertia.Y := Y;
  //Check if outside the no-go radius...
  if FkinMouseDownPoint.Y <> -1 then //Check if CurPos is accurate first...
    if (FkinMouseCurrentPoint.Y > (FkinMouseDownPoint.Y+FkinIgnoreRadius)) or (FkinMouseCurrentPoint.Y < (FkinMouseDownPoint.Y-FkinIgnoreRadius)) or
       (FkinMouseCurrentPoint.X > (FkinMouseDownPoint.X+FkinIgnoreRadius)) or (FkinMouseCurrentPoint.X < (FkinMouseDownPoint.X-FkinIgnoreRadius)) then
      begin//We have moved outside the radius, do some moving...
        if not FkinInertiaActive then
          begin // Inertia...
            FkinInertiaActive := True;
            if not FkinTimer.Enabled then
              FkinTimer.Enabled := True;
          end
        else
          begin // Manual drag...
            // Get the Y distance
            if (FkinMouseCurrentPoint.Y > FkinMouseDownPoint.Y) then
              dist.Y := (FkinMouseCurrentPoint.Y - FkinMouseDownPoint.Y)
            else
              dist.Y := (FkinMouseDownPoint.Y - FkinMouseCurrentPoint.Y)*-1;
            // Get the X distance
            if (FkinMouseCurrentPoint.X > FkinMouseDownPoint.X) then
              dist.X := (FkinMouseCurrentPoint.X - FkinMouseDownPoint.X)
            else
              dist.X := (FkinMouseDownPoint.X - FkinMouseCurrentPoint.X)*-1;

            // Get the ScrollArea
            SArea := TQtAbstractScrollArea(FTargetControl.Handle);
            // Do the math
            SAreaSize.Y := Round((FTargetControl.Height*((SArea.verticalScrollBar.getMax-SArea.verticalScrollBar.getMin)+SArea.verticalScrollBar.getPageStep))/SArea.verticalScrollBar.getPageStep);
            amountY := Round((Dist.Y * SArea.verticalScrollBar.getMax) / SAreaSize.Y);

            SAreaSize.X := Round((FTargetControl.Width*((SArea.horizontalScrollBar.getMax-SArea.horizontalScrollBar.getMin)+SArea.horizontalScrollBar.getPageStep))/SArea.horizontalScrollBar.getPageStep);
            amountX := Round((Dist.X * SArea.horizontalScrollBar.getMax) / SAreaSize.X);

            // Scroll...
            SArea.verticalScrollBar.setValue(FkinOriginalScrollBarValue.Y-amountY);
            SArea.horizontalScrollBar.setValue(FkinOriginalScrollBarValue.X-amountX);
            if FDoInvalidate then
              begin
                FTargetControl.DoAdjustClientRectChange();
              end;
          end;
     end;
end;

procedure TTxKinScroller.MouseUp(X, Y: Integer);
begin
  if not FEnabled then
    Exit;
  FkinMouseDownPoint.X := -1;
  FkinMouseDownPoint.Y := -1;
  FkinMouseIsDown := False;
end;

procedure TTxKinScroller.FkinTimerTimer(Sender: TObject);
var
  SArea : TQtAbstractScrollArea;
  SAreaSize : TPoint;
  tempY, tempX : integer;
begin
  if FkinMouseIsDown and FkinInertiaActive then
    begin // Gather some energy...
      SArea := TQtAbstractScrollArea(FTargetControl.Handle);
      SAreaSize.Y := Round((FTargetControl.Height*((SArea.verticalScrollBar.getMax-SArea.verticalScrollBar.getMin)+SArea.verticalScrollBar.getPageStep))/SArea.verticalScrollBar.getPageStep);
      SAreaSize.X := Round((FTargetControl.Height*((SArea.horizontalScrollBar.getMax-SArea.horizontalScrollBar.getMin)+SArea.horizontalScrollBar.getPageStep))/SArea.horizontalScrollBar.getPageStep);

      tempY := (FkinMouseCurrentPointInertia.Y - FkinMouseDownPointInertia.Y);
      tempX := (FkinMouseCurrentPointInertia.X - FkinMouseDownPointInertia.X);
      // Do the math..
      FkinInertiaY := Round((tempY * SArea.verticalScrollBar.getMax) / SAreaSize.Y);
      FkinMouseDownPointInertia.Y := FkinMouseCurrentPointInertia.Y;

      FkinInertiaX := Round((tempX * SArea.horizontalScrollBar.getMax) / SAreaSize.X);
      FkinMouseDownPointInertia.X := FkinMouseCurrentPointInertia.X;
    end
  else if (not FkinMouseIsDown) and FkinInertiaActive then
    begin // Release the energy...

      // Apply the Y axis friction..
      if FkinInertiaY > 0 then
        Dec(FkinInertiaY)
      else
        Inc(FkinInertiaY);
      // Apply the X axis friction..
      if FkinInertiaX > 0 then
        Dec(FkinInertiaX)
      else
        Inc(FkinInertiaX);

      // Dont boomerang..
      if abs(FkinInertiaY) < 2 then
        FkinInertiaY := 0;
      if abs(FkinInertiaX) < 2 then
        FkinInertiaX := 0;

      if (FkinInertiaY = 0) and (FkinInertiaX = 0) then
        kinStopInertia;

      // Scroll it..
      SArea := TQtAbstractScrollArea(FTargetControl.Handle);

      SArea.verticalScrollBar.setValue(SArea.verticalScrollBar.getValue-FkinInertiaY);
      SArea.horizontalScrollBar.setValue(SArea.horizontalScrollBar.getValue-FkinInertiaX);
      if FDoInvalidate then
        begin
          FTargetControl.DoAdjustClientRectChange();
        end;
    end
  else
    begin // Do nothing, stop the motion...(reset);
      kinStopInertia;
    end;
end;

end.

