/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "twittertweetinfo.h"

/*! \brief TwitterTweetInfo constructor.
 *  \param _username Twitter username
 *  \param _text Text of respective username
 *  \param _imageUrl Profile image of respective username
 */
TwitterTweetInfo::TwitterTweetInfo(Tweet *tweet, QGraphicsWidget *parent)
                : TwitterApplicationItem(parent),
                  _tweet(tweet),
                  _entry(tweet->getEntry()),
                  _defaultImage(QImage(":images/resources/avatar.png")),
                  _startPos(QPointF(0, 0)),
                  _endPos(QPointF(0, 0)),
                  _anim(false)
{
    font.setFamily("Nokia Sans");
    font.setStyle(QFont::StyleNormal);
    font.setStyleStrategy(QFont::PreferAntialias);
    font.setWeight(QFont::Normal);
    font.setPixelSize(18);

    _urls = new KineticList(KineticList::Vertical, QSizeF(800,195),this);
    _urls->setScrollBarPolicy(KineticList::Vertical,Qt::ScrollBarAlwaysOff);

    _container = new TwitterApplicationItem(NULL);

    if(tweet->getImage().isNull())
        _image = _defaultImage;

    _paintTweet();
    _createButtons();
    _showUrls(_createUrls(_entry.originalText));

    connect(_tweet, SIGNAL(imageDownloaded()), this, SLOT(_paintAvatar()));
    connect(_backButton, SIGNAL(released()),SLOT(_exitAnimation()));

    // Change Orientation
    _changeOrientation();
    connect(QApplication::desktop(), SIGNAL(resized(int)), SLOT(_changeOrientation()));
}

/*!
 * \brief Overridden.
 * \param painter Painter object.
 * \param option Styles to drawing.
 */
void TwitterTweetInfo::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *)
{
    Q_UNUSED(option);

    // Draw the white rect
    painter->setPen(Qt::NoPen);
    painter->setBrush(Qt::white);
    painter->drawRect(0, 60, size().width(), size().height() - 60);

    // Draw avatar
    painter->drawImage(QRect(13, 73, 48, 48), _image);
}

/*!
 * \brief This function parses the message and puts all URLs that were found in a list.
 * \return URL list.
 */
QStringList TwitterTweetInfo::_createUrls(QString text)
{
    QStringList textSplitted = text.split(" ");
    QStringList listUrls;

    for(int i=0; i<textSplitted.count(); i++)
        if(textSplitted.at(i).contains(QRegExp("((https?|ftp)://[^ ]+)( ?)", Qt::CaseInsensitive)))
            listUrls << textSplitted.at(i);

    return listUrls;
}

/*!
 * \brief This function creates all buttons on the detailed tweet screen.
 */
void TwitterTweetInfo::_createButtons()
{
    _backButton = new TwitterButton(TwitterButton::Back, "", this);
    _replyButton = new TwitterButton(TwitterButton::Reply  , NULL, this);
    _retweetButton = new TwitterButton(TwitterButton::Retweet, NULL, this);

    if(_entry.isOwn) {
        _deleteButton = new TwitterButton(TwitterButton::Delete, NULL , this);
        connect(_deleteButton,SIGNAL(released()), this, SLOT(_deleteTweet()));
    }

    connect(_retweetButton, SIGNAL(released()), this, SLOT(_retweet()));
    connect(_replyButton, SIGNAL(released()), this, SLOT(_reply()));
}

void TwitterTweetInfo::_paintTweet(void)
{
    _tweetText = new QGraphicsTextItem(this);
    _tweetText->setDefaultTextColor(QColor(0, 0, 0));
    _tweetText->setDocument(_tweet->getTweetDocument());
    _tweetText->setPos(74,60);

    _sep1 = new QGraphicsLineItem(this);
    _sep1->setPen(QPen(QColor(216, 216, 216)));

    _sep2 = new QGraphicsLineItem(this);
    _sep2->setPen(QPen(QColor(216, 216, 216)));

    _paintAvatar();
}

void TwitterTweetInfo::_reply(void)
{
    QString replyMessage = "@" + _entry.userInfo.screenName;
    emit reply(replyMessage, _entry.id);
}

void TwitterTweetInfo::_retweet(void)
{
    QString retweetMessage = "RT: @" + _entry.userInfo.screenName + " " + _entry.originalText;
    if (retweetMessage.length() > 140)
        retweetMessage.remove(136, retweetMessage.length() - 136);

    emit retweet(retweetMessage);
}

void TwitterTweetInfo::_paintAvatar(void)
{
    _image = _tweet->getImage();
    update();
}

void TwitterTweetInfo::_showUrls(QStringList urls)
{
    font.setPixelSize(18);

    _message = new QGraphicsTextItem(_container);
    _message->setFont(font);
    _message->setDefaultTextColor(QColor(0, 0, 0));
    _message->setPos(12, 6);

    if(urls.isEmpty()) {
        _message->setPlainText("No links available in this tweet.");
    } else {
        _message->setPlainText("Links found in this tweet: ");

        TwitterButton *urlButton /*, *imageButton = NULL*/;

        for(int i = 0; i < urls.count(); i++) {
            urlButton = new TwitterButton(TwitterButton::Url, urls.at(i), _container);
            urlButton->setPos(15, 40 + 62*i);

//            if(_twitterImage->isImageSupported(urls.at(i))) {
//                imageButton = new TwitterButton(TwitterButton::TwitterImage, urls.at(i), _container);
//                imageButton->setPos(288, 40 + 62*i);
//                connect(imageButton, SIGNAL(imageUrl(QString)), SLOT(_openImage(QString)));
//            }
        }
        _buttonsHeight = urlButton->pos().y() + urlButton->boundingRect().height();
    }
    _urls->addToList(_container);
}

void TwitterTweetInfo::_deleteTweet()
{
    _exitAnimation();
    emit del(_entry.id, _tweet);
}

void TwitterTweetInfo::startAnimation(void)
{
    _anim = true;

    QPropertyAnimation *mvWhiteRect = new QPropertyAnimation(this,"pos");
    mvWhiteRect->setStartValue(_startPos);
    mvWhiteRect->setEndValue(_endPos);
    mvWhiteRect->setDuration(250);
    mvWhiteRect->setEasingCurve(QEasingCurve::Linear);
    mvWhiteRect->start();
}

void TwitterTweetInfo::_exitAnimation(void)
{
    _anim = false;

    QPropertyAnimation *mvWhiteRect = new QPropertyAnimation(this,"pos");
    mvWhiteRect->setStartValue(_endPos);
    mvWhiteRect->setEndValue(_startPos);
    mvWhiteRect->setDuration(250);
    mvWhiteRect->setEasingCurve(QEasingCurve::Linear);
    mvWhiteRect->start();

    connect(mvWhiteRect, SIGNAL(finished()),SLOT(deleteLater()));
}

void TwitterTweetInfo::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    Q_UNUSED(event);
}

void TwitterTweetInfo::_openImage(QString url)
{
//    _twitterImage->downloadImage(url);
}

void TwitterTweetInfo::_changeOrientation(void)
{
    QRect screenGeometry = QApplication::desktop()->screenGeometry();
    resize(screenGeometry.size());

    if(screenGeometry.width() > screenGeometry.height()) {
        _startPos = QPointF(0, 480);
    } else if(screenGeometry.width() < screenGeometry.height()) {
        _startPos = QPointF(0, 800);
    }

    if(_anim) {
        setPos(_endPos);
    } else {
        setPos(_startPos);
    }

    _tweetText->setDocument(_tweet->getTweetDocument());
    _sep1->setLine(7, _tweetText->pos().y() + _tweetText->boundingRect().height() + 1,
                   size().width() - 7, _tweetText->pos().y() + _tweetText->boundingRect().height() + 1);

    _backButton->setPos(size().width() - _backButton->size().width(), 0);
    _replyButton->setPos(size().width() - _replyButton->boundingRect().width() - 10, _sep1->line().y1()+12);
    _retweetButton->setPos(_replyButton->pos().x() - _retweetButton->boundingRect().width() - 10, _sep1->line().y1()+12);

    _sep2->setLine(_sep1->line().p1().x(), _sep1->line().p1().y()+86,
                    _sep1->line().p2().x(), _sep1->line().p2().y()+86);

    _container->setMinimumSize(-1, -1);
    _container->setMaximumSize(-1, -1);

    int _auxHeight = _buttonsHeight;
    if(_buttonsHeight < size().height() - _sep2->line().y1() - 120)
        _auxHeight = size().height() - _sep2->line().y1() - 120;

    _container->setMinimumSize(size().width(), _auxHeight);
    _container->setMaximumSize(_container->minimumSize());

    _urls->setPos(0, _sep2->line().y1() + 1);
    _urls->setSize(QSizeF(size().width(), size().height() - _urls->pos().y() - 60));

    if(_entry.isOwn) {
        _deleteButton->setPos(_retweetButton->pos().x() - _deleteButton->boundingRect().width() - 10, _sep1->line().y1()+12);
    }
}
