/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "kineticlist.h"

KineticList::KineticList(Orientation orientation, QSizeF size, QGraphicsWidget *parent)
    : TwitterApplicationItem(parent),
    _or(orientation),
    _sel(-1)
{
    this->setMinimumSize(size);
    this->setFlag(QGraphicsItem::ItemHasNoContents, true);
    this->setFlag(QGraphicsItem::ItemClipsChildrenToShape, true);
    this->setFlag(QGraphicsItem::ItemClipsToShape, true);

    // Create view
    _view = new QGraphicsView;
    _view->setFixedSize(size.width(),size.height());
    _view->move(0,0);

    _view->setAttribute(Qt::WA_TranslucentBackground, true);
    QPalette p = _view->viewport()->palette();
    p.setColor(QPalette::Base, Qt::transparent);
    _view->viewport()->setPalette(p);

    _view->setViewportUpdateMode(QGraphicsView::BoundingRectViewportUpdate);
    _view->setOptimizationFlags(QGraphicsView::DontSavePainterState);
    _view->setFrameShape(QFrame::NoFrame);
    _view->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);

    // Kinetic scroll
    #ifdef Q_WS_MAEMO_5
        QAbstractKineticScroller *scroller = _view->property("kineticScroller").value<QAbstractKineticScroller *>();
        if (scroller)
            scroller->setEnabled(true);
    #endif

    // Create scene and add to the view
    _scene = new QGraphicsScene;
    _view->setScene(_scene);

    // Create QGraphicsWidget that will be the list
    _list = new QGraphicsWidget();

    // Set its layout
    if (orientation == KineticList::Horizontal) {
        _layout = new QGraphicsLinearLayout(Qt::Horizontal,_list);
        _layout->setMinimumHeight(size.height());
    } else {
        _layout = new QGraphicsLinearLayout(Qt::Vertical,_list);
        _layout->setMinimumWidth(size.width());
    }
    _layout->setSpacing(3);
    _layout->setContentsMargins(2, 0, 2, 0);

    // Add list to the scene
    _scene->addItem(_list);

    // Create view proxy to add to the screen
    _viewProxy = new QGraphicsProxyWidget(this);
    _viewProxy->setPos(0,0);
    _viewProxy->setWidget(_view);

}

void KineticList::removeFromList(int pos)
{
    // Remove Item
    QGraphicsLayoutItem *item;
    item = _layout->itemAt(pos);
    _layout->removeAt(pos);
    delete item;

    resizeScene();
}

void KineticList::removeFromList(TwitterApplicationItem *item)
{
    _layout->removeItem(item);
    delete item;

    resizeScene();
}

void KineticList::clearList(void)
{
    while (_layout->count() > 0)
        removeFromList(_layout->count()-1);
}

void KineticList::addToList(TwitterApplicationItem *item, int pos)
{
    _layout->insertItem(pos,item);

    // handle selection
    connect(item, SIGNAL(selected(int)), SLOT(unselect(int)));
    connect(item, SIGNAL(createTweetInfo(Tweet*)),SIGNAL(createTweetInfo(Tweet*)));

    resizeScene();
}

void KineticList::unselect(int pos)
{
    for(int i=0; i<_layout->count(); ++i)
    {
        if(pos != i)
        {
            QGraphicsLayoutItem *item;
            item = _layout->itemAt(i);
            (dynamic_cast<User*>(item))->unselect();
        }
    }
    _sel = pos;
    emit selected(true);
}

int KineticList::getSelected(void)
{
    return _sel;
}

void KineticList::setScrollBarPolicy(Orientation orientation, Qt::ScrollBarPolicy policy)
{
    if (orientation == KineticList::Vertical)
        _view->setVerticalScrollBarPolicy(policy);
    else
        _view->setHorizontalScrollBarPolicy(policy);
}

void KineticList::setSelected(int pos)
{
    QGraphicsLayoutItem *item;
    item = _layout->itemAt(pos);
    (dynamic_cast<User*>(item))->select();
    _sel = pos;
}

void KineticList::addMoreButton()
{
    _moreButton = new TwitterButton(TwitterButton::More,"More Tweets",NULL);
    _moreButton->setOpacity(1.0);
    addToList(_moreButton,_layout->count());

    connect(_moreButton, SIGNAL(released()), SLOT(_moreTweets()));
}

void KineticList::_moreTweets()
{
    removeFromList(_layout->count()-1);
    emit moreTweets();
}

void KineticList::resizeScene(void)
{
    // Set scene height
    if (_or == KineticList::Vertical)
        _scene->setSceneRect(0,0,_view->width()-10, _layout->minimumHeight());
    else
        _scene->setSceneRect(0, 0, _layout->minimumWidth(), _view->height());
}
