/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "twitpic.h"

/*! \brief TwitPic constructor.
 */
TwitPic::TwitPic(QGraphicsWidget *parent) :
        TwitterApplicationItem(parent)
{
    m_manager = new QNetworkAccessManager( this );
    connect(m_manager, SIGNAL(finished(QNetworkReply*)), this, SLOT(handleNetworkRequest(QNetworkReply*)));

    setGeometry(0, 0, 800, 480);
    setMinimumSize(800, 480);
    setMaximumSize(800, 480);
}

/*!
 * \brief TwitPic destructor.
 */
TwitPic::~TwitPic()
{
}

/*!
 * \brief This function sends the request to download the profile image
 * \param Url the url where the profile image is hosted
 */
void TwitPic::download(QString Url)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    m_isImage = false;
    m_paintImage = false;

    QNetworkRequest request;
    request.setUrl(Url);
    m_manager->get(request);
}

/*!
 * \brief This function handles the reply and loads the image
 * \param reply The network reply
 */
void TwitPic::handleNetworkRequest(QNetworkReply *reply)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    if(m_isImage) {
        if(!m_image.load(reply,0))
            m_image = QImage();
        m_paintImage = true;
        update();
    }
    else
        parseHtml(QString(reply->readAll()));

    reply->deleteLater();
}

void TwitPic::parseHtml(QString html)
{
    QString a;
    a = html.section("img src=\"", 2, 2);
    QStringList stringList = a.split("\"");

    downloadImage(stringList[0]);
}

void TwitPic::downloadImage(QString imageUrl)
{
    m_isImage = true;

    QNetworkRequest request;

    QByteArray array = imageUrl.toAscii();
    QUrl newUrl;
    newUrl.setEncodedUrl(array);

    request.setUrl(newUrl);
    qDebug() << newUrl;
    m_manager->get(request);
}

/*!
 * \brief Overridden.
 * \param painter Painter object.
 * \param option Styles to drawing.
 */
void TwitPic::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *)
{
    Q_UNUSED(option);

    painter->setBrush(QBrush(QColor(0, 0, 0, 150)));
    painter->setPen(Qt::NoPen);
    painter->drawRoundedRect(rect(), 5, 5);

    /* Draw Image */
    if (m_paintImage) {
        m_imageScaled = m_image.scaled(760, 440, Qt::KeepAspectRatio);
        painter->drawImage((800 - m_imageScaled.width()) / 2, 20, m_imageScaled);
    }
}

/*!
 * \brief Overridden.
 */
void TwitPic::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
//    if(contains(event->pos()))
//        update();
}

/*!
 * \brief Overridden.
 */
void TwitPic::mouseReleaseEvent(QGraphicsSceneMouseEvent *event)
{
    int exitSpace = (800 - m_imageScaled.width()) / 2;
    if(event->pos().x() < exitSpace || event->pos().x() > (m_imageScaled.width() + exitSpace)) {
        emit animExit();
        return;
    }

    if(m_paintImage) {
        QMessageBox msgBox;
        msgBox.setText("Save image?");
        msgBox.setStandardButtons(QMessageBox::Save | QMessageBox::Cancel);
        msgBox.setDefaultButton(QMessageBox::Save);
        int ret = msgBox.exec();

        if(ret == QMessageBox::Save) {
            QFileDialog dialog;
            #ifdef LIBLOCATION
            QString fileName = dialog.getSaveFileName(0, tr("Save File"), "/home/user/.images/untitled.jpg", tr("Images (*.png *.gif *.jpg)"));
            #else
            QString fileName = dialog.getSaveFileName(0, tr("Save File"), "/home/roda/untitled.jpg", tr("Images (*.png *.gif *.jpg)"));
            #endif

            if(fileName != NULL)
                    m_image.save(fileName);
            else
                return;
        }
        else
            if(ret == QMessageBox::Cancel)
                return;
    }
    else
        emit animExit();
}
