/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "timelinectrl.h"
#include "applicationItems/tweet.h"

/*! \brief Constructs a TimelineCtrl.
 */
TimelineCtrl::TimelineCtrl(QGraphicsScene *scene)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    _scene = scene;

    _timelineView = new TimelineView();

    isFollowing = true;
    updateTimeline = false;
    isGPS = false;
    pageToDownload = 2;
    mentionsPageToDownload = 2;
    addHome = true;
    addMentions = true;

    connect(_timelineView, SIGNAL(postUpdate(void)), this, SLOT(_postNewUpdate(void)));
    connect(_timelineView, SIGNAL(favorite(QString, bool)), this, SLOT(_handleFavorite(QString, bool)));
    connect(_timelineView, SIGNAL(deleteTweet(QString, int)), this, SLOT(_handleDeleteTweet(QString, int)));
    connect(_timelineView, SIGNAL(logout()), this, SLOT(_logout()));
    connect(_timelineView, SIGNAL(unfollow(QString, bool)), this, SLOT(_handleFriendship(QString, bool)));
    connect(_timelineView, SIGNAL(done()),this, SLOT(deleteLater()));
    connect(_timelineView, SIGNAL(changeAvatar(QString)), this, SLOT(_updateProfileImage(QString)));

    connect(_timelineView, SIGNAL(showLocationDialog()), this, SLOT(_dialogLocation()));
    connect(_timelineView, SIGNAL(gotGPS()), this, SLOT(_handleGPSLocation()));

    connect(Backend::getInstance(), SIGNAL(updatePosted(void)), this, SLOT(_updateTimeline(void)));
    connect(Backend::getInstance(), SIGNAL(updateError(QString)), this, SLOT(_updateWarning(QString)));
    connect(Backend::getInstance(), SIGNAL(connectionError(QString)), this, SLOT(_refreshWarning(QString)));

    connect(_timelineView, SIGNAL(addHomeTimelineToScene(StatusList *)), this,
            SLOT(_addHomeTimelineToScene(StatusList *)));
    connect(_timelineView, SIGNAL(addMentionsTimelineToScene(StatusList*)), this,
            SLOT(_addMentionsTimelineToScene(StatusList*)));
    connect(_timelineView, SIGNAL(removeHomeTimelineFromScene(StatusList*)), this,
            SLOT(_removeHomeTimelineFromScene(StatusList*)));
    connect(_timelineView, SIGNAL(removeMentionsTimelineFromScene(StatusList*)), this,
            SLOT(_removeMentionsTimelineFromScene(StatusList*)));

    connect(Backend::getInstance(), SIGNAL(statusMessage(const QString&, const QString&, const QString&,
                                                         const QString&, const QString&, const QString&,
                                                         const bool&, const QString&, bool, bool)),
            _timelineView->m_homeTimeline, SLOT(_addToList(const QString&, const QString&, const QString&,
                                                          const QString&, const QString&, const QString&,
                                                          const bool&, const QString&, bool, bool)));

    connect(Backend::getInstance(), SIGNAL(mentionsStatusMessage(const QString&, const QString&, const QString&,
                                                         const QString&, const QString&, const QString&,
                                                         const bool&, const QString&, bool, bool)),
            _timelineView->m_mentionsTimeline, SLOT(_addToList(const QString&, const QString&, const QString&,
                                                          const QString&, const QString&, const QString&,
                                                          const bool&, const QString&, bool, bool)));

    connect(_timelineView, SIGNAL(moreButtonHome(StatusList *)), this, SLOT(_moreTweetsToHome(StatusList *)));
    connect(_timelineView, SIGNAL(moreButtonMentions(StatusList *)), this, SLOT(_moreTweetsToMentions(StatusList *)));

    showView();

    animationDuration = 700;
    easingCurve.setType(QEasingCurve::OutCirc);
    easingCurve.setOvershoot(2.5);

    _createListsScene();
}

/*!
 * \brief TimelineCtrl destructor.
 */
TimelineCtrl::~TimelineCtrl()
{
    qDebug() << "~  " << __PRETTY_FUNCTION__;

    if(_timelineView)
        delete _timelineView;

    emit done();
}

/*!
 *\brief This function adds the timelineview to the scene.
 */
void TimelineCtrl::showView()
{
    _timelineView->createUi();
    _scene->addItem(_timelineView);
}

/*!
 *\brief This function gets what user typed and updates as a new status and calls
 *       timelineview to update user stats and timeline.
 */
void TimelineCtrl::_postNewUpdate()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    bool updated = false;

    if(_timelineView->getHome())
        _timelineView->m_homeTimeline->moreButtonEnable(false);
    else
        _timelineView->m_mentionsTimeline->moreButtonEnable(false);

    if (_timelineView->getUpdateMessage() != "") {
        if(!_timelineView->_locationCheckBox->isChecked()) {
            if(!_timelineView->isReply)
                updated = Backend::getInstance()->update(_timelineView->getUpdateMessage(), NULL, NULL);
            else {
                updated = Backend::getInstance()->update(_timelineView->getUpdateMessage(), NULL, NULL, _timelineView->m_statusId);
                qDebug() << "status id = " << _timelineView->m_statusId;
                _timelineView->isReply = false;
                _timelineView->m_statusId.clear();
            }
        }
        else {
            if(!isGPS) {
                if(!_timelineView->isReply) {
                    updated = Backend::getInstance()->update(_timelineView->getUpdateMessage(),
                                                     QString::number(_timelineView->_ipLocation->getLat()),
                                                     QString::number(_timelineView->_ipLocation->getLon()));
                }
                else {
                    updated = Backend::getInstance()->update(_timelineView->getUpdateMessage(),
                                                     QString::number(_timelineView->_ipLocation->getLat()),
                                                     QString::number(_timelineView->_ipLocation->getLon()), _timelineView->m_statusId);
                    _timelineView->isReply = false;
                    _timelineView->m_statusId.clear();
                }
            }
            else {
                if(!_timelineView->isReply)
                    updated = Backend::getInstance()->update(_timelineView->getUpdateMessage(), gpsLatitude, gpsLongitude);
                else {
                    updated = Backend::getInstance()->update(_timelineView->getUpdateMessage(), gpsLatitude, gpsLongitude, _timelineView->m_statusId);
                    _timelineView->isReply = false;
                    _timelineView->m_statusId.clear();
                }
            }
        }
    } else
        _genericError("The message cannot be empty");

    if(updated)
        _timelineView->setUpdateMessage("") ;

    if(_timelineView->getHome())
        _timelineView->m_homeTimeline->moreButtonEnable(true);
    else
        _timelineView->m_mentionsTimeline->moreButtonEnable(true);
}

/*!
 *\brief This function handles any connection errors that might occur.
 *\param message message that will appear on the messagebox.
 */
void TimelineCtrl::_connectionError(QString message)
{
    dialog = new DialogView(DialogView::Error, message);
    _scene->addItem(dialog);
    _timelineView->connectionError();
}

/*!
 *\brief This function provides an error messagebox to any error that might occur.
 *\param message message that will appear on the messagebox.
 */
void TimelineCtrl::_genericError(QString message)
{
    dialog = new DialogView(DialogView::Error, message);
    _scene->addItem(dialog);
    _timelineView->endLoader();
}

/*!
 *\brief This function shows an error while trying to update the status and asks
 * if the user wants to retry or quit.
 *\param message message that will appear on the messagebox.
 */
void TimelineCtrl::_updateWarning(QString message)
{
    dialog = new DialogView(DialogView::Warning,message);
    connect(dialog, SIGNAL(yesPressed()), this, SLOT(_postNewUpdate()));
    connect(dialog, SIGNAL(noPressed()), _timelineView, SLOT(endLoader()));
    _scene->addItem(dialog);
}

/*!
 *\brief This function shows an error while trying to refresh the time and asks
 * if the user wants to retry or quit.
 *\param message message that will appear on the messagebox.
 */
void TimelineCtrl::_refreshWarning(QString message)
{
    dialog = new DialogView(DialogView::Warning,message);
    connect(dialog, SIGNAL(yesPressed()), this, SLOT(_updateTimeline()));
    connect(dialog, SIGNAL(noPressed()), _timelineView, SLOT(_backToLogin()));
    _scene->addItem(dialog);
}

/*!
 *\brief Gets status id from selected status and adds it as favorite.
 */
void TimelineCtrl::_handleFavorite(QString tweetId,bool isFavorite)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    bool success;
    success = Backend::getInstance()->addFavorite(tweetId.toLongLong(), isFavorite);

    if(success) {
        _timelineView->_tweetInfo->updateTweetInfo("Favorite", !isFavorite);
        _timelineView->endLoader();
    }
    else
        this->_genericError("Could not favorite tweet.");
}

/*!
 *\brief Gets status id from selected status and deletes it.
 */
void TimelineCtrl::_handleDeleteTweet(QString tweetId, int posInList)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    bool success;

    if(_timelineView->getHome())
        _timelineView->m_homeTimeline->moreButtonEnable(false);
    else
        _timelineView->m_mentionsTimeline->moreButtonEnable(false);

    _timelineView->changeToTimeline();

    success = Backend::getInstance()->removeStatus(tweetId.toLongLong());
    if(success) {
        _timelineView->m_homeTimeline->removeFromList(posInList);
        this->_updateTimeline();
    }
    else
        this->_genericError("Could not remove status.");

    if(_timelineView->getHome())
        _timelineView->m_homeTimeline->moreButtonEnable(true);
    else
        _timelineView->m_mentionsTimeline->moreButtonEnable(true);
}

/*!
 *\brief Gets username from selected status and removes/adds it from/to following list.
 */
void TimelineCtrl::_handleFriendship(QString username, bool unfollow)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    bool success;

    if(unfollow) {
        success = Backend::getInstance()->removeFriendship(username);
        _updateTimeline();
    }
    else {
        success = Backend::getInstance()->addFriendship(username);
        _updateTimeline();
    }

    if(success) {
        _timelineView->_tweetInfo->updateTweetInfo("Follow",!unfollow);
        _timelineView->m_homeTimeline->populateFollowing(username, !unfollow);
    }
    else
        this->_genericError("Error");
}

/*!
 *\brief Shows a confirmation dialog when user tries to logout.
 */
void TimelineCtrl::_logout()
{
    dialog = new DialogView(DialogView::Warning, QString("Are you sure you want to quit?"));
    connect(dialog, SIGNAL(yesPressed()), _timelineView, SLOT(_backToLogin()));
    _scene->addItem(dialog);
}

/*!
 * \brief Clears all information on the current list
 * and creates a new list with updated information.
 */
void TimelineCtrl::_updateTimeline()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    Backend::getInstance()->verifyCredentials();
    _timelineView->updateTopBar();
    _timelineView->endLoader();
}

/*!
 * \brief Updates the user avatar.
 * \param image User's new image.
 */
void TimelineCtrl::_updateProfileImage(QString image)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    bool newImage = Backend::getInstance()->updateProfileImage(image);

    if(newImage) {
        _timelineView->updateTopBar();
        this->_genericError("The new avatar has been updated!\nThe new avatar will be loaded when you perform another action.");
    }
    else
        this->_genericError("The new avatar could not be updated.");
}

/*!
 * \brief Shows the dialog when user clicks on the location checkbox and calls the location method especified by the user
 */
void TimelineCtrl::_dialogLocation()
{
    if(!_timelineView->_locationCheckBox->isChecked()) {
        dialog = new DialogView(DialogView::Location,"Choose one method to obtain your coordinates");
        connect(dialog, SIGNAL(getIP()), _timelineView, SLOT(getIPCoordinates()));
        connect(dialog, SIGNAL(getGPS()), _timelineView, SLOT(getGPSCoordinates()));
        connect(dialog, SIGNAL(cancelPressed()), _timelineView, SLOT(setLocationCheckboxFalse()));
        _scene->addItem(dialog);
    }
    else {
        isGPS = false;
        #ifdef LIBLOCATION
            _timelineView->_gpsLocation->stop();
        #endif // LIBLOCATION
    }
}

/*!
 * \brief Saves latitude and longitude in a string and shows a banner when acquiring and stopping GPS
 */
void TimelineCtrl::_handleGPSLocation()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    #ifdef LIBLOCATION
    if(_timelineView->_locationCheckBox->isChecked())
        QMaemo5InformationBox::information(NULL, tr("GPS information updated!"), QMaemo5InformationBox::DefaultTimeout);
    else
        QMaemo5InformationBox::information(NULL, tr("GPS stopped"), QMaemo5InformationBox::DefaultTimeout);

    gpsLatitude = QString::number(_timelineView->_gpsLocation->getLat());
    gpsLongitude = QString::number(_timelineView->_gpsLocation->getLon());
    #endif // LIBLOCATION
    isGPS = true;
}

void TimelineCtrl::_createListsScene()
{
    _timelineView->view2->setFixedSize(800, 480 - _timelineView->_locationCheckBox->boundingRect().height() -
                       _timelineView->_logoutButton->boundingRect().height() - 8);
    _timelineView->view2->move(0, _timelineView->_logoutButton->boundingRect().height() + 4);

    this->_scene->addWidget(_timelineView->view2);

    QPalette p = _timelineView->view2->viewport()->palette();
    p.setColor(QPalette::Base, Qt::transparent);
    _timelineView->view2->viewport()->setPalette(p);
    _timelineView->view2->setAttribute(Qt::WA_TranslucentBackground, true);

    #ifdef LIBLOCATION
        QAbstractKineticScroller *scroller = _timelineView->view2->property("kineticScroller")
                                             .value<QAbstractKineticScroller *>();
        if (scroller)
            scroller->setEnabled(true);
    #endif // LIBLOCATION
}

void TimelineCtrl::_addHomeTimelineToScene(StatusList *homeTimeline)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    QGraphicsScene *homeScene = new QGraphicsScene;
    m_scene = homeScene;

    _timelineView->view2->setScene(homeScene);

    Backend::getInstance()->setFriendsPage(0);
    Backend::getInstance()->getTimeline(Timeline::Friends);
    homeScene->addItem(homeTimeline);
    homeTimeline->addMoreButton();
    _timelineView->endLoader();
}

void TimelineCtrl::_addMentionsTimelineToScene(StatusList *mentionsTimeline)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    QGraphicsScene *mentionsScene = new QGraphicsScene;
    m_scene = mentionsScene;

    _timelineView->view2->setScene(mentionsScene);

    Backend::getInstance()->setMentionsPage(0);
    Backend::getInstance()->getMentions();
    mentionsScene->addItem(mentionsTimeline);
    mentionsTimeline->addMoreButton();
    _timelineView->endLoader();
}

void TimelineCtrl::_addDMTimelineToScene(StatusList *dmTimeline)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

//    Backend::getInstance()->getTimeline(Timeline::Friends);
//    dmScene->addItem(dmTimeline);
}

void TimelineCtrl::_removeHomeTimelineFromScene(StatusList *homeTimeline)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    m_scene->removeItem(homeTimeline);
    delete m_scene;
    homeTimeline->clearList();
}

void TimelineCtrl::_removeMentionsTimelineFromScene(StatusList *mentionsTimeline)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    m_scene->removeItem(mentionsTimeline);
    delete m_scene;
    mentionsTimeline->clearList();
}

void TimelineCtrl::_removeDMTimelineFromScene(StatusList *dmTimeline)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    m_scene->removeItem(dmTimeline);
}

void TimelineCtrl::_moreTweetsToHome(StatusList *homeTimeline)
{
    Backend::getInstance()->setFriendsPage(pageToDownload++);
    Backend::getInstance()->getTimeline(Timeline::Friends);
    homeTimeline->removeMoreButton();
    homeTimeline->addMoreButton();
    _timelineView->endLoader();
}

void TimelineCtrl::_moreTweetsToMentions(StatusList *mentionsTimeline)
{
    Backend::getInstance()->setMentionsPage(mentionsPageToDownload++);
    Backend::getInstance()->getMentions();
    mentionsTimeline->removeMoreButton();
    mentionsTimeline->addMoreButton();
    _timelineView->endLoader();
}
