/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "twitterbutton.h"

/*! \brief TwitterButton constructor.
 *  \param buttonType
 *  \param buttonLabel String to be written on the button.
 */
TwitterButton::TwitterButton(ButtonType buttonType, QString buttonLabel, QGraphicsWidget *parent)
        : TwitterApplicationItem(parent),
        _buttonLabel(buttonLabel)
{
    _pen.setColor(QColor(15, 149, 218));
    _buttonType = buttonType;
    m_isPressed = false;

    setTypeButton(buttonType);
}

/*!
 * \brief Overridden.
 */
void TwitterButton::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    if(contains(event->pos())) {
        m_isPressed = true;
        emit pressed();
    }

    update();
}

/*!
 * \brief Overridden.
 */
void TwitterButton::mouseReleaseEvent(QGraphicsSceneMouseEvent *event)
{
    if(!contains(event->pos()))
        return;

    m_isPressed = false;
    update();

    if(_url.isEmpty())
        emit released();
    else
        emit released(_url);
}

/*!
 * \brief Overridden.
 */
void TwitterButton::mouseMoveEvent(QGraphicsSceneMouseEvent *event)
{
    if(contains(event->pos()))
        m_isPressed = true;
    else
        m_isPressed = false;

    update();
}

/*!
 * \brief Overridden.
 */
QRectF TwitterButton::boundingRect() const
{
    return _buttonImage.rect();
}

/*!
 * \brief Overridden.
 */
QPainterPath TwitterButton::shape() const
{
    QPainterPath path;
    //path.addRoundedRect(boundingRect(), 28, 28);
    path.addRect(boundingRect());
    return path;

}

/*!
 * \brief Overridden.
 * \param painter Painter object.
 * \param option Styles to drawing.
 */
void TwitterButton::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *)
{
    bool down = option->state & QStyle::State_Sunken;

    if (down && m_isPressed) {
        painter->translate(2, 2);
        painter->setOpacity(0.7);

        switch(_buttonType) {
            case Update:
                _buttonSelectedImage = QPixmap(":images/resources/buttons/update_pressed.png");
                break;
            case Expose:
                _buttonSelectedImage = QPixmap(":images/resources/buttons/expose_pressed.png");
                break;
            case Logout:
                _buttonSelectedImage = QPixmap(":images/resources/buttons/logout_pressed.png");
                break;
            case Generic:
                _buttonSelectedImage = QPixmap(":images/resources/buttons/generic_pressed.png");
                break;
            case Retweet:
                _buttonSelectedImage = QPixmap(":images/resources/buttons/rt_pressed.png");
                break;
            case Reply:
                _buttonSelectedImage = QPixmap(":images/resources/buttons/reply_pressed.png");
                break;
            case GoUrl:
                _buttonSelectedImage = QPixmap(":images/resources/buttons/go-to-url_pressed.png");
                break;
            case Delete:
                _buttonSelectedImage = QPixmap(":images/resources/buttons/delete_pressed.png");
                break;
            case Back:
                _buttonSelectedImage = QPixmap(":images/resources/buttons/back-pressed.png");
                break;
            case Favorite:
                _buttonSelectedImage = QPixmap(":images/resources/buttons/favorite_pressed.png");
                break;
            case NoFavorite:
                _buttonSelectedImage = QPixmap(":images/resources/buttons/no-fave_pressed.png");
                break;
            case More:
                _buttonSelectedImage = QPixmap(":images/resources/buttons/more-pressed.png");
                break;
        }

    } else
        _buttonSelectedImage = 0;

    painter->drawPixmap(0, 0, _buttonImage);
    painter->setOpacity(1.0);
    painter->drawPixmap(0, 0, _buttonSelectedImage);

    QFont font;
    font.setFamily("Nokia Sans");
    font.setStyle(QFont::StyleNormal);
    font.setStyleStrategy(QFont::PreferAntialias);
    font.setWeight(QFont::Bold);

    font.setPixelSize(16);

    painter->setFont(font);
    painter->setPen(_pen);
    painter->drawText(boundingRect(), Qt::TextWordWrap | Qt::AlignCenter, _buttonLabel);

}

/*!
 * \brief This function enables or disables a button.
 * \param option true to enable and false to disable button.
 */
void TwitterButton::enable(bool option)
{
    if(!option)
        _pen.setColor(QColor(117, 117, 117));
    else
        _pen.setColor(QColor(15, 149, 218));
    this->setEnabled(option);
}

/*!
 * \brief This function sets the button type, changing the image.
 * \param buttonType The type of button.
 */
void TwitterButton::setTypeButton(int buttonType)
{
    switch(buttonType){
        case Update:
            _buttonImage = QPixmap(":images/resources/buttons/update.png");
            break;
        case Expose:
            _buttonImage = QPixmap(":images/resources/buttons/expose.png");
            break;
        case Logout:
            _buttonImage = QPixmap(":images/resources/buttons/logout.png");
            break;
        case Generic:
            _buttonImage = QPixmap(":images/resources/buttons/generic.png");
            break;
        case Retweet:
            _buttonImage = QPixmap(":images/resources/buttons/rt.png");
            break;
        case Reply:
            _buttonImage = QPixmap(":images/resources/buttons/reply.png");
            break;
        case Favorite:
            _buttonImage = QPixmap(":images/resources/buttons/favorite.png");
            break;
        case NoFavorite:
            _buttonImage = QPixmap(":images/resources/buttons/no-fave.png");
            break;
        case GoUrl:
            _buttonImage = QPixmap(":images/resources/buttons/go-to-url.png");
            break;
        case Delete:
            _buttonImage = QPixmap(":images/resources/buttons/delete.png");
            break;
        case Back:
            _buttonImage = QPixmap(":images/resources/buttons/back.png");
            break;
        case Tab:
            _buttonImage = QPixmap(":images/resources/buttons/tab.png");
            break;
        case More:
            _buttonImage = QPixmap(":images/resources/buttons/more.png");
            break;
    }
    this->_buttonType = buttonType;
    update();
}

/*!
 * \brief This function changes a button's label.
 * \param label The button's label.
 */
void TwitterButton::setLabel(QString label)
{
    this->_buttonLabel = label;
}

/*!
 * \brief This function sets the url in the button.
 * \param response The button's url.
 */
void TwitterButton::setUrl(QString url)
{
    this->_url = url;
}
