/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "statuslist.h"

/*!
 * \brief Class Constructor
 * \param parent The list parent
 */
StatusList::StatusList(QGraphicsWidget *parent)
    : TwitterApplicationItem(parent),
      m_visibleRect(QRectF(0, 0, 0, 0))
{
    m_layout = new QGraphicsLinearLayout(Qt::Vertical, this);
    m_layout->setSpacing(3);
    m_layout->setContentsMargins(0, 0, 0, 0);
    setLayout(m_layout);
}

/*!
 * \brief Add the More Button to the end of the timeline
 */
void StatusList::addMoreButton()
{
    m_moreButton = new TwitterButton(TwitterButton::More, "More", this);
    connect(m_moreButton, SIGNAL(released()), this, SLOT(_moreTweets()));

    m_moreButton->setOpacity(1.0);

    m_layout->addItem(m_moreButton);
}

/*!
 * \brief Add a new tweet to the list
 * \param name The username
 * \param text The tweet message
 * \param imageUrl The user's avatar url
 * \param statusId The tweet id
 * \param createdAt When the tweet was created
 * \param source Which app was used when creating the message
 * \param favorite If the tweet is favorite
 * \param coordinates Coordinates where the tweet was made
 * \param isMyTweet If the tweet belongs to the user
 */
void StatusList::_addToList(const QString &name, const QString &text, const QString &imageUrl,
                            const QString &statusId, const QString &createdAt, const QString &source,
                            const bool &favorite, const QString &coordinates, bool isMyTweet, bool isUpdate)
{
    Tweet *status = new Tweet(name, text, imageUrl, statusId, createdAt, source, favorite, coordinates, isMyTweet, m_layout->count(), this);
    status->setGeometry(0, 0, 776, 74);
    if(!isUpdate)
        m_layout->addItem(status);
    else {
        m_layout->insertItem(0, status);
        status->setPosition(0);
    }

    connect(status, SIGNAL(createTweetInfo(Tweet*)), this, SIGNAL(createTweetInfo(Tweet*)));
}

/*!
 * \brief Sets the visible area in the scrolling.
 * \param rect Visible rect.
 */
void StatusList::setVisibleRect(const QRectF &rect)
{
    m_visibleRect = rect;
}

/*!
 * \brief Gets the visible rect of the list.
 */
QRectF StatusList::getVisibleRect() const
{
    return m_visibleRect;
}

/*!
 * \brief Clears the timeline
 */
void StatusList::clearList()
{
    if(m_moreButton)
        delete m_moreButton;

    QGraphicsLayoutItem *status;
    while ((status = m_layout->itemAt(0)) != 0)
        delete status;
}

/*!
 * \brief Removes the more button from the timeline
 */
void StatusList::removeMoreButton()
{
    delete m_moreButton;
}

/*!
 * \brief Removes a tweet from the timeline
 * \param position The position in the timeline
 */
void StatusList::removeFromList(int position)
{
    QGraphicsLayoutItem *status2;
    status2 = m_layout->itemAt(position);
    m_layout->removeAt(position);
    delete status2;
}

/*!
 * \brief Emits the signal to download a new Tweet
 */
void StatusList::_moreTweets()
{
    this->moreButtonEnable(false);
    emit this->morePressed();
}

/*!
 * \brief Used to change the follow button in tweet
 * \param username The tweet's username
 * \param follow The follow flag
 */
void StatusList::populateFollowing(QString username, bool follow)
{
    Tweet *tweet;
    for(int i = 0; i < m_layout->count(); i++) {
        tweet = (Tweet *) m_layout->itemAt(i);
        QString screenName = tweet->getUsername();
        if(screenName == username)
            tweet->setFollowing(screenName, follow);
    }
}

void StatusList::moreButtonEnable(bool enable)
{
    m_moreButton->enable(enable);
}
