/* Copyright (C) 2006 - 2010 Jan Kundrát <jkt@gentoo.org>

   This file is part of the Trojita Qt IMAP e-mail client,
   http://trojita.flaska.net/

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or the version 3 of the License.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include <QAuthenticator>
#include <QDesktopServices>
#include <QDir>
#include <QDockWidget>
#include <QHeaderView>
#include <QInputDialog>
#include <QItemSelectionModel>
#include <QMenuBar>
#include <QMessageBox>
#include <QProgressBar>
#include <QScrollArea>
#include <QSplitter>
#include <QStatusBar>
#include <QToolBar>
#include <QToolButton>
#include <QUrl>
#include <QSortFilterProxyModel>
#include <QMaemo5InformationBox>

#include "MessageListWindow.h"
#include "MessageWindow.h"
#include "ComposeWidget.h"
#include "ProtocolLoggerWidget.h"
#include "MessageView.h"
#include "MsgListView.h"
#include "SettingsDialog.h"
#include "SettingsNames.h"
#include "Imap/Model/Model.h"
#include "Imap/Model/MailboxModel.h"
#include "Imap/Model/MailboxTree.h"
#include "Imap/Model/ModelWatcher.h"
#include "Imap/Model/MsgListModel.h"
#include "Imap/Model/CombinedCache.h"
#include "Imap/Model/MemoryCache.h"
#include "Imap/Model/PrettyMailboxModel.h"
#include "Streams/SocketFactory.h"

#include "iconloader/qticonloader.h"

/** @short All user-facing widgets and related classes */
namespace Gui {

using namespace Imap::Mailbox;

MessageListWindow::MessageListWindow(const QModelIndex& index, Model* model, QWidget* parent): 
    BaseWindow(model, parent), messageWindow(NULL), model(model)
{
    TreeItemMailbox* mbox = dynamic_cast<TreeItemMailbox*>( Model::realTreeItem( index ));
    Q_ASSERT( mbox );
    if ( ! mbox->isSelectable() )
        return;
    
    setWindowTitle( mbox->mailbox() );
    createWidgets();

    QSettings s;
    if ( ! s.contains( SettingsNames::imapMethodKey ) ) {
        QTimer::singleShot( 0, this, SLOT(slotShowSettings()) );
    }
    
    setupModels();
    createActions();
    createMenus();

    updateActionsOnlineOffline(model->isNetworkAvailable());

    // Setup message list model
    msgListModel->setMailbox(mbox, model);
    
    // Please note that Qt 4.6.1 really requires passing the method signature this way, *not* using the SLOT() macro
    QDesktopServices::setUrlHandler( QLatin1String("mailto"), this, "slotComposeMailUrl" );
}

void MessageListWindow::createActions()
{
    resyncMbox = new QAction( QtIconLoader::icon( QLatin1String("view-refresh") ), tr("Check for New Messages"), this );
    connect( resyncMbox, SIGNAL(triggered()), this, SLOT(slotResyncMbox()) );

    expunge = new QAction( QtIconLoader::icon( QLatin1String("trash-empty") ),  tr("Empty Trash"), this );
    connect( expunge, SIGNAL(triggered()), this, SLOT(slotExpunge()) );

    markAsRead = new QAction( QtIconLoader::icon( QLatin1String("mail-mark-read") ),  tr("Mark as Read"), this );
    markAsRead->setCheckable( true );
    markAsRead->setShortcut( Qt::Key_M );
    msgListTree->addAction( markAsRead );
    connect( markAsRead, SIGNAL(triggered(bool)), this, SLOT(handleMarkAsRead(bool)) );

    markAsDeleted = new QAction( QtIconLoader::icon( QLatin1String("list-remove") ),  tr("Mark as Deleted"), this );
    markAsDeleted->setCheckable( true );
    markAsDeleted->setShortcut( Qt::Key_Delete );
    msgListTree->addAction( markAsDeleted );
    connect( markAsDeleted, SIGNAL(triggered(bool)), this, SLOT(handleMarkAsDeleted(bool)) );
}

void MessageListWindow::createMenus()
{
    QMenuBar* menubar = menuBar();
    QMenu* imapMenu = menubar->addMenu( tr("IMAP") );
    imapMenu->addAction( composeMail );
    imapMenu->addSeparator()->setText( tr("Network Access") );
    QMenu* netPolicyMenu = imapMenu->addMenu( tr("Network Access"));
    netPolicyMenu->addAction( netOffline );
    netPolicyMenu->addAction( netExpensive );
    netPolicyMenu->addAction( netOnline );
    imapMenu->addSeparator();
    imapMenu->addAction( configSettings );
    
    QMenu* mailboxMenu = menubar->addMenu( tr("Mailbox") );
    mailboxMenu->addAction( resyncMbox );
    mailboxMenu->addAction( expunge );
}

void MessageListWindow::createWidgets()
{
    msgListTree = new MsgListView(this);
    msgListTree->setContextMenuPolicy(Qt::CustomContextMenu);
    setCentralWidget(msgListTree);

    connect( msgListTree, SIGNAL( customContextMenuRequested( const QPoint & ) ),
            this, SLOT( showContextMenuMsgListTree( const QPoint& ) ) );
}

class ReverseProxy : public QSortFilterProxyModel {
    public:
        ReverseProxy(QObject * parent = 0) : QSortFilterProxyModel( parent ) {};
        virtual QModelIndex mapFromSource  ( const QModelIndex & sourceIndex ) const {
            if(!sourceIndex.model())
                return QModelIndex();
            int row = sourceIndex.model()->rowCount()-sourceIndex.row()-1;
            int col = sourceIndex.column();
            return index(row, col);
        }
        
        virtual QModelIndex mapToSource ( const QModelIndex & proxyIndex ) const {
            if(!sourceModel())
                return QModelIndex();
            int row = sourceModel()->rowCount()-proxyIndex.row()-1;
            int col = proxyIndex.column();
            return sourceModel()->index(row, col);
        }
        
        virtual QVariant data ( const QModelIndex  & proxyIndex, int role = Qt::DisplayRole ) const {
            if(role == Qt::ToolTipRole)
                return QVariant();
            
            QVariant qv = QSortFilterProxyModel::data(proxyIndex, role);
            if(qv.type() == QVariant::DateTime ) {
                Q_ASSERT(qv.canConvert(QVariant::DateTime));
                QDateTime dt = qv.toDateTime();
                
                if ( QDateTime::currentDateTime().toTime_t() - dt.toTime_t() < 24 * 60 * 60 /* seconds in a day */)
                    return dt.time().toString( Qt::SystemLocaleShortDate );
                else
                    return dt.date().toString( Qt::DefaultLocaleShortDate );
            }
            return qv;
        }
};

void MessageListWindow::setupModels()
{
    model->setObjectName( QLatin1String("model") );
    msgListModel = new Imap::Mailbox::MsgListModel( this, model );
    msgListModel->setObjectName( QLatin1String("msgListModel") );

    connect( msgListModel, SIGNAL( mailboxChanged() ), this, SLOT( slotResizeMsgListColumns() ) );
    connect( msgListModel, SIGNAL( dataChanged(QModelIndex,QModelIndex) ), this, SLOT( updateMessageFlags() ) );
    connect( msgListTree, SIGNAL( clicked(const QModelIndex&) ), this, SLOT( msgListClicked(const QModelIndex&) ) );

    connect( model, SIGNAL( alertReceived( const QString& ) ), this, SLOT( alertReceived( const QString& ) ) );
    connect( model, SIGNAL( connectionError( const QString& ) ), this, SLOT( connectionError( const QString& ) ) );
    connect( model, SIGNAL( authRequested( QAuthenticator* ) ), this, SLOT( authenticationRequested( QAuthenticator* ) ) );

    connect( model, SIGNAL( networkPolicyOffline() ), this, SLOT( updateActionsOffline() ) );
    connect( model, SIGNAL( networkPolicyExpensive() ), this, SLOT( updateActionsOnline() ) );
    connect( model, SIGNAL( networkPolicyOnline() ), this, SLOT( updateActionsOnline() ) );

    // Setup Model Proxies
    msgListProxy = new ReverseProxy(this);
    msgListProxy->setSourceModel(msgListModel);
    
    // Tie models to views
    msgListTree->setModel( msgListProxy );
    connect( msgListTree->selectionModel(), SIGNAL( selectionChanged( const QItemSelection&, const QItemSelection& ) ),
             this, SLOT( msgListSelectionChanged( const QItemSelection&, const QItemSelection& ) ) );

    // Hide some of the (excess) columns
    // enum { SUBJECT, SEEN, FROM, TO, DATE, SIZE, COLUMN_COUNT };
    msgListTree->setColumnHidden(Imap::Mailbox::MsgListModel::SEEN, true);
    msgListTree->setColumnHidden(Imap::Mailbox::MsgListModel::TO, true);
    msgListTree->setColumnHidden(Imap::Mailbox::MsgListModel::SIZE, true);
}

void MessageListWindow::msgListSelectionChanged( const QItemSelection& selected, const QItemSelection& deselected )
{
    Q_UNUSED( deselected );
    if ( selected.indexes().isEmpty() )
        return;

    QModelIndex index = selected.indexes().front();
}

void MessageListWindow::msgListClicked( const QModelIndex& index )
{
    // cleanup previous window
    if(messageWindow) {
        messageWindow->disconnect();
        messageWindow->deleteLater();
    }

    // create new window
    messageWindow = new MessageWindow(index, model, this);
    QTimer::singleShot(0, messageWindow, SLOT(show()));
}

void MessageListWindow::slotResizeMsgListColumns()
{
    for ( int i = 0; i < msgListTree->header()->count(); ++i )
        msgListTree->resizeColumnToContents( i );
}

void MessageListWindow::showContextMenuMsgListTree( const QPoint& position )
{
    QList<QAction*> actionList;
    QModelIndex index = msgListTree->indexAt( position );
    if ( index.isValid() ) {
        markAsRead->setChecked( msgListModel->data( index, Imap::Mailbox::MsgListModel::RoleIsMarkedAsRead ).toBool() );
        actionList.append( markAsRead );
        markAsDeleted->setChecked( msgListModel->data( index, Imap::Mailbox::MsgListModel::RoleIsMarkedAsDeleted ).toBool() );
        actionList.append( markAsDeleted );
    }
    if ( ! actionList.isEmpty() )
        QMenu::exec( actionList, msgListTree->mapToGlobal( position ) );
}

/** @short Request a check for new messages in selected mailbox */
void MessageListWindow::slotResyncMbox()
{
    if ( ! model->isNetworkAvailable() )
        return;

    // TODO: make this work...
    /*
    QModelIndexList indices = mboxTree->selectionModel()->selectedIndexes();
    for ( QModelIndexList::const_iterator it = indices.begin(); it != indices.end(); ++it ) {
        Q_ASSERT( it->isValid() );
        if ( it->column() != 0 )
            continue;
        Imap::Mailbox::TreeItemMailbox* mbox = dynamic_cast<Imap::Mailbox::TreeItemMailbox*>(
                    Imap::Mailbox::Model::realTreeItem( *it )
                );
        Q_ASSERT( mbox );
        model->resyncMailbox( mbox );
    }
    */
}

void MessageListWindow::handleMarkAsRead( bool value )
{
    QModelIndexList indices = msgListTree->selectionModel()->selectedIndexes();
    for ( QModelIndexList::const_iterator it = indices.begin(); it != indices.end(); ++it ) {
        Q_ASSERT( it->isValid() );
        Q_ASSERT( it->model() == msgListModel );
        if ( it->column() != 0 )
            continue;
        Imap::Mailbox::TreeItemMessage* message = dynamic_cast<Imap::Mailbox::TreeItemMessage*>(
                Imap::Mailbox::Model::realTreeItem( *it )
                );
        Q_ASSERT( message );
        model->markMessageRead( message, value );
    }
}

void MessageListWindow::handleMarkAsDeleted( bool value )
{
    QModelIndexList indices = msgListTree->selectionModel()->selectedIndexes();
    for ( QModelIndexList::const_iterator it = indices.begin(); it != indices.end(); ++it ) {
        Q_ASSERT( it->isValid() );
        Q_ASSERT( it->model() == msgListModel );
        if ( it->column() != 0 )
            continue;
        Imap::Mailbox::TreeItemMessage* message = dynamic_cast<Imap::Mailbox::TreeItemMessage*>(
                Imap::Mailbox::Model::realTreeItem( *it )
                );
        Q_ASSERT( message );
        model->markMessageDeleted( message, value );
    }
}

void MessageListWindow::slotExpunge()
{
    model->expungeMailbox( msgListModel->currentMailbox() );
}

void MessageListWindow::updateMessageFlags()
{
    markAsRead->setChecked( msgListModel->data( msgListTree->currentIndex(), Imap::Mailbox::MsgListModel::RoleIsMarkedAsRead ).toBool() );
    markAsDeleted->setChecked( msgListModel->data( msgListTree->currentIndex(), Imap::Mailbox::MsgListModel::RoleIsMarkedAsDeleted ).toBool() );
}

void MessageListWindow::updateActionsOnlineOffline( bool online )
{
    resyncMbox->setEnabled( online );
    expunge->setEnabled( online );
    markAsDeleted->setEnabled( online );
    markAsRead->setEnabled( online );
}


}


