/* Copyright (C) 2006 - 2010 Jan Kundrát <jkt@gentoo.org>
   Copyright (C) 2010 Benson Tsai <btsai@vrwarp.com>

   This file is part of the Trojita Qt IMAP e-mail client,
   http://trojita.flaska.net/

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or the version 3 of the License.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include <QApplication>
#include <QAuthenticator>
#include <QInputDialog>
#include <QMenuBar>
#include <QMessageBox>
#include <QStatusBar>
#include <QToolButton>
#include <QMaemo5InformationBox>

#include "BaseWindow.h"
#include "ComposeWidget.h"
#include "SettingsDialog.h"
#include "SettingsNames.h"
#include "Imap/Model/Model.h"

#include "ui_CreateMailboxDialog.h"

#include "iconloader/qticonloader.h"

/** @short All user-facing widgets and related classes */
namespace Gui {

BaseWindow::BaseWindow(Imap::Mailbox::Model* model, QWidget* parent):
    QMainWindow(parent), model(model), status()
{
    setWindowTitle( trUtf8("Trojitá") );
    setAttribute(Qt::WA_Maemo5StackedWindow);
    statusBar()->hide();
    
    QSettings s;
    if ( ! s.contains( SettingsNames::imapMethodKey ) ) {
        QTimer::singleShot( 0, this, SLOT(slotShowSettings()) );
    }
    
    init();
}
    
BaseWindow::BaseWindow(QWidget* parent):
    QMainWindow(parent), model(NULL), status()
{
    setWindowTitle( trUtf8("Trojitá") );
    setAttribute(Qt::WA_Maemo5StackedWindow);
    statusBar()->hide();
    
    QSettings s;
    if ( ! s.contains( SettingsNames::imapMethodKey ) ) {
        QTimer::singleShot( 0, this, SLOT(slotShowSettings()) );
    }
}
    
void BaseWindow::init() {
    setupModels();
    createActions();
    
    if(model->isNetworkAvailable()) {
        if(model->isNetworkOnline())
            networkPolicyOnline();
        else
            networkPolicyExpensive();
    }
    else
        networkPolicyOffline();
}

void BaseWindow::createActions()
{
    QActionGroup* netPolicyGroup = new QActionGroup( this );
    netPolicyGroup->setExclusive( true );
    netOffline = new QAction( QIcon( QLatin1String(":/icons/network-offline") ), tr("Offline"), netPolicyGroup );
    netOffline->setCheckable( true );
    // connect later
    netExpensive = new QAction( QIcon( QLatin1String(":/icons/network-expensive") ), tr("Expensive Connection"), netPolicyGroup );
    netExpensive->setCheckable( true );
    // connect later
    netOnline = new QAction( QIcon( QLatin1String(":/icons/network-online") ), tr("Free Access"), netPolicyGroup );
    netOnline->setCheckable( true );
    // connect later

    configSettings = new QAction( QtIconLoader::icon( QLatin1String("configure") ),  tr("Settings..."), this );
    connect( configSettings, SIGNAL( triggered() ), this, SLOT( slotShowSettings() ) );

    composeMail = new QAction( QtIconLoader::icon( QLatin1String("document-edit") ),  tr("Compose Mail..."), this );
    connect( composeMail, SIGNAL(triggered()), this, SLOT(slotComposeMail()) );

    aboutTrojita = new QAction( trUtf8("About Trojitá..."), this );
    connect( aboutTrojita, SIGNAL(triggered()), this, SLOT(slotShowAboutTrojita()) );

    connectModelActions();
}

void BaseWindow::connectModelActions()
{
    connect( netOffline, SIGNAL( triggered() ), model, SLOT( setNetworkOffline() ) );
    connect( netExpensive, SIGNAL( triggered() ), model, SLOT( setNetworkExpensive() ) );
    connect( netOnline, SIGNAL( triggered() ), model, SLOT( setNetworkOnline() ) );
}

void BaseWindow::setupModels()
{
    connect( model, SIGNAL( alertReceived( const QString& ) ), this, SLOT( alertReceived( const QString& ) ) );
    connect( model, SIGNAL( connectionError( const QString& ) ), this, SLOT( connectionError( const QString& ) ) );
    connect( model, SIGNAL( authRequested( QAuthenticator* ) ), this, SLOT( authenticationRequested( QAuthenticator* ) ) );

    connect( model, SIGNAL( networkPolicyOffline() ), this, SLOT( networkPolicyOffline() ) );
    connect( model, SIGNAL( networkPolicyExpensive() ), this, SLOT( networkPolicyExpensive() ) );
    connect( model, SIGNAL( networkPolicyOnline() ), this, SLOT( networkPolicyOnline() ) );

    connect( model, SIGNAL(connectionStateChanged(QObject*,Imap::ConnectionState)),
             this, SLOT(showConnectionStatus(QObject*,Imap::ConnectionState)) );

    connect( model, SIGNAL(activityHappening(bool)), this, SLOT(updateBusyParsers(bool)) );
}

void BaseWindow::alertReceived( const QString& message )
{
    QMessageBox::warning( this, tr("IMAP Alert"), message );
}

void BaseWindow::connectionError( const QString& message )
{
    if ( QSettings().contains( SettingsNames::imapMethodKey ) ) {
        QMessageBox::critical( this, tr("Connection Error"), message );
    } else {
        // hack: this slot is called even on the first run with no configuration
        // We shouldn't have to worry about that, since the dialog is already scheduled for calling
        // -> do nothing
    }
    netOffline->trigger();
}

void BaseWindow::networkPolicyOffline()
{
    netOffline->setChecked( true );
    netExpensive->setChecked( false );
    netOnline->setChecked( false );
}

void BaseWindow::networkPolicyExpensive()
{
    netOffline->setChecked( false );
    netExpensive->setChecked( true );
    netOnline->setChecked( false );
}

void BaseWindow::networkPolicyOnline()
{
    netOffline->setChecked( false );
    netExpensive->setChecked( false );
    netOnline->setChecked( true );
}

void BaseWindow::slotShowSettings()
{
    SettingsDialog* dialog = new SettingsDialog();
    if ( dialog->exec() == QDialog::Accepted ) {
        // FIXME: clear model if necessary
    }
}

void BaseWindow::authenticationRequested( QAuthenticator* auth )
{
    QSettings s;
    QString user = s.value( SettingsNames::imapUserKey ).toString();
    QString pass = s.value( SettingsNames::imapPassKey ).toString();
    if ( pass.isEmpty() ) {
        bool ok;
        pass = QInputDialog::getText( this, tr("Password"),
                                      tr("Please provide password for %1").arg( user ),
                                      QLineEdit::Password, QString::null, &ok );
        if ( ok ) {
            auth->setUser( user );
            auth->setPassword( pass );
        }
    } else {
        auth->setUser( user );
        auth->setPassword( pass );
    }
}

void BaseWindow::slotComposeMail()
{
    invokeComposeDialog();
}

void BaseWindow::invokeComposeDialog( const QString& subject, const QString& body,
                                      const QList<QPair<QString,QString> >& recipients )
{
    QSettings s;
    ComposeWidget* w = new ComposeWidget( NULL );
    w->setData( QString::fromAscii("%1 <%2>").arg(
                s.value( SettingsNames::realNameKey ).toString(),
                s.value( SettingsNames::addressKey ).toString() ),
        recipients, subject, body );
    w->setAttribute( Qt::WA_DeleteOnClose, true );
    w->show();
}

void BaseWindow::showConnectionStatus( QObject* parser, Imap::ConnectionState state )
{
    Q_UNUSED( parser );
    using namespace Imap;
    QString message;
    enum { SHORTDURATION = 1000, LONGDURATION = 365*24*60*60*1000 };
    bool transient = false;
    switch ( state ) {
    case CONN_STATE_NONE:
        return;
    case CONN_STATE_HOST_LOOKUP:
        message = tr("Resolving hostname...");
        break;
    case CONN_STATE_CONNECTING:
        message = tr("Connecting to the IMAP server...");
        break;
    case CONN_STATE_STARTTLS:
        message = tr("Negotiating encryption...");
        break;
    case CONN_STATE_ESTABLISHED:
        message = tr("Connection established.");
        transient = true;
        break;
    case CONN_STATE_LOGIN:
        message = tr("Logging in...");
        break;
    case CONN_STATE_LOGIN_FAILED:
        message = tr("Login failed.");
        break;
    case CONN_STATE_AUTHENTICATED:
        message = tr("Logged in.");
        transient = true;
        break;
    case CONN_STATE_SELECTING:
        message = tr("Opening mailbox...");
        break;
    case CONN_STATE_SYNCING:
        message = tr("Synchronizing mailbox...");
        break;
    case CONN_STATE_SELECTED:
        message = tr("Mailbox opened.");
        transient = true;
        break;
    case CONN_STATE_FETCHING_PART:
        message = tr("Downloading message...");
        break;
    case CONN_STATE_FETCHING_MSG_METADATA:
        message = tr("Downloading message structure...");
        break;
    case CONN_STATE_LOGOUT:
        message = tr("Logged out.");
        break;
    }

    QWidget* old = status.widget();
    status.setTimeout(transient?SHORTDURATION:LONGDURATION);
    status.setWidget(new QLabel(message, &status));
    status.show();
}

void BaseWindow::updateBusyParsers(bool busy)
{
    static bool wasBusy = false;
    // Check to see if the busy status changed
    if ( wasBusy ^ busy ) {
        wasBusy = busy;
        setAttribute(Qt::WA_Maemo5ShowProgressIndicator, busy);
    }
}

void BaseWindow::slotShowAboutTrojita()
{
    QMessageBox::about( this, trUtf8("About Trojitá"),
                        trUtf8("<p>This is <b>Trojitá</b>, a Qt IMAP e-mail client</p>"
                               "<p>Copyright &copy; 2007-2010 Jan Kundrát &lt;jkt@flaska.net&gt;</p>"
                               "<p>More information at http://trojita.flaska.net/</p>"
                               "<p>You are using version %1.</p>").arg(
                                       QApplication::applicationVersion() ) );
}

}


