# -*- coding: utf-8 -*-
import codecs
import os, time, re
from PyQt4.QtCore import QAbstractTableModel, QObject, QSize
from PyQt4.Qt import Qt
from languages.model import LanguageModel
from common import IconCache

CACHE_PO = os.path.expanduser("~/.cache/transifex-mobile")

class GenericFileModel:
    def title(self):
        return self.local_path
    def update_timestamp(self):
        self.last_access = int(time.time())

class RemoteFileModel(GenericFileModel):
    RE_REMOTE_FILE = r"""^https?://(?P<server>[-\w\.]+)/projects/p/(?P<project_slug>[-\w]+)/c/(?P<component_slug>[-\w]+)/raw/(?P<filename>.*)$"""

    @classmethod
    def getByURL(cls, url):
        m = re.match(cls.RE_REMOTE_FILE, url)
        if m:
            server, project_slug, component_slug, file_path = m.groups()
            from servers.model import ServerModel
            instance = ServerModel(server)
            return cls(
                project_slug,
                component_slug,
                file_path,
                client=instance)
        return None

    def __init__(self, project_slug, component_slug, file_path,
        client=None, last_access=None, locked=False, lang_code=None,
        translated=0, fuzzy=0, untranslated=0):
        self.lang_code = lang_code
        self.project_slug = project_slug
        self.component_slug = component_slug
        self.remote_path = file_path
        self.client = client
        self.pulled = False
        self.pushed = False
        self.last_access = last_access or int(time.time())
        self.local_path = "%s/%s;;%s;;%s" % \
            (CACHE_PO, project_slug, component_slug, file_path.replace("/", ";"))
        self.locked = locked
        self.translated = translated
        self.fuzzy = fuzzy
        self.untranslated = untranslated
        self.title = "%s (%s)" % (LanguageModel.getNameByCode(self.lang_code), self.remote_path)
        self.long_title = "%s from %s/%s" % (self.title, self.project_slug, self.component_slug)
        
    def progress(self):
        try:
            return self.translated*100/(self.translated+self.fuzzy+self.untranslated)
        except ZeroDivisionError:
            return 0

    def pull(self):
        if not self.pulled:
            try:
                os.makedirs(os.path.dirname(self.local_path))
            except:
                pass
            data = self.client.pullFile(
                self.project_slug,
                self.component_slug,
                self.remote_path)
            fh = open(self.local_path, "w")
            fh.write(data)
            fh.close()

            self.update_timestamp()
            self.pulled = True
            self.pushed = False

    def push(self):
        if not self.pushed:
            self.update_timestamp()
            self.client.pushFile(
                self.project_slug,
                self.component_slug,
                self.remote_path,
                self.local_path)
            self.pushed = True
            self.pulled = False

    def lock(self):
        self.client.lockFile(
            self.project_slug,
            self.component_slug,
            self.remote_path)
        self.client.locksInvalidated.emit()

    def unlock(self):
        self.client.unlockFile(
            self.project_slug,
            self.component_slug,
            self.remote_path)
        self.client.locksInvalidated.emit()

    def uri(self):
        # TODO: Fix this, this isn't nice at all
        return "http://www.transifex.net/%s/%s/%s" % \
            (self.project_slug, self.component_slug, self.remote_path)

class LocalFileModel(GenericFileModel):
    def __init__(self, path):
        self.local_path = path
        self.locked = False
        self.lang_code = None
    def pull(self):
        pass
    def push(self):
        pass
    def lock(self):
        pass
    def unlock(self):
        pass
    def uri(self):
        return "file://%s" % self.local_path

class FileListModel(QAbstractTableModel):
    HEADER = ["Language", "Translations", "Filename"]
    def __init__(self, project=None, component=None, parent=None):
        QAbstractTableModel.__init__(self, parent=parent)
        self.project = project
        self.component = component
        self.objs = project.client.pullFiles(
            project.slug, component.slug)
        self.size = QSize(250, 32)

    def rowCount(self, parent):
        return len(self.objs)
        
    def columnCount(self, parent):
        return 3
        
    def headerData(self, section, orientation, role=Qt.DisplayRole):
        if orientation == Qt.Horizontal:
            if role == Qt.DisplayRole:
                try:
                    return self.HEADER[section]
                except IndexError:
                    pass
        return None

    def data(self, index, role=Qt.UserRole):
        if not index.isValid():
            return None
        row, column = index.row(), index.column()
        if row < 0 or row >= len(self.objs):
            return None
            
        if role == Qt.SizeHintRole:
            if column in [0,1,2]:
                return self.size
        elif role == Qt.DecorationRole and column == 0:
            if self.objs[row].locked:
                return IconCache.getLocked()
            else:
                return IconCache.getUnlocked()
        elif role == Qt.DisplayRole:
            if column == 0:
                return LanguageModel.getNameByCode(self.objs[row].lang_code)
            elif column == 1:
                return "%i%% (%i/%i/%i)" % (self.objs[row].progress(),
                    self.objs[row].translated, self.objs[row].fuzzy,
                    self.objs[row].untranslated)
            elif column == 2:
                return self.objs[row].remote_path
        elif role == Qt.UserRole:
            return self.objs[row]
        return None
