#include "songlistwindow.h"
#include "songinfowindow.h"
#include "songlistdelegate.h"
#include "nowplayingwindow.h"
#include "listview.h"
#include "nowplayingbutton.h"
#include "../shared/songmodel.h"
#include "../mediaplayer/mobility/mediaplayer.h"
#include <QAction>
#include <QLabel>
#include <QMenuBar>
#include <QMaemo5Style>

SongListWindow::SongListWindow(QWidget *parent) :
    StackedWindow(parent),
    m_view(new ListView(this)),
    m_model(new SongModel(this)),
    m_contextMenu(new QMenu(this)),
    m_addToQueueContextAction(m_contextMenu->addAction(tr("Add to playback queue"), this, SLOT(queueSong()))),
    m_noResultsLabel(new QLabel(QString("<p style='color:%1;font-size:40px;'><center><b>%2</b></center></p>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(tr("No songs found")), this))
{
    this->setWindowTitle(tr("Songs"));
    this->setCentralWidget(m_view);

    SongListDelegate *delegate = new SongListDelegate(m_view);

    m_view->setModel(m_model);
    m_view->setItemDelegate(delegate);
    m_view->setMonitorScrolling(true);
    m_noResultsLabel->hide();

    this->connect(m_model, SIGNAL(loadingChanged(bool)), this, SLOT(onLoadingChanged(bool)));
    this->connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    this->connect(delegate, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(delegate, SIGNAL(play(QModelIndex)), this, SLOT(playSong(QModelIndex)));
    this->connect(m_view, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    this->connect(m_view, SIGNAL(scrollingStarted()), NowPlayingButton::instance(), SLOT(hide()));
    this->connect(m_view, SIGNAL(scrollingStopped()), NowPlayingButton::instance(), SLOT(poke()));

    this->onOrientationChanged();
}

void SongListWindow::showEvent(QShowEvent *event) {
    this->onLoadingChanged(m_model->loading());
    StackedWindow::showEvent(event);
}

void SongListWindow::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void SongListWindow::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 700);
}

void SongListWindow::getSongs() {
    m_model->getSongs();
}

void SongListWindow::onLoadingChanged(bool loading) {
    this->toggleBusy(loading);
    m_noResultsLabel->setVisible((!loading) && (m_model->rowCount() == 0));
}

void SongListWindow::onCountChanged(int count) {
    m_noResultsLabel->setVisible((count == 0) && (!m_model->loading()));
}

void SongListWindow::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_addToQueueContextAction);
}

void SongListWindow::queueSong() {
    MediaPlayer::instance()->addSong(m_model->get(m_view->currentIndex()));
}

void SongListWindow::onItemClicked(const QModelIndex &index) {
    SongInfoWindow *window = new SongInfoWindow(m_model->get(index), this);
    window->show();
}

void SongListWindow::playSong(const QModelIndex &index) {
    MediaPlayer::instance()->play(m_model->get(index));
    NowPlayingWindow *window = new NowPlayingWindow(this);
    window->show();
}
