#include "songmodel.h"
#include "music.h"
#include "notifications.h"
#include "definitions.h"
#include "artworkcache.h"

SongModel::SongModel(QObject *parent) :
    QAbstractListModel(parent),
    m_cache(new ArtworkCache(this)),
    #ifndef QML_USER_INTERFACE
    m_artworkPressedRow(-1),
    #endif
    m_loading(false)
{
    m_roleNames[SongRoles::IdRole] = "id";
    m_roleNames[SongRoles::TitleRole] = "title";
    m_roleNames[SongRoles::ArtistRole] = "artist";
    m_roleNames[SongRoles::ArtistIdRole] = "artistId";
    m_roleNames[SongRoles::AlbumTitleRole] = "albumTitle";
    m_roleNames[SongRoles::AlbumArtistRole] = "albumArtist";
    m_roleNames[SongRoles::AlbumIdRole] = "albumId";
    m_roleNames[SongRoles::GenreRole] = "genre";
    m_roleNames[SongRoles::FilePathRole] = "filePath";
    m_roleNames[SongRoles::FileSuffixRole] = "fileSuffix";
    m_roleNames[SongRoles::MimeTypeRole] = "mimeType";
    m_roleNames[SongRoles::UrlRole] = "url";
    m_roleNames[SongRoles::ArtworkUrlRole] = "artworkUrl";
    m_roleNames[SongRoles::ArtworkRole] = "artwork";
    m_roleNames[SongRoles::StreamUrlRole] = "streamUrl";
    m_roleNames[SongRoles::YearRole] = "year";
    m_roleNames[SongRoles::DiscNumberRole] = "discNumber";
    m_roleNames[SongRoles::BitRateRole] = "bitRate";
    m_roleNames[SongRoles::DurationRole] = "duration";
    m_roleNames[SongRoles::SizeRole] = "size";
#ifndef QML_USER_INTERFACE
    m_roleNames[SongRoles::ArtworkPressedRole] = "artworkPressed";
#endif
#if QT_VERSION < 0x050000
    this->setRoleNames(m_roleNames);
#endif
    this->connect(m_cache, SIGNAL(artworkReady()), this, SLOT(onArtworkReady()));
}

SongModel::~SongModel() {
    qDeleteAll(m_list);
    m_list.clear();
}

#if QT_VERSION >= 0x050000
QHash<int, QByteArray> SongModel::roleNames() const {
    return m_roleNames;
}
#endif

int SongModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return m_list.size();
}

QVariant SongModel::data(const QModelIndex &index, int role) const {
    if (Song *song = this->get(index)) {
        switch (role) {
        case SongRoles::IdRole:
            return song->id();
        case SongRoles::TitleRole:
            return song->title();
        case SongRoles::ArtistRole:
            return song->artist();
        case SongRoles::ArtistIdRole:
            return song->artistId();
        case SongRoles::AlbumTitleRole:
            return song->albumTitle();
        case SongRoles::AlbumArtistRole:
            return song->albumArtist();
        case SongRoles::AlbumIdRole:
            return song->albumId();
        case SongRoles::GenreRole:
            return song->genre();
        case SongRoles::FilePathRole:
            return song->filePath();
        case SongRoles::FileSuffixRole:
            return song->fileSuffix();
        case SongRoles::MimeTypeRole:
            return song->mimeType();
        case SongRoles::UrlRole:
            return song->url();
        case SongRoles::ArtworkUrlRole:
            return song->artworkUrl();
        case SongRoles::ArtworkRole:
            return m_cache->artwork(song->artworkUrl(), THUMBNAIL_SIZE);
        case SongRoles::StreamUrlRole:
            return song->streamUrl();
        case SongRoles::YearRole:
            return song->year();
        case SongRoles::DiscNumberRole:
            return song->discNumber();
        case SongRoles::DurationRole:
            return song->duration();
        case SongRoles::SizeRole:
            return song->size();
#ifndef QML_USER_INTERFACE
        case SongRoles::ArtworkPressedRole:
            return QVariant(m_artworkPressedRow == index.row());
#endif
        default:
            break;
        }
    }

    return QVariant();
}

QVariant SongModel::data(int row, int role) const {
    return this->data(this->index(row), role);
}

bool SongModel::setData(const QModelIndex &index, const QVariant &value, int role) {
    Q_UNUSED(index)

#ifndef QML_USER_INTERFACE
    switch (role) {
    case SongRoles::ArtworkPressedRole:
        m_artworkPressedRow = value.toInt();
        return true;
    default:
        break;
    }
#endif
    return false;
}

bool SongModel::setData(int row, const QVariant &value, int role) {
    return this->setData(this->index(row), value, role);
}

QMap<int, QVariant> SongModel::itemData(const QModelIndex &index) const {
    QMap<int, QVariant> map;

    for (int role = SongRoles::IdRole; role <= SongRoles::SizeRole; role++) {
        map[role] = this->data(index, role);
    }

    return map;
}

QMap<int, QVariant> SongModel::itemData(int row) const {
    return this->itemData(this->index(row));
}

Song* SongModel::get(const QModelIndex &index) const {
    if ((index.row() >= 0) && (index.row() < m_list.size())) {
        return m_list.at(index.row());
    }

    return 0;
}

Song* SongModel::get(int row) const {
    return this->get(this->index(row));
}

bool SongModel::loading() const {
    return m_loading;
}

void SongModel::setLoading(bool loading) {
    if (loading != this->loading()) {
        m_loading = loading;
        emit loadingChanged(loading);
    }
}

QList<Song*> SongModel::songs() const {
    return m_list;
}

QString SongModel::albumId() const {
    return m_albumId;
}

void SongModel::setAlbumId(const QString &id) {
    m_albumId = id;
}

QString SongModel::playlistId() const {
    return m_playlistId;
}

void SongModel::setPlaylistId(const QString &id) {
    m_playlistId = id;
}

void SongModel::appendSong(Song *song) {
    this->beginInsertRows(QModelIndex(), m_list.size(), m_list.size());
    m_list.append(song);
    this->endInsertRows();

    emit countChanged(this->rowCount());
}

void SongModel::appendSongs(QList<Song *> songs) {
    this->beginInsertRows(QModelIndex(), m_list.size(), m_list.size() + songs.size() - 1);
    m_list.append(songs);
    this->endInsertRows();

    emit countChanged(this->rowCount());
}

void SongModel::insertSong(int i, Song *song) {
    if ((i >= 0) && (i < m_list.size())) {
        this->beginInsertRows(QModelIndex(), i, i);
        m_list.insert(i, song);
        this->endInsertRows();
    }
    else {
        this->appendSong(song);
    }

    emit countChanged(this->rowCount());
}

void SongModel::insertSongs(int i, QList<Song *> songs) {
    if ((i >= 0) && (i < m_list.size())) {
        this->beginInsertRows(QModelIndex(), i, i + songs.size() - 1);

        foreach (Song *song, songs) {
            m_list.insert(i, song);
            i++;
        }

        this->endInsertRows();
    }
    else {
        this->appendSongs(songs);
    }

    emit countChanged(this->rowCount());
}

void SongModel::removeSong(int i) {
    if ((i >= 0) && (i < m_list.size())) {
        this->beginRemoveRows(QModelIndex(), i, i);
        m_list.takeAt(i)->deleteLater();
        this->endRemoveRows();
    }

    emit countChanged(this->rowCount());
}

void SongModel::removeSongs(int i, int count) {
    if ((i >= 0) && ((i + count) <= m_list.size())) {
        this->beginRemoveRows(QModelIndex(), i, i + count - 1);

        for (int ii = i; ii < (i + count); ii++) {
            m_list.takeAt(i)->deleteLater();
        }

        this->endRemoveRows();
    }

    emit countChanged(this->rowCount());
}

void SongModel::reload() {
    this->clear();

    if (!this->albumId().isEmpty()) {
        this->getAlbumSongs(this->albumId());
    }
    else if (!this->playlistId().isEmpty()) {
        this->getPlaylistSongs(this->playlistId());
    }
    else {
        this->getSongs();
    }
}

void SongModel::clear() {
    this->removeSongs(0, m_list.size());
}

void SongModel::getSongs() {
    this->setLoading(true);
    this->setAlbumId(QString());
    this->setPlaylistId(QString());

    SongList *list = Music::getSongs();
    this->connect(list, SIGNAL(ready(SongList*)), this, SLOT(addSongs(SongList*)));
}

void SongModel::getAlbumSongs(const QString &albumId) {
    this->setLoading(true);
    this->setAlbumId(albumId);

    SongList *list = Music::getAlbumSongs(albumId);
    this->connect(list, SIGNAL(ready(SongList*)), this, SLOT(addSongs(SongList*)));
}

void SongModel::getPlaylistSongs(const QString &playlistId) {
    this->setLoading(true);
    this->setPlaylistId(playlistId);

    SongList *list = Music::getPlaylistSongs(playlistId);
    this->connect(list, SIGNAL(ready(SongList*)), this, SLOT(addSongs(SongList*)));
}

void SongModel::addSongs(SongList *list) {
    switch (list->error()) {
    case ReplyError::NoError:
        if (list->count() > 0) {
            this->appendSongs(list->songs());
        }

        break;
    default:
        Notifications::showError(list->errorString());
        break;
    }

    this->setLoading(false);

    list->deleteLater();
}

void SongModel::onArtworkReady() {
    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
}
