#ifndef MEDIAPLAYER_H
#define MEDIAPLAYER_H

#include "song.h"
#include "musicstream.h"
#include <QMediaPlayer>

using namespace QtUbuntuOne;

class PlaybackQueueModel;

class Media : public QObject
{
    Q_OBJECT

    Q_ENUMS(PlaybackMode MediaStatus State Error)

public:
    enum PlaybackMode {
        Sequential = 0,
        RepeatAll,
        RepeatOne
    };

    enum MediaStatus {
        Unknown = 0,
        NoMedia,
        LoadingMedia,
        LoadedMedia,
        StalledMedia,
        BufferingMedia,
        BufferedMedia,
        EndOfMedia,
        InvalidMedia
    };

    enum State {
        StoppedState = 0,
        PlayingState,
        PausedState
    };

    enum Error {
        NoError = 0,
        ResourceError,
        FormatError,
        NetworkError,
        AccessDeniedError,
        ServiceMissingError
    };
};

class MediaPlayer : public QObject
{
    Q_OBJECT

    Q_PROPERTY(int currentIndex
               READ currentIndex
               NOTIFY currentIndexChanged)
    Q_PROPERTY(int count
               READ count
               NOTIFY countChanged)
    Q_PROPERTY(Song* currentSong
               READ currentSong
               NOTIFY currentSongChanged)
    Q_PROPERTY(qint64 streamPosition
               READ streamPosition
               NOTIFY streamPositionChanged)
    Q_PROPERTY(qint64 streamSize
               READ streamSize
               NOTIFY streamSizeChanged)
    Q_PROPERTY(qint64 position
               READ position
               WRITE setPosition
               NOTIFY positionChanged)
    Q_PROPERTY(qint64 duration
               READ duration
               NOTIFY durationChanged)
    Q_PROPERTY(int volume
               READ volume
               WRITE setVolume
               NOTIFY volumeChanged)
    Q_PROPERTY(int bufferStatus
               READ bufferStatus
               NOTIFY bufferStatusChanged)
    Q_PROPERTY(Media::PlaybackMode playbackMode
               READ playbackMode
               WRITE setPlaybackMode
               NOTIFY playbackModeChanged)
    Q_PROPERTY(Media::MediaStatus mediaStatus
               READ mediaStatus
               NOTIFY mediaStatusChanged)
    Q_PROPERTY(Media::State state
               READ state
               NOTIFY stateChanged)
    Q_PROPERTY(QString errorString
               READ errorString
               NOTIFY error)

public:
    ~MediaPlayer();

    static MediaPlayer* instance();

    PlaybackQueueModel* queue() const;

    int currentIndex() const;

    int count() const;

    Song* currentSong() const;

    qint64 streamPosition() const;

    qint64 streamSize() const;

    qint64 position() const;

    qint64 duration() const;

    int volume() const;

    int bufferStatus() const;

    Media::PlaybackMode playbackMode() const;

    Media::MediaStatus mediaStatus() const;

    Media::State state() const;

    Media::Error error() const;

    QString errorString() const;

public slots:
    void setPosition(qint64 position);

    void setVolume(int volume);

    void setPlaybackMode(Media::PlaybackMode mode);
    void togglePlaybackMode();

    void play(Song *song);
    void play(const QList<Song*> &songs);
    void play(int index);
    void play();
    void pause();
    void togglePlayPause();
    void stop();

    void next();
    void previous();

    void addSong(Song *song);
    void addSongs(const QList<Song*> &songs);
    void removeSong(int index);
    void clearSongs();

private:
    MediaPlayer();

    void setCurrentIndex(int index);
    void setCurrentSong(Song *song);
    void play_p(Song *song);

private slots:
    void onQueueCountChanged(int count);
    void onStreamStatusChanged(StreamStatus::Status status);
    void onMediaStatusChanged();
    void onStateChanged();
    void onError();
    void onCacheMusicStreamsChanged(bool cache);

signals:
    void countChanged(int count);
    void currentIndexChanged(int index);
    void currentSongChanged(Song *song);
    void streamPositionChanged(qint64 position);
    void streamSizeChanged(qint64 size);
    void positionChanged(qint64 position);
    void durationChanged(qint64 duration);
    void volumeChanged(int volume);
    void bufferStatusChanged(int percent);
    void playbackModeChanged(Media::PlaybackMode mode);
    void mediaStatusChanged(Media::MediaStatus status);
    void stateChanged(Media::State state);
    void error(Media::Error error);

private:
    static MediaPlayer *m_instance;

    QMediaPlayer *m_player;

    PlaybackQueueModel *m_queue;

    Song *m_song;

    MusicStream *m_stream;

    int m_index;

    Media::PlaybackMode m_playbackMode;

    QMap<QUrl, MusicStream*> m_cache;
};

#endif // MEDIAPLAYER_H
