#include "nowplayinginfotab.h"
#include "separatorlabel.h"
#include "artworklabel.h"
#include "scrollarea.h"
#include "screen.h"
#include "../shared/utils.h"
#include "../shared/artworkcache.h"
#include "../mediaplayer/mobility/mediaplayer.h"
#include <QGridLayout>
#include <QLabel>

NowPlayingInfoTab::NowPlayingInfoTab(QWidget *parent) :
    AbstractTab(parent),
    m_titleLabel(new QLabel(this)),
    m_artistLabel(new QLabel(this)),
    m_dateLabel(new QLabel(this)),
    m_genreLabel(new QLabel(this)),
    m_formatLabel(new QLabel(this)),
    m_sizeLabel(new QLabel(this)),
    m_lengthLabel(new QLabel(this)),
    m_cache(new ArtworkCache(this)),
    m_artworkLabel(new ArtworkLabel(this)),
    m_propertiesSeparator(new SeparatorLabel(tr("Properties"), this)),
    m_scrollArea(new ScrollArea(this))
{
    m_artworkLabel->setFixedSize(124, 124);
    m_artworkLabel->setInteractive(false);
    m_titleLabel->setWordWrap(true);

    QVBoxLayout *vbox = new QVBoxLayout(this);
    vbox->setContentsMargins(0, 0, 0, 0);
    vbox->addWidget(m_scrollArea);

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QGridLayout *grid = new QGridLayout(scrollWidget);

    grid->setContentsMargins(0, 0, 0, 0);
    grid->addItem(new QSpacerItem(10, 10, QSizePolicy::Maximum), 0, 0);
    grid->addWidget(m_titleLabel, 0, 1, 1, 2, Qt::AlignTop);
    grid->addWidget(m_artworkLabel, 1, 1, 2, 1, Qt::AlignTop);
    grid->addWidget(m_artistLabel, 1, 2, Qt::AlignTop);
    grid->addWidget(m_dateLabel, 2, 2, Qt::AlignTop);
    grid->addWidget(m_propertiesSeparator, 3, 1, 1, 2);
    grid->addWidget(m_genreLabel, 4, 1, 1, 2);
    grid->addWidget(m_formatLabel, 5, 1, 1, 2);
    grid->addWidget(m_sizeLabel, 6, 1, 1, 2);
    grid->addWidget(m_lengthLabel, 7, 1, 1, 2);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    this->connect(m_cache, SIGNAL(artworkReady()), this, SLOT(setArtwork()));
    this->connect(Screen::instance(), SIGNAL(screenLockStateChanged(bool)), this, SLOT(onScreenLockStateChanged(bool)));

    this->onOrientationChanged();
}

void NowPlayingInfoTab::showEvent(QShowEvent *event) {
    this->connectPlaybackSignals();
    event->accept();
}

void NowPlayingInfoTab::hideEvent(QHideEvent *event) {
    this->disconnectPlaybackSignals();
    event->accept();
}

void NowPlayingInfoTab::setLandscapeLayout() {
    m_scrollArea->widget()->setMaximumWidth(800);
}

void NowPlayingInfoTab::setPortraitLayout() {
    m_scrollArea->widget()->setMaximumWidth(480);
}

void NowPlayingInfoTab::connectPlaybackSignals() {
    this->connect(MediaPlayer::instance(), SIGNAL(currentSongChanged(Song*)), this, SLOT(onCurrentSongChanged(Song*)));
    this->onCurrentSongChanged(MediaPlayer::instance()->currentSong());
}

void NowPlayingInfoTab::disconnectPlaybackSignals() {
    this->disconnect(MediaPlayer::instance(), SIGNAL(currentSongChanged(Song*)), this, SLOT(onCurrentSongChanged(Song*)));
}

void NowPlayingInfoTab::setArtwork() {
    if (!MediaPlayer::instance()->currentSong()) {
        return;
    }

    QImage artwork = m_cache->artwork(MediaPlayer::instance()->currentSong()->artworkUrl(), QSize(124, 124));

    if (artwork.isNull()) {
        artwork = QImage("/usr/share/icons/hicolor/124x124/hildon/mediaplayer_default_album.png");
    }

    m_artworkLabel->setImage(artwork);
}

void NowPlayingInfoTab::onCurrentSongChanged(Song *song) {
    if (song) {
        m_titleLabel->setText(QString("<p style='font-size:32px;'>%1</p>").arg(song->title()));
        m_artistLabel->setText(QString("%1: %2").arg(tr("Artist")).arg(song->artist()));
        m_dateLabel->setText(QString("%1: %2").arg(tr("Year")).arg(song->year()));
        m_genreLabel->setText(QString("%1: %2").arg(tr("Genre")).arg(song->genre().isEmpty() ? tr("Unknown") : song->genre()));
        m_formatLabel->setText(QString("%1: %2").arg(tr("Format")).arg(song->fileSuffix()));
        m_sizeLabel->setText(QString("%1: %2").arg(tr("Size")).arg(Utils::fileSizeFromBytes(song->size())));
        m_lengthLabel->setText(QString("%1: %2").arg(tr("Length")).arg(Utils::durationFromSecs(song->duration())));

        this->setArtwork();
    }
}

void NowPlayingInfoTab::onScreenLockStateChanged(bool locked) {
    if (locked) {
        this->disconnectPlaybackSignals();
    }
    else if (this->isVisible()) {
        this->connectPlaybackSignals();
    }
}
