/*
  TimerJinni - A general timing application
  Copyright (C) 2009  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "Countdown.h"

#include <Lum/Base/L10N.h>
#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Dlg/TimeSpanChooser.h>

#include <Lum/ButtonRow.h>
#include <Lum/Button.h>
#include <Lum/Panel.h>
#include <Lum/Toggle.h>

#include <Lum/OS/Manager/Repository.h>

Countdown::Countdown()
 : startAction(new Lum::Model::Action()),
   paused(new Lum::Model::Boolean(false)),
   stopAction(new Lum::Model::Action()),
   timer(new Lum::Model::Action()),
   seconds(new Lum::Model::SizeT()),
   initialSeconds(4*60),
   state(stateUninitialized),
   clock(new StopWatch()),
   beepSound(NULL)
{
  Lum::Base::Path soundPath;

  soundPath.SetNativeDir(Lum::Base::Path::GetCWD());
  soundPath.AppendDir(L"data");
  soundPath.AppendDir(L"sounds");
  soundPath.SetBaseName(L"alarm.wav");

#if defined(APP_DATADIR)
  if (!soundPath.Exists()) {
    soundPath.SetNativeDir(Lum::Base::StringToWString(APP_DATADIR));
    soundPath.AppendDir(L"sounds");
    soundPath.SetBaseName(L"alarm.wav");
  }
#endif

  if (soundPath.Exists()) {
    beepSound=Lum::Audio::GetPlaySound();
    if (beepSound!=NULL) {
      beepSound->SetFilename(soundPath.GetPath(),Lum::Audio::PlaySound::typeWav);
    }
  }

  seconds->Set(initialSeconds);
  Observe(seconds);

  Observe(clock->GetClickAction());

  Observe(startAction);
  Observe(paused);
  Observe(stopAction);
  Observe(timer);
}

Countdown::~Countdown()
{
  delete beepSound;
}

void Countdown::SetValue()
{
  if (clock!=NULL) {
    clock->SetText(Lum::Base::TimeSpanToWString(seconds->Get()));
  }
}

void Countdown::PlayAlert()
{
  if (beepSound!=NULL) {
    beepSound->Play();
  }
  else {
    Lum::OS::display->Beep();
  }
}

void Countdown::AllowScreenBlanking(bool allow)
{
  Lum::OS::Manager::DisplayManager *displayManager=Lum::OS::Manager::repository->GetDisplayManager();

  if (displayManager!=NULL) {
    displayManager->AllowScreenBlanking(allow);
  }
}

void Countdown::SetState(State state)
{
  if (this->state==state) {
    return;
  }

  this->state=state;

  // Visual representation

  switch (state) {
  case stateUninitialized:
    assert(false);
    break;
  case stateRunning:
    clock->GetClickAction()->Disable();
    clock->SetAlarm(false);
    startAction->Enable();
    paused->Enable();
    stopAction->Enable();
    AllowScreenBlanking(false);
    break;
  case statePaused:
    clock->GetClickAction()->Disable();
    clock->SetAlarm(false);
    startAction->Enable();
    paused->Enable();
    stopAction->Enable();
    AllowScreenBlanking(false);
    break;
  case stateStoped:
    clock->GetClickAction()->Enable();
    clock->SetAlarm(false);
    startAction->Enable();
    paused->Disable();
    stopAction->Disable();
    AllowScreenBlanking(true);
    break;
  case stateAlerting:
    clock->GetClickAction()->Disable();
    clock->SetAlarm(true);
    startAction->Disable();
    paused->Disable();
    stopAction->Enable();
    AllowScreenBlanking(false);
    break;
  }

  // Timer

  switch (state) {
  case stateUninitialized:
    assert(false);
    break;
  case stateRunning:
    paused->Set(false);
    if (time.IsPausing()) {
      time.Resume();
    }
    else {
      time.Start();
    }
    seconds->Set(initialSeconds-time.GetTime());
    alertTime.SetToNow();
    alertTime.Add(seconds->Get());
    Lum::OS::display->AddTimer(0,100000,timer);
    break;
  case statePaused:
    seconds->Set(initialSeconds-time.GetTime());
    time.Pause();
    Lum::OS::display->RemoveTimer(timer);
    break;
  case stateStoped:
    if (time.IsStarted()) {
      seconds->Set(initialSeconds-time.GetTime());
    }
    time.Reset();
    paused->Set(false);
    Lum::OS::display->RemoveTimer(timer);
    break;
  case stateAlerting:
    seconds->Set(initialSeconds-time.GetTime());
    time.Reset();
    paused->Set(false);
    PlayAlert();
    alertTime.SetToNow();
    alertTime.Add(3);
    Lum::OS::display->RemoveTimer(timer);
    Lum::OS::display->AddTimer(0,100000,timer);
    break;
  }
}

void Countdown::CalcSize()
{
  Lum::Object *countdown;

  clock->SetText(L"4:00");
  clock->SetFlex(true,true);

  countdown=Lum::VPanel::Create(true,true)
            ->Add(clock)
            ->AddSpace()
            ->Add(Lum::ButtonRow::Create(true,false)
                  ->Add(Lum::Button::Create(_(L"BUTTON_START",L"_Start"),startAction,true,true))
                  ->Add(Lum::Toggle::Create(_(L"BUTTON_PAUSE",L"Pause"),paused,true,true))
                  ->Add(Lum::Button::Create(_(L"BUTTON_STOP",L"Sto_p"),stopAction,true,true)));

  container=countdown;

  SetState(stateStoped);

  Component::CalcSize();
}

void Countdown::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==clock->GetClickAction() &&
      clock->GetClickAction()->IsFinished() &&
      state==stateStoped) {
    Lum::Dlg::TimeSpanChooser *chooser;

    chooser=new Lum::Dlg::TimeSpanChooser(seconds);
    chooser->SetParent(GetWindow());
    if (chooser->Open()) {
      chooser->EventLoop();
      chooser->Close();
    }

    if (chooser->Success()) {
      initialSeconds=seconds->Get();
    }

    delete chooser;
  }
  else if (model==startAction &&
           startAction->IsFinished()) {
    SetState(stateStoped);
    SetState(stateRunning);
  }
  else if (model==paused) {
    if (state==stateRunning &&
        paused->Get()) {
      SetState(statePaused);
    }
    else if (state==statePaused &&
             !paused->Get()) {
      SetState(stateRunning);
    }
  }
  else if (model==stopAction &&
           stopAction->IsFinished()) {
    SetState(stateStoped);
  }
  else if (model==timer && timer->IsFinished()) {
    if (state==stateRunning) {
      Lum::Base::SystemTime now;

      seconds->Set(initialSeconds-time.GetTime());

      if (now>=alertTime) {
        SetState(stateAlerting);
      }
      else {
        Lum::OS::display->AddTimer(0,100000,timer);
      }
    }
    else if (state==stateAlerting) {
      Lum::Base::SystemTime now;

      if (now.GetTime()>alertTime) {
        PlayAlert();
        alertTime.SetToNow();
        alertTime.Add(3);
      }

      Lum::OS::display->AddTimer(0,100000,timer);
    }
  }
  else if (model==seconds) {
    SetValue();
  }

  Lum::Component::Resync(model,msg);
}
