/*
    A Qt game based on Tic Tac Toe for Maemo
    Copyright (C) 2010  Victor Blazquez Francisco <victor.blazquez@gmail.com>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "graphicsscene.h"
#include "shape.h"
#include "machineia.h"
#include "finalline.h"

#include <QGraphicsItem>
#include <QGraphicsSceneMouseEvent>
#include <QPropertyAnimation>
#include <QMessageBox>
#include <QTimer>
#include <QObject>
#include <QDebug>
#include <QEvent>

GraphicsScene::GraphicsScene(QObject* parent): QGraphicsScene(parent)
{
	QGraphicsPixmapItem* m_background = addPixmap(QPixmap("/usr/share/threetictac/bg.png", 0));
	m_background->setZValue(-3);
	m_background->setPos(-350, -180);
	m_background->setData(Shapes , Img);
	
	QGraphicsPixmapItem* m_gridImage = addPixmap(QPixmap("/usr/share/threetictac/lines.png", 0));
	m_gridImage->setZValue(-2);
	m_gridImage->setPos(-103,-103);
	m_gridImage->setData(Shapes , Img);
	
	m_textLevel = new QGraphicsTextItem;
	m_textLevel->setPos(-300,200);
	QFont serifFont("Times", 18, QFont::Bold);
	m_textLevel->setFont(serifFont);
	m_textLevel->setDefaultTextColor(Qt::white);
	m_textLevel->setData(Shapes, "textLevel");
	addItem(m_textLevel);
	updateLevelText();
	
	m_winner = new QGraphicsTextItem;
	m_winner->setPos(-50, -150);
	serifFont.setPointSize(24);
	m_winner->setFont(serifFont);
	m_winner->setDefaultTextColor(Qt::white);
	addItem(m_winner);
	
	m_restart = new QGraphicsTextItem;
	m_restart->setPos(250, 180);
	serifFont.setPointSize(24);
	m_restart->setFont(serifFont);
	m_restart->setDefaultTextColor(Qt::green);
	m_restart->setData(Shapes, "restartGame");
	addItem(m_restart);
	
	
	m_grid.resize(All);
	int *p = matrix[0];
	for (int i=0 ; i< All ; ++i, p++) 
	{
		m_grid[i] = new Shape(i);
		m_grid[i]->addData(Shapes, Empty);
		addItem(m_grid[i]);
		*p = Empty;
	}
	fl = new FinalLine;
	block = false;
	endGame = false;
}

void GraphicsScene::changeLevel()
{	
	m_machine.setLevel(m_machine.level() + 1);
	updateLevelText();
}

void GraphicsScene::changeLevel(int level)
{
	m_machine.setLevel(level);
	updateLevelText();
}

void GraphicsScene::updateLevelText()
{
	QString level_string;
	int machine_level = m_machine.level();
	if (machine_level == 0)
		level_string = "Level: I";
	else
		level_string = "Level: II";
	m_textLevel->setPlainText(level_string);
}

void GraphicsScene::mousePressEvent(QGraphicsSceneMouseEvent* event)
{
	QGraphicsScene::mousePressEvent(event);
	QGraphicsItem* item = itemAt(event->scenePos());

	if (!block && !endGame)
	{
		block = true;
		
		if (item->data(Shapes) == Empty) {
			int element = item->data(Positions).toInt();
			
			animOp = new QPropertyAnimation(m_grid[element], "opacity");
			animOp->setStartValue(0);
			animOp->setDuration(500);
			animOp->setStartValue(0);
			animOp->setEndValue(1);
			animOp->start();	
			m_grid[element]->addData(Shapes,Human);
			matrix[element/3][element%3] = Human;

			connect(animOp, SIGNAL(valueChanged(QVariant)), this, SLOT(endUserAnimation(QVariant)));
		}
		else
			block = false;
	}
	
	if (item->data(Shapes) == "textLevel")
		changeLevel();
	if (item->data(Shapes) == "restartGame")
		cleanAll();
}

bool GraphicsScene::algunaCasillaLibre() 
{
	for (int i=0 ; i< All ; ++i)
	{
		if (m_grid[i]->data(Shapes).toInt() == Empty) 
			return true;
	}
	return false;
}

void GraphicsScene::endUserAnimation(QVariant valueChanged)
{
	if (valueChanged.toDouble() >= 1) 
	{
		if (!checkFinish())
		{
			block = true;
			CPUMove();
		}
	}
}

void GraphicsScene::CPUMove()
{
	m_machine.updateMatrix(matrix);
	int element = m_machine.bestPosition();
	
	animOp = new QPropertyAnimation(m_grid[element], "opacity");
	animOp->setStartValue(0);
	animOp->setDuration(1000);
	animOp->setStartValue(0);
	animOp->setEndValue(1);
	animOp->start();	

	m_grid[element]->addData(Shapes,Machine);
	matrix[element/3][element%3] = Machine;
	connect(animOp, SIGNAL(valueChanged(QVariant)), this, SLOT(endMachineAnimation(QVariant)));
}

void GraphicsScene::endMachineAnimation(QVariant valueChanged)
{
	if (valueChanged.toDouble() >= 1) 
	{
		checkFinish();
		block = false;
	}
}

bool GraphicsScene::checkFinish() 
{		
	// Machine = maquina  // Human = jugador
	int result = -2;
	/*	0	1	2	*/
	/*	3	4	5	*/
	/*	6	7	8	*/
	/********** GANA MAQUINA ********/
	// Filas
	if (matrix[0][0] == Machine && matrix[0][1] == Machine && matrix[0][2] == Machine) {
		result = 1;
		fl->setPoints(QPoint(-90,-50), QPoint(190,-50));
	}
	if (matrix[1][0] == Machine && matrix[1][1] == Machine && matrix[1][2] == Machine) {
		result = 1;
		fl->setPoints(QPoint(-90,50), QPoint(190,50));
	}
	if (matrix[2][0] == Machine && matrix[2][1] == Machine && matrix[2][2] == Machine) {
		result = 1;
		fl->setPoints(QPoint(-90,150), QPoint(190,150));
	}
	// Columnas
	if (matrix[0][0] == Machine && matrix[1][0] == Machine && matrix[2][0] == Machine) {
		result = 1;
		fl->setPoints(QPoint(-50,-90), QPoint(-50, 190));
	}
	if (matrix[0][1] == Machine && matrix[1][1] == Machine && matrix[2][1] == Machine) {
		result = 1;
		fl->setPoints(QPoint(50,-90), QPoint(50,190));
	}
	if (matrix[0][2] == Machine && matrix[1][2] == Machine && matrix[2][2] == Machine) {
		result = 1;
		fl->setPoints(QPoint(150,-90), QPoint(150,190));
	}
	// Diagonal
	if (matrix[0][0] == Machine && matrix[1][1] == Machine && matrix[2][2] == Machine) {
		result = 1;
		fl->setPoints(QPoint(-75,-75), QPoint(175,175));
	}
	if (matrix[0][2] == Machine && matrix[1][1] == Machine && matrix[2][0] == Machine) {
		result = 1;
		fl->setPoints(QPoint(175,-75), QPoint(-75,175));
	}

	/********** GANA JUGADOR ********/
	//Columnas
	if (matrix[0][0] == Human && matrix[0][1] == Human && matrix[0][2] == Human) {
		result = 0;
		fl->setPoints(QPoint(-90,-50), QPoint(190,-50));
	}
	if (matrix[1][0] == Human && matrix[1][1] == Human && matrix[1][2] == Human) {
		result = 0;
		fl->setPoints(QPoint(-90,50), QPoint(190,50));
	}
	if (matrix[2][0] == Human && matrix[2][1] == Human && matrix[2][2] == Human) {
		result = 0;
		fl->setPoints(QPoint(-90,150), QPoint(190,150));
	}
	//Filas
	if (matrix[0][0] == Human && matrix[1][0] == Human && matrix[2][0] == Human) {
		result = 0;
		fl->setPoints(QPoint(-50,-90), QPoint(-50, 190));
	}
	if (matrix[0][1] == Human && matrix[1][1] == Human && matrix[2][1] == Human) {
		result = 0;
		fl->setPoints(QPoint(50,-90), QPoint(50,190));
	}
	if (matrix[0][2] == Human && matrix[1][2] == Human && matrix[2][2] == Human) {
		result = 0;
		fl->setPoints(QPoint(150,-90), QPoint(150,190));
	}
	// Diagonal
	if (matrix[0][0] == Human && matrix[1][1] == Human && matrix[2][2] == Human) {
		result = 0;
		fl->setPoints(QPoint(-75,-75), QPoint(175,175));
	}
	if (matrix[0][2] == Human && matrix[1][1] == Human && matrix[2][0] == Human) {
		result = 0;
		fl->setPoints(QPoint(175,-75), QPoint(-75,175));
	}

	// Mensaje si el juego esta acabado
	if (result == 0) {
		block = true;
		endGame = true;
		addItem(fl);
		fl->setZValue(1);
		update();
		ganador = Human;
		endFinalLine();
		
		return true;
	}
	if (result == 1) {
		block = true;
		endGame = true;
		addItem(fl); 
		fl->setZValue(2);
		update();
		ganador = Machine;
		endFinalLine();
		
		return true;
	}
	if (result == -2 && !algunaCasillaLibre()) 	{
		block = true;
		endGame = true;
		update();
		ganador = Human+Machine;
		endFinalLine();
		
		return true;
	}
	return false;
}

void GraphicsScene::endFinalLine()
{
	if (ganador == Machine)
		m_winner->setPlainText("Machine's win!");
	if (ganador == Human)
		m_winner->setPlainText("Player's win!");
	if (ganador == Human+Machine)
		m_winner->setPlainText("Tie Game!");
	
	m_restart->setPlainText("Restart");
}

void GraphicsScene::cleanAll()
{
	int* p = matrix[0];
	for (int i = 0 ; i < All ; ++i, p++)
	{
		delete m_grid[i];
		m_grid[i] = new Shape(i);
		m_grid[i]->addData(Shapes, Empty);
		addItem(m_grid[i]);
		*p = Empty;
	}
	removeItem(fl);
	m_winner->setPlainText("");
	m_restart->setPlainText("");
	block = false;
	endGame = false;
	if (ganador == Human)
		CPUMove();
}

#include "graphicsscene.moc"

