/* The MIT License:

Copyright (c) 2009 Ivan Gagis

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE. */

/*
* aumiks 0.1
*/

#pragma once

#include <ting/debug.hpp>
#include <ting/Buffer.hpp>
#include <ting/Array.hpp>
#include <ting/Ref.hpp>

#include "config.hpp"

namespace aumiks{

using namespace ting;

class Sample : public RefCounted{
public:
	real rate;//sampling rate (e.g. 44100 Hz)
	Array<real> data;
	real freq;//base frequency of the sound

	static Ref<Sample> GenerateFromLineString(
			Buffer<float>& x,
			Buffer<float>& y,
			ting::uint outputSampleLength,
			float baseFreq
		)
	{
		ASSERT(x.Size() == y.Size())

		Ref<Sample> ret(new Sample());

		ret->freq = baseFreq;

		ret->data.Init(outputSampleLength);

		ting::uint numPoints = x.Size();
		ASSERT(numPoints != 0)

		float t = float(ret->data.Size());//period in samples
		float d = t / x[numPoints - 1];// time scale

		ting::uint j = 0;

		for(ting::uint i = 0; i < numPoints - 1 && j < ret->data.Size(); ++i){
			float segLenX = (x[i + 1] - x[i]) * d;
			float segDelta = y[i + 1] - y[i];
			float k = segDelta/segLenX;

			float b = - k * x[i] * d + y[i];

			while(j < x[i + 1] * d && j < ret->data.Size()){
				ASSERT_INFO(j < ret->data.Size(), "j = " << j << " ret->data.Size() = " << ret->data.Size())
				ret->data[j] = k * j + b;
	//			TRACE(<< (ret->data[j]) << std::endl)
				++j;
			}
		}

		return ret;
	}


	static Ref<Sample> GenerateFromFourier(
			Buffer<float>& c,
			Buffer<float>& s,
			ting::uint outputSampleLength,
			float baseFreq
		)
	{
		ASSERT(c.Size() == s.Size())

		Ref<Sample> ret(new Sample());

		ret->freq = baseFreq;

		ret->data.Init(outputSampleLength);

		float f = D2Pi<float>() * 1 / float(outputSampleLength);

		for(ting::uint i = 0; i < ret->data.Size(); ++i){
			ASSERT(i < ret->data.Size())

			ret->data[i] = 0;

			for(ting::uint j = 0; j < c.Size(); ++j){
				ret->data[i] += c[j] * Cos((j + 1) * f * i) + s[j] * Sin((j + 1) * f * i);
			}
		}

		return ret;
	}
};

}//~namespace
