using GLib;
using Gtk;
using Gdk;

public class Tear.GestureManager : GLib.Object {    // The Gesture Manager is a modified version of the 
													  // JS gesture manager Moousture by MaXPert

	construct {
		
	}
		
}

public class Tear.Probe : GLib.Object {
	
	private int posX;
	private int posY;
	
	public bool stopEvent { get; set construct; }
	public Gtk.Widget target { get; set construct; }

    public Probe ( Gtk.Widget target, bool? stopEvent ) {
        this.target = target;
        
        if (stopEvent)
        	this.stopEvent = stopEvent;
        else
        	this.stopEvent = false;
    }

	construct {
		this.posX = this.posY = -1;
		
		this.target.motion_notify_event += _track;
	}
	
	public bool _track(Gtk.Widget widget, Gdk.EventMotion event) {
		this.posX = (int) event.x;
		this.posY = (int) event.y;
		
		return !this.stopEvent;
	}
	
    public int[] probe ()
    {
        return {this.posX, this.posY};
    }	
}

public class Tear.Monitor : GLib.Object {

	private int timer;
	private int prevX;
	private int prevY;
	private bool wasStable;
	private Tear.Probe prober;
	private eventFunc onStable;
	private eventFunc onUnstable;
	private eventFunc onMove;

	public int delay { get; set construct; }
	public int threshold { get; set construct; }

    public Monitor ( int? delay, int? threshold ) {
	    
		this.delay = 20;
	    if (delay != null)
        	this.delay = delay;
        
       	this.threshold = 1;
        if (threshold != null)
        	this.threshold = threshold;
    }

	construct {
        this.prevX = 0;
        this.prevY = 0;
        this.wasStable = false;
   	}
	
    public bool _monitor() {
        var positions = this.prober.probe();
        var posX = positions[0];
        var posY = positions[1];
        
        if ( (posX - this.prevX).abs() < this.threshold && ( posY - this.prevY ).abs() < this.threshold ) {
            if( !this.wasStable && this.onStable != null )
                this.onStable( positions );
            this.wasStable = true;
        } else {
            if( this.wasStable && this.onUnstable != null )
                this.onUnstable( positions );
            else
            	if (this.onMove != null)
                	this.onMove( positions );
            this.wasStable = false;
		}

		this.prevX = posX;
		this.prevY = posY;
		
		return true;
	}

	public void start ( Tear.Probe prober, eventFunc? onStable, eventFunc? onUnstable, eventFunc? onMove ){
		if ( this.timer != -1 )
			this.stop();
		this.prober = prober;
		this.onStable = onStable;
		this.onUnstable = onUnstable;
		this.onMove = onMove;
		this.timer = (int) GLib.Timeout.add (this.delay, this._monitor);
	}
    
    public void stop() {
        GLib.Source.remove (this.timer);
	    this.timer = -1;
    }
    
    public delegate void eventFunc ( int[] pos );

}

public class Tear.Recorder : GLib.Object {
	
	private Tear.GestureTrack[] movLog;
	
	public Tear.GestureMatcher matcher { get; set construct; }
	public int minSteps { get; set construct; }
	public int maxSteps { get; set construct; }

    public Recorder ( int? minSteps, int? maxSteps, Tear.GestureMatcher? matcher ) {

   		this.matcher = matcher;
	    
		this.minSteps = 4;
	    if (minSteps != null)
        	this.minSteps = minSteps;
        
       	this.maxSteps = 8;
        if (maxSteps != null)
        	this.maxSteps = maxSteps;
    }

	construct {
		this.movLog = new Tear.GestureTrack[0];
		movLog += new Tear.GestureTrack(0, 0);
	}
	
    public void onStable ( int[] position ) {
        if( this.movLog.length < this.minSteps ){
            this.movLog = new Tear.GestureTrack[0];
            return;
        }
    
        if(this.matcher != null && this.matcher.match != null)
            this.matcher.match(this.movLog);
    
//	    this.fireEvent('complete', [this.movLog]);
        this.movLog = new Tear.GestureTrack[0];
    }
  
    public void onUnstable ( int[] position ) {
    	this.movLog = new Tear.GestureTrack[0];
        this.movLog += position;
//	    this.fireEvent('start');
    }
    
    public void onMove ( int[] position ){
        if(this.movLog.length > this.maxSteps)
            return;
        this.movLog += position;
    }

}

public class Tear.GestureMatcher : GLib.Object {

	public delegate void GestureCallback ( double movement );
	public GestureCallback[] mCallbacks;
	public int[,] mGestures;
	
	construct {
		this.mCallbacks = new GestureCallback[0];
		this.mGestures = new int[0,0];
	}

    public int[] angelize( Tear.GestureTrack[] track ) {
        var ret = new int[0];
        
        for( int i = 1; i < track.length - 1; i++ )
            ret += this.getAngles( track[i], track[i+1] );
            
        return ret;
    }
    
    private int getAngles ( GestureTrack oldP, GestureTrack newP ) {
        int diffx = newP.x - oldP.x;
        int diffy = newP.y - oldP.y;
        double a = Math.atan2( diffy, diffx ) + Math.PI / 8;
        
        if( a < 0 ) 
        	a = a + (2 * Math.PI);
        
        a = Math.floor( a / ( 2 * Math.PI ) * 360 ) / 45;
        return (int) Math.floor( a );
    }
 
	private void addGesture( int[] gesture, GestureCallback mcallback ) {
		this.mCallbacks += mcallback;
		this.mGestures += gesture;
	}

    public virtual void match( Tear.GestureTrack[] track ) {}

}

public class Tear.LevenMatcher : Tear.GestureMatcher {
 
	public void onMatch ( int[] mov ) {
		var cbLen = this.mCallbacks.length;

		if ( cbLen < 1 )
			return;

		var minIndex = 0;
		var minDist = this.levenDistance( mov, this.mGestures[0] );

        for ( int p = 1; p < cbLen; p++ ) {
			var nwDist = this.levenDistance( mov, this.mGestures[p] );
			if( nwDist < minDist ) {
				minDist = nwDist;
				minIndex = p;
			}
		}
 
		this.mCallbacks[ minIndex ] ( minDist / mov.length );
	}
 
	public int levenDistance( int[] v1, int[] v2 ) {
        var d = new int[0,0];
        
		if (v1[0] != v2[0])
			d[0,0] = 1;
		else
			d[0,0] = 0;
 
        for( int i = 1; i < v1.length; i++)
            d[i,0] = d[i-1,0] + 1;
 
        for( int j = 1; j < v2.length; j++)
			d[0] += d[0,j-1] + 1;
            
        for( int i = 1; i < v1.length; i++) {
            for( int j=1; j < v2.length; j++) {
                int cost = 0;
                if (v1[i] != v2[j])
                    cost = 1;
                
                d[i,j] = d[i-1,j] + 1;
                if ( d[i,j] > d[i,j-1]+1 ) 
                	d[i,j] = d[i,j-1] + 1;
                if ( d[i,j] > d[i-1,j-1] + cost ) 
                	d[i,j] = d[i-1,j-1] + cost;
            }
		}

		int? output = (int) d[v1.length-1,v2.length-1];
        return output != null ? output : 0;
    }
    
    public override void match( Tear.GestureTrack[] track ) {
		var a = this.angelize( track );
		
		this.onMatch( a );
    }
}

public class Tear.ReducedLevenMatcher : Tear.LevenMatcher {
 
	private int?[] reduce ( int?[] seq ) {
		var ret = new int[0];
		 
		ret += seq[0];
		var prev = seq[0];
		 
		for ( int i = 1; i < seq.length; i++ )
			if ( prev != seq[i] ) {
				ret += seq[i];
				prev = seq[i];
			}
		 
		return ret;
	}
 
	public new void onMatch ( owned int?[] mov ) {
		mov = this.reduce( mov );
		 
		var cbLen = this.mCallbacks.length;
		 
		if ( cbLen < 1 || mov[0] != null )
			return;
		 
		var minIndex = 0;
		var minDist = this.levenDistance( mov, this.mGestures[0] );
			
		for ( int p = 1; p < cbLen; p++ ) {
			var nwDist = this.levenDistance( mov, this.mGestures[p] );
			 
			if( nwDist < minDist ) {
				minDist = nwDist;
				minIndex = p;
			}
		}
		 
		this.mCallbacks[minIndex] ( minDist / mov.length );
	}
	
    public override void match( Tear.GestureTrack[] track ) {
		var a = this.angelize( track );
		
		this.onMatch( a );
    }
}

public class Tear.GestureTrack : GLib.Object {
	
	public int x { get; set construct; }
	public int y { get; set construct; }
	
    public GestureTrack ( int dx = 0, int dy = 0 ) {
        this.x = dx;
        this.y = dy;
    }
	
}

