import QtQuick 1.0

Item {
    id: container
    width: 640
    height: 480

    property bool txtSmooth: true
    property color backgroundColor: "ivory"
    property color currentItemColor: "#FFF8C6"
    property color pressColor: "#FDE688"
    property int listFontSize : 32

    property string activePos: "http://finance.google.com/finance/feeds/default/portfolios/1/positions"
    property int selectedIdx: 0
    property bool editMode: false    

    property bool pfoLocal: false
    property bool pfoIsYahoo: false

    property int gPfoLength: 0

    signal close
    signal update

    PosEdit {
        id: posEditView
        anchors.fill: parent
        visible: false

        // Assigning pfoLocal not working
        editMode: true //pfoLocal
        pfoLocal: true //pfoLocal

        onClose: {
            updatePfoNum(posEditView.pfoLocalIdx);
            container.state = "edit";
        }
    }

    ListView {
        id: pfoView
        anchors.fill: parent
        anchors.bottomMargin: 90
        model: pfoModel
        delegate: pfoDelegate
        snapMode: ListView.SnapToItem
    }

    Component {
        id: pfoDelegate

        Item {
            id: pfoDelegateItem
            width: pfoView.width
            height: 70

            Rectangle {
                id: background
                x: 5; y: 2; width: parent.width - x*2; height: parent.height - y*2
                color: container.backgroundColor
                border.color: "#3577B9"
                radius: 5
            }

            Text {
                id: nameBox
                anchors { top: background.top; left: background.left; leftMargin: 20; }//verticalCenter: parent.verticalCenter }

                smooth: txtSmooth
                font.bold: true
                font.pixelSize: listFontSize
                //verticalAlignment: Text.AlignVCenter
                text: name
            }

            Text {
                id: excerptBox
                anchors { top: nameBox.bottom; left: background.left; leftMargin: 40; }

                smooth: txtSmooth
                color: "#808080"
                font.pixelSize: listFontSize - 14
                //verticalAlignment: Text.AlignVCenter
                text: excerpt
            }

            Text {
                id: countBoxHidden
                visible: false
                text: num
            }

            MouseArea {
                id: mouseArea
                anchors.fill: background
                onClicked: {
                    selectedIdx = index;
                    pfoDelegateItem.ListView.view.currentIndex = index

                    // Don't load for locally-created portfolio
                    if (!local) {
                        pfoLocal  = false;
                        activePos = feedLink;
                        console.log(activePos);

                        container.update();
                    }
                    else {
                        activePos = index - gPfoLength;

                        // Delete
                        if (mouse.x > (parent.width*5/6)) {
                            if (local&editMode) {
                                // Get local portfolio index
                                fileHandler.removePfo(activePos);
                                pfoModel.remove(index);
                                fileHandler.removePos(activePos);
                            }
                        }
                        // Go to posEditView or launch pfo
                        else {
                            pfoLocal  = true;
                            pfoIsYahoo = isYahoo;
                            posEditView.pfoLocalIdx = activePos;
                            posEditView.pfoIsYahoo  = isYahoo;

                            if (editMode) {
                                // Load data into posModel
                                loadLocalDataToPosModel(activePos);
                                container.state = "posEdit";
                            }
                            else          {
                                console.log("Local pfo: " + posEditView.pfoLocalIdx);
                                container.update();
                            }
                        }
                    }
                }

                // Edit current local portfolio
                onPressAndHold: {
                    selectedIdx = index;
                    activePos = index - gPfoLength;
                    pfoDelegateItem.ListView.view.currentIndex = index

                    if (local&&editMode) {
                        reloadEditDialog(name, excerpt, num);
                        container.state = "entryUpdate";
                    }
                }
            }

            SimpleRect {
                id: countBox
                anchors { right: background.right; rightMargin: 10; }
                anchors.verticalCenter: background.verticalCenter
                width: countBoxHidden.width+20
                color: (local) ? ((editMode) ? "#FF5555" : "#FFE9CA") : "#E0ECF7"
                text:  (local&editMode) ? "X" : num
                textColor: (local&editMode) ? "white" : "grey"
                textSize: listFontSize
                radius: 10
           }

            states: [
                State {
                    name: "Pressed"
                    when: mouseArea.pressed
                    PropertyChanges { target: background; color: container.pressColor }
                },
                State {
                    name: "highlightCurrentItem"
                    when: selectedIdx == index
                    PropertyChanges { target: background; color: container.currentItemColor }
                }
            ]
        }
    }

    Fader {
        fadingOpacity: 0.6
        // Bind our visibility status with Fader show/hide states.
        state: (container.state=="entry"||container.state=="entryUpdate") ? "faded" : ""
        transitionDuration: 200
        // Use the whole available area for fading & capturing key clicks.
        anchors.fill: parent
        // Use "light" fading color.
        color: "#f5f5f5"
    }

    BackButton {
        id: backButton
        anchors { bottom: container.bottom; left: container.left; leftMargin: 20; bottomMargin: 10; }
        width: 65
        height: 65
        onClicked: {
            if (container.state=="") {
                if (pfoLocal) { container.update(); }
                else           { container.close(); }
            }
            else if (container.state=="edit") { container.state = ""; }
        }
    }

    TextButton {
        id: storeButton
        anchors { bottom: parent.bottom; right: editButton.left; rightMargin: 20; bottomMargin: 10; }
        width: 60
        height: storeButton.width
        radius: storeButton.width
        visible: false

        color: "#F2F2F2"
        lightColor: "#FFFFFF"
        pressColor: storeButton.color

        text: "="
        textColor: "#B0D8FF"
        textSize: 60
        textBold: true

        onClicked: {
            // Save pfo
            fileHandler.storePfoAll();
            container.state = "";
        }
    }

    // Overload to do delete/save
    TextButton {
        id: editButton
        anchors { bottom: parent.bottom; right: parent.right; rightMargin: 20; bottomMargin: 10; }
        width: 60
        height: editButton.width
        radius: editButton.width

        color: "#F2F2F2"
        lightColor: "#FFFFFF"
        pressColor: editButton.color

        text: "+"
        textColor: "#B0D8FF"
        textSize: 50
        textBold: true

        onClicked: {
            if (container.state=="") { container.state = "edit"; }
            else {
                // Add new entry
                resetEditDialog();
                container.state = "entry";
            }
        }
    }

    Edit {
        id: editDialog
        state: "hidden"
        pfoMode: true

        onSave: {
            if (editDialog.myPfoName!="") {
                if (container.state=="entryUpdate") {
                    fileHandler.setPfo(activePos, editDialog.myPfoName, editDialog.myPfoDesc, editDialog.myIsYahoo);
                    fileHandler.storePfoAll();

                    pfoModel.set(selectedIdx, {
                                 "local": 1,
                                 "name": editDialog.myPfoName,
                                 "feedLink": "",
                                 "excerpt": editDialog.myPfoDesc,
                                 "isYahoo": editDialog.myIsYahoo,
                                 "num": 0
                    });
                }
                else {
                    fileHandler.storePfo(editDialog.myPfoName, editDialog.myPfoDesc, editDialog.myIsYahoo);

                    pfoModel.append({
                                    "local": 1,
                                    "name": editDialog.myPfoName,
                                    "feedLink": "",
                                    "excerpt": editDialog.myPfoDesc,
                                    "isYahoo": editDialog.myIsYahoo,
                                    "num": 0
                    });
                }
            }

            container.state = "edit";
        }

        onCancel: { container.state = "edit"; }
    }

    states: [
        State {
            name: "entry"
            PropertyChanges { target: editDialog; state: "shown"  }
            StateChangeScript {
                script: { console.log("pfoMenu.entry") }
            }
        },
        State {
            name: "entryUpdate"
            PropertyChanges { target: editDialog; state: "shown"  }
            StateChangeScript {
                script: { console.log("pfoMenu.entryUpdate") }
            }
        },
        State {
            name: "edit"
            PropertyChanges { target: storeButton; visible: true  }

            PropertyChanges { target: container; editMode: true  }

            StateChangeScript {
                name: "debug"
                script: { console.log("pfoMenu.edit") }
            }
        },
        State {
            name: "posEdit"
            PropertyChanges { target: posEditView; visible: true  }
            PropertyChanges { target: pfoView; visible: false  }

            PropertyChanges { target: backButton; visible: false  }
            PropertyChanges { target: storeButton; visible: false  }
            PropertyChanges { target: editButton; visible: false  }

            StateChangeScript {
                name: "debug"
                script: { console.log("pfoMenu.posEdit") }
            }
        }
    ]

    function resetEditDialog() {
        editDialog.myPfoName   = "";
        editDialog.myPfoDesc   = "";
    }

    function reloadEditDialog(name, excerpt) {
        editDialog.myPfoName   = name;
        editDialog.myPfoDesc  = excerpt;
    }


    function loadLocalDataToPosModel(idx) {
        // Don't use clear, for some reasons the clear affects posEditView
        //posModel.clear();
        for (var i=posModel.count-1; i>=0; i--) {
            //console.log("Remove "+posModel.get(i).name);
            posModel.remove(i);
        }

        var fileOk = fileHandler.loadPos(idx);

        if (fileOk==0){
            for (var i=0; i < fileHandler.localPosSymbol.length; i++) {
                posModel.append({
                    "name": fileHandler.localPosSymbol[i],
                    "exchange": fileHandler.localPosExg[i],
                    "share": fileHandler.localPosShare[i],
                    "shareCost": fileHandler.localPosCost[i]
                    //"stopPrice": fileHandler.localPosStop[i]
                });
            }
        }
    }

    function updatePfoNum(idx) {
        // C-model pfoNum is saved in storePos functions
        pfoModel.set(selectedIdx, {"num": fileHandler.localPosSymbol.length});
        fileHandler.storePfoAll();
    }
}


