/*
    Stockona is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Stockona is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with NewsFlow. If not, see <http://www.gnu.org/licenses/>.
*/
import QtQuick 1.0

import "js/script.js" as Script
import "js/storage.js" as Storage

Rectangle {
    id: mainWindow

    width: 640
    height: 480
    color: "#B0D8FF"

    property bool showWaitIndicator: waitState.state == "shown"
    property int startIndicator: 0

    property string activePos: ""
    property int pfoSelectedIdx: 0
    property int posSelectedIdx: 0

    property bool pfoIsLocal: false
    property bool pfoIsYahoo: false

    property string pfoName: ""
    property string posName: ""
	
    //////////////////////////
    //  functions
    //////////////////////////

    function convertExchangeForYahoo (name, exg) {
        // replace .* with ^*
        if (name.indexOf(".")==0)
            return name.replace(".", "^");

        //console.log(name + exg);
        if (exg=="NYSE" || exg=="NASDAQ") { return name;                }
        else if (exg=="OTC")              { return (name + "." + "OB"); }
        else if (exg=="AMEX")             { return (name + "." + "OB"); }
        else if (exg=="TSXV")             { return (name + "." + "V");  }
        else if (exg=="TSX")              { return (name + "." + "TO"); }
        else if (exg=="LON")              { return (name + "." + "L");  }
        // Australia
        else if (exg=="ASX")              { return (name + "." + "AX"); }
        // Asian
        else if (exg=="TPE")              { return (name + "." + "TW"); }
        else if (exg=="HKG")              { return (name + "." + "HK"); }
        else if (exg=="SHA")              { return (name + "." + "SS"); }
        else if (exg=="SHE")              { return (name + "." + "SZ"); }
        else if (exg=="SEO")              { return (name + "." + "KS"); }
        else                              { return "NULL";              }
    }

    function storeParams(key, data) {
        if (key=="username")
            Storage.setKeyValue(key, settingMenu.username);

        if (key=="password")
            Storage.setKeyValue(key, settingMenu.password);

        // Robustness, minimum timer == 5
        if (key=="timer" && (data != settingMenu.timer)) {
            if (settingMenu.timer < "15") {
                Storage.setKeyValue(key, "15");
                settingMenu.timer="15";
            }
            else {
                Storage.setKeyValue(key, settingMenu.timer);
            }
        }

        if (key=="newsWebview" && (data != settingMenu.useWebview))
            Storage.setKeyValue(key, settingMenu.useWebview);

        if (key=="pfoStats") {
            var tmp;
            tmp = (settingMenu.useWidgetView<<1)^(settingMenu.usePortfolioView);

            Storage.setKeyValue(key, tmp);
        }

        if (key=="activePos" && (data != mainWindow.activePos))
            Storage.setKeyValue(key, mainWindow.activePos);

        if (key=="pfoActiveIdx" && (data != mainWindow.pfoSelectedIdx))
            Storage.setKeyValue(key, mainWindow.pfoSelectedIdx);

        if (key=="pfoIsLocal") {
            var tmp;
            tmp = (mainWindow.pfoIsLocal<<1)^(mainWindow.pfoIsYahoo);
            Storage.setKeyValue(key, tmp);
        }
    }

    // Restore settings
    function setParams (key, data) {
        if(key=="timer") {
            if (data == "")   { settingMenu.timer = 15; }
            else              { settingMenu.timer = data; }
        }
        else if (key=="newsWebview") {
            if (data != "")   { settingMenu.useWebview = data; }
        }
        // Overload to pfoStats/posWidget
        else if (key=="pfoStats") {
            if (data != "")   {
                settingMenu.useWidgetView    = data&2;
                settingMenu.usePortfolioView = data&1;
            }
        }
        else if (key=="activePos") {
            if (data != "") { mainWindow.activePos = data; }
        }
        else if (key=="pfoActiveIdx") {
            if (data != "") { mainWindow.pfoSelectedIdx = data; }
        }
        // Overload to pfoIsLocal/pfoIsYahoo
        else if (key=="pfoIsLocal") {
            if (data != "") {
                //console.log("pfoIsLocal=" + data);
                mainWindow.pfoIsLocal = data&2;
                mainWindow.pfoIsYahoo = data&1;
            }
        }
    }

    function setLoginCredentials(key, data) {
        if(key=="username") {
            settingMenu.username = data;
        } else {
            settingMenu.password = data;
        }
        // Auto login
        if(settingMenu.username.length>0 &&
                settingMenu.password.length>0) {
            mainWindow.state = "inLogin";
        } else {
            startIndicator++;
            if(startIndicator==2) {
                console.log("Try to show login dialog");
                mainWindow.state = "inSettingView";
            }
        }
    }

    function invokeWorker(mode) {
        var msg = {'action': mode,
            'model': posModel,
            'activePos': activePos,
            //'pfoIsYahoo': pfoIsYahoo,
            'gainColorMode': false,
            'sid': Script.returnSid()
        };

        worker.sendMessage(msg);
    }

    function loadPosition() {
        if (waitState.state=="hidden") {
//            Script.loadOnePosition(activePos, pfoIsLocal, pfoIsYahoo, 0);
//            fetchPosData();
            if (pfoIsYahoo||pfoIsLocal) {
                Script.loadOnePosition(activePos, pfoIsLocal, pfoIsYahoo, 0);
                fetchPosData();
            }
            else  {
                invokeWorker('loadPosition');
            }
        }
    }

    function doLogin() {
        if (mainWindow.state=="inLogin")
            Script.login(settingMenu.username, settingMenu.password);
        else
            mainWindow.state = "inLogin";
    }

    function checkForDelimiter(str) {
        return str.replace(/:/, ",");
    }
	
    function fetchPosData() {
        if (posModel.count!=0) {
        posLoader.item.myName             = posModel.get(posLoader.item.selectedIdx).name;
        posLoader.item.myExg              = posModel.get(posLoader.item.selectedIdx).exchange;
        posLoader.item.myFullname         = posModel.get(posLoader.item.selectedIdx).fullName;
        posLoader.item.myQuotePrice       = posModel.get(posLoader.item.selectedIdx).quotePrice      ;
        posLoader.item.myQuoteChgColor    = posModel.get(posLoader.item.selectedIdx).quoteChgColor   ;
        posLoader.item.myQuoteChg         = posModel.get(posLoader.item.selectedIdx).quoteChg        ;
        posLoader.item.myQuoteChgPtg      = posModel.get(posLoader.item.selectedIdx).quoteChgPtg     ;
        posLoader.item.myQuoteAsk         = posModel.get(posLoader.item.selectedIdx).quoteAsk        ;
        posLoader.item.myQuoteBid         = posModel.get(posLoader.item.selectedIdx).quoteBid        ;
        posLoader.item.myShare            = posModel.get(posLoader.item.selectedIdx).share           ;
        posLoader.item.myShareValue       = posModel.get(posLoader.item.selectedIdx).shareValue      ;
        posLoader.item.myShareCost        = posModel.get(posLoader.item.selectedIdx).shareCost       ;
        posLoader.item.myShareDayGain     = posModel.get(posLoader.item.selectedIdx).shareDayGain    ;
        posLoader.item.myShareGain        = posModel.get(posLoader.item.selectedIdx).shareGain       ;
        posLoader.item.myShareGainPercent = posModel.get(posLoader.item.selectedIdx).shareGainPercent;
        posLoader.item.myQuoteDayHi       = posModel.get(posLoader.item.selectedIdx).quoteDayHi ;
        posLoader.item.myQuoteDayLo       = posModel.get(posLoader.item.selectedIdx).quoteDayLo ;
        posLoader.item.myQuote52wHi       = posModel.get(posLoader.item.selectedIdx).quote52wHi ;
        posLoader.item.myQuote52wLo       = posModel.get(posLoader.item.selectedIdx).quote52wLo ;
        posLoader.item.myQuoteMktCap      = posModel.get(posLoader.item.selectedIdx).quoteMktCap;
        posLoader.item.myQuoteEps         = posModel.get(posLoader.item.selectedIdx).quoteEps   ;
        posLoader.item.myQuotePe          = posModel.get(posLoader.item.selectedIdx).quotePe    ;
        posLoader.item.myQuoteBeta        = posModel.get(posLoader.item.selectedIdx).quoteBeta  ;
        posLoader.item.myQuoteVol         = posModel.get(posLoader.item.selectedIdx).quoteVol   ;
        posLoader.item.myQuoteAvgVol      = posModel.get(posLoader.item.selectedIdx).quoteAvgVol;
        }
    }

    function restoreThemeColor () {
        header.color = "#7DBFFF";
        mainWindow.color = "#B0D8FF";
    }

    function storeSettings() {
        Storage.getKeyValue("timer", storeParams);
        Storage.getKeyValue("pfoStats", storeParams);
        Storage.getKeyValue("newsWebview", storeParams);
    }

    function storeAccount() {
        Storage.getKeyValue("username", storeParams);
        Storage.getKeyValue("password", storeParams);
        Storage.getKeyValue("activePos", storeParams);
    }

    function toggleView () {
        // normal - portfolio - widget
        if (posLoader.item.usePortfolioView==true) {
            posLoader.item.useWidgetView    = true;
            posLoader.item.usePortfolioView = false;
        }
        else if (posLoader.item.useWidgetView == false) {
            posLoader.item.useWidgetView    = false;
            posLoader.item.usePortfolioView = true;
            restoreThemeColor();
        }
        else {
            posLoader.item.useWidgetView    = false;
            posLoader.item.usePortfolioView = false;
            restoreThemeColor();
        }
    }

    /////////////////////////////////////////////////
    // Main
    /////////////////////////////////////////////////

    WorkerScript {
        id: worker
        source: "js/network.js"
        onMessage: { waitState.state = "hidden"}
    }

    // Loader to improve performance
    Loader {
        id: newsLoader
        width: parent.width
        anchors { top: header.bottom; bottom: parent.bottom; left: parent.left; right: parent.right; }
    }

    Connections {
        id: newsLoaderConnect
        target: newsLoader.item

        onClose: {
            mainWindow.state = 'inStatsView'
        }
    }

    Loader {
        id: pfoLoader
        width: parent.width
        anchors { top: header.bottom; bottom: parent.bottom; left: parent.left; right: parent.right; }

        onLoaded: {
            pfoLoader.item.pfoSelectedIdx = mainWindow.pfoSelectedIdx;
            pfoLoader.item.pfoIsLocal = mainWindow.pfoIsLocal ;
            pfoLoader.item.pfoIsYahoo = mainWindow.pfoIsYahoo ;
        }
    }

    Connections {
        id: pfoLoaderConnect
        target: pfoLoader.item

        onUpdate: {
            mainWindow.activePos = pfoLoader.item.activePos;
            mainWindow.pfoSelectedIdx = pfoLoader.item.pfoSelectedIdx;
            mainWindow.pfoIsLocal = pfoLoader.item.pfoIsLocal;
            mainWindow.pfoIsYahoo = pfoLoader.item.pfoIsYahoo;

            Storage.getKeyValue("activePos", storeParams);
            Storage.getKeyValue("pfoActiveIdx", storeParams);
            Storage.getKeyValue("pfoIsLocal", storeParams);

            mainWindow.state = "inLoadPortfolio";
        }

        onClose: {
            mainWindow.state = "inMainView";
        }
    }

    Loader {
        id: posLoader
        width: parent.width
        anchors { top: header.bottom; bottom: parent.bottom; left: parent.left; right: parent.right; }

        onLoaded: {
            posLoader.item.selectedIdx      = mainWindow.posSelectedIdx;
            posLoader.item.usePortfolioView = settingMenu.usePortfolioView;
            posLoader.item.useWidgetView    = settingMenu.useWidgetView;

            fetchPosData();
        }
    }

    Connections {
        id: posLoaderConnect
        target: posLoader.item

        onInNewsView:  {
            mainWindow.posSelectedIdx = posLoader.item.selectedIdx
            mainWindow.state = 'inNewsView'
        }
        onInStatsView: { mainWindow.state = 'inStatsView' }
        onClose:  { mainWindow.state = 'inMainView' }
    }

    XmlListModel {
        id: rssModel

        query: "/rss/channel/item"

        XmlRole { name: "title"; query: "title/string()" }
        XmlRole { name: "link"; query: "link/string()" }
        XmlRole { name: "desc"; query: "description/string()" }
        XmlRole { name: "date"; query: "pubDate/string()"; }//isKey: true }

//        onStatusChanged: {
//            if (status == XmlListModel.Ready) {
//                console.log("FeedViewModel Status: ready")
//            } else if (status == XmlListModel.Error) {
//                console.log("FeedViewModel Status: error")
//            } else if (status == XmlListModel.Loading) {
//                console.log("FeedViewModel Status: loading")
//            }
//        }
    }

    ListModel {
         id: pfoModel
    }

    ListModel {
         id: posModel
    }

    Component.onCompleted: {
        Script.setComponents(pfoModel, posModel, waitState, errorState);
        Storage.getKeyValue("username", setLoginCredentials);
        Storage.getKeyValue("password", setLoginCredentials);
        Storage.getKeyValue("timer", setParams);
        Storage.getKeyValue("pfoStats", setParams);
        Storage.getKeyValue("newsWebview", setParams);
        Storage.getKeyValue("activePos", setParams);
        Storage.getKeyValue("pfoActiveIdx", setParams);
        Storage.getKeyValue("pfoIsLocal", setParams);
    }

    // Header: Minimize/Notification/Close
    TitleBar {
        id: header
        anchors { top: parent.top; left: parent.left; }
        width: parent.width
        showBusy: true
        text: "stockona"
        z: 1
        //height: (windowHelper.isMaemo5()) ? 40 : 0
        //visible: windowHelper.isMaemo5()

        onCallback: { mainWindow.state = "inPfoView"; }
    }

    Component {
        id: newsViewComp

        NewsView {
            id: newsView
            webViewEnb: settingMenu.useWebview

            // onClose: { mainWindow.state = 'inStatsView' }
        }
    }

    Component {
        id: posViewComp
        PosListView {
            id: mainView
            //visible: false
            //anchors { top: header.bottom; bottom: parent.bottom; left: parent.left; right: parent.right; }
            //anchors.fill: parent
            usePortfolioView: settingMenu.usePortfolioView
            useWidgetView: settingMenu.useWidgetView;

            //onInNewsView:  { mainWindow.state = 'inNewsView' }
            //onInStatsView: { mainWindow.state = 'inStatsView' }
            //onClose:  { mainWindow.state = 'inMainView' }
        }
    }

    Component {
        id: pfoMenuComp
        PfoListView {
            id: pfoMenu
            //visible: false
            //anchors.fill: parent
            //anchors { top: header.bottom; bottom: parent.bottom; left: parent.left; right: parent.right; }

            //onUpdate: {
            //    Storage.getKeyValue("activePos", storeParams);
            //    Storage.getKeyValue("pfoActiveIdx", storeParams);
            //    mainWindow.state = "inLoadPortfolio";
            //}
            //onClose: { mainWindow.state = "inMainView"; }
        }
    }

    SettingMenu {
        id: settingMenu
        state: "hidden"

        onCallPfo: {
            mainWindow.state = "inPfoView";
        }
        onSignin: {
            activePos = "";

            storeAccount();
            storeSettings();
            doLogin();
        }
        onSignout: {
            settingMenu.username = settingMenu.password = activePos = "";

            storeAccount();

            posModel.clear();
            pfoModel.clear();

            mainWindow.state = "inMainView";
        }
        onClose: {
            // Call storeParams function
            storeSettings();
            mainWindow.state = "inMainView";
        }
    }

    Row {
        anchors { right: parent.right; bottom: parent.bottom; margins: 20 }
        spacing: 10

        TextButton {
            id: menuButton
            text: "Menu"
            onClicked: {
                mainWindow.state = "inSettingView";
            }
        }

        TextButton {
            id: funcButton
            text: "View"
            onClicked: {
                toggleView();
            }
        }

        TextButton {
            id: refreshButton
            text: "Refresh"
            onClicked: {
                //console.log("##Refresh: " + pfoIsLocal + " " + pfoIsYahoo);
                loadPosition();
                mainWindow.state = "inMainView";
            }
        }
    }

    BusyDialog {
        id: msgDialog
        anchors.centerIn: parent
        busyIndicatorOn: true
        opacity: 0

        MouseArea {
            id: msgMouseArea
            anchors.fill: parent
            onClicked: { waitState.state = "hidden"; }
        }
    }

    StateGroup {
        id: waitState

        states: [
            State {
                name: "hidden"
            },
            State {
                name: "shown"
                PropertyChanges{ target: mainWindow; showWaitIndicator: true }
            },
            // Control by javascript core
            State {
                name: "portfolio"
                StateChangeScript { name: "gotoPorfolio"; script: { fsmTimer.restart(); } }
            }
        ]
    }

    // Error dialog
    BusyDialog {
        id: errorDialog
        anchors.centerIn: parent
        busyIndicatorOn: false
        opacity: 0
        message: errorState.reason

        MouseArea {
            id: errorMouseArea
            anchors.fill: parent
            onClicked: errorState.state = "hidden"
        }
    }

    // Error state is reset whenever not in mainView
    StateGroup {
        id: errorState

        property string reason: ""

        states: [
            State {
                name: "hidden"; PropertyChanges { target: errorDialog; opacity: 0 }
            },
            State {
                name: "shown"; PropertyChanges { target: errorDialog; opacity: 1 }
            },
            State {
                name: "promotePfo";
                StateChangeScript { name: "showPfo"; script: { fsmTimer.restart();} }
            }
        ]
    }

    // Timer to control update interval
    Timer {
        id: updateTimer
        interval: settingMenu.timer*1000
        repeat: true
        onTriggered: {
            //console.log("Timer="+settingMenu.timer+"s");
            loadPosition();
        }
    }

    // Use a timer to delay events to avoid race conditions
    Timer {
        id: fsmTimer
        interval: 800
        repeat: false
        onTriggered: {
            //console.log("Timer="+fsmTimer.interval+"s");
            // Stop if in setting menu
            if (mainWindow.state!="inSettingView") { mainWindow.state="inLoadPortfolio"; }
        }
    }

    Timer {
        id: fsmTimer2
        interval: 500
        repeat: false
        onTriggered: {
            //console.log("fsmTimer2");
            mainWindow.state="inMainView";
        }
    }

    ////////////////////////////
    // FSM
    ////////////////////////////
    states: [
        State {
            name: "inLogin";

            PropertyChanges { target: msgDialog; message: "Logging in"; opacity: 1; }
            PropertyChanges { target: errorState; state: "hidden"; }

            StateChangeScript {
                name: "loginStep"
                script: {
                    console.log("State.login");
                    Script.login(settingMenu.username, settingMenu.password);
                }
            }
        },

        State {
            name: "inLoadPortfolio";

            PropertyChanges { target: msgDialog; message: "Loading portfolio"; opacity: 1 }
            PropertyChanges { target: errorState; state: "hidden"; }

            StateChangeScript {
                name: "PortfolioStep"
                script: {
                    console.log("State.loadPortfolio");
                    // clear posModel here
                    Script.cleanModel(posModel);
                    Script.loadAllPortfolios();
                    Script.loadOnePosition(activePos, pfoIsLocal, pfoIsYahoo, 1);

                    // switch to posView, delayed to avoid race condition
                    fsmTimer2.restart();
                }
            }
        },	

        State {
            name: "inMainView";

            PropertyChanges { target: posLoader; sourceComponent: posViewComp }
            //PropertyChanges { target: newsLoader; sourceComponent: undefined }
            PropertyChanges { target: pfoLoader; sourceComponent: undefined }
            //PropertyChanges { target: newsLoader; sourceComponent: newsViewComp }

            PropertyChanges { target: menuButton; visible: true     }
            PropertyChanges { target: funcButton; visible: true     }
            PropertyChanges { target: refreshButton; visible: true  }

            StateChangeScript {
                name: "mainView"
                script: {
                    console.log("State.main");
                    updateTimer.restart();
                }
            }
        },

        State {
            name: "inPfoView";
            PropertyChanges { target: header; text: "Portfolio" }

            PropertyChanges { target: pfoLoader; sourceComponent: pfoMenuComp }

            // Still keep posView to prevent long delay in transition back
            //PropertyChanges { target: posLoader; sourceComponent: posViewComp }
            //PropertyChanges { target: posLoader.item; opacity: 0 }

            PropertyChanges { target: errorState; state: "hidden" }
            PropertyChanges { target: menuButton; visible: false     }
            PropertyChanges { target: funcButton; visible: false     }
            PropertyChanges { target: refreshButton; visible: false  }

            StateChangeScript {
                name: "pfoView"
                script: {
                    console.log("State.pfoView");

                    // Restore color
                    restoreThemeColor();

                    updateTimer.stop();
                    if (pfoModel.count==0) { console.log("Reload portfolio"); Script.loadAllPortfolios(); }
                }
            }
        },

        State {
            name: "inSettingView";

            PropertyChanges { target: settingMenu; visible: true; state: "shown" }
            PropertyChanges { target: header; text: "Setting" }

            PropertyChanges { target: posLoader; opacity: 0; sourceComponent: posViewComp }

            PropertyChanges { target: errorState; state: "hidden" }
            PropertyChanges { target: menuButton; visible: false     }
            PropertyChanges { target: funcButton; visible: false     }
            PropertyChanges { target: refreshButton; visible: false  }

            StateChangeScript {
                name: "settingView"
                script: {
                    console.log("State.settingView");

                    fsmTimer.stop();
                    fsmTimer2.stop();
                    updateTimer.stop();
                }
            }
        },

        State {
            name: "inNewsView";

            PropertyChanges { target: newsLoader; sourceComponent: newsViewComp }
            PropertyChanges { target: header; text: posName + " News" }

            // Don't destroy posLoader... otherwise back leads to default view.
            PropertyChanges { target: posLoader; opacity: 0; sourceComponent: posViewComp }

            PropertyChanges { target: errorState; state: "hidden" }
            PropertyChanges { target: menuButton; visible: false     }
            PropertyChanges { target: funcButton; visible: false     }
            PropertyChanges { target: refreshButton; visible: false  }

            PropertyChanges { target: mainWindow; color: "#B0D8FF" }

            StateChangeScript {
                name: "newsView"
                script: {
                    console.log("State.newsView");
                    updateTimer.stop();
                }
            }
        },

        State {
            name: "inStatsView";

            PropertyChanges { target: posLoader; sourceComponent: posViewComp }

            // Pre-load newsview
            PropertyChanges { target: newsLoader; opacity: 0; sourceComponent: newsViewComp }

            PropertyChanges { target: errorState; state: "hidden" }
            PropertyChanges { target: menuButton; visible: false     }
            PropertyChanges { target: funcButton; visible: false     }
            PropertyChanges { target: refreshButton; visible: false  }

            StateChangeScript { name: "statsView"; script: { console.log("State.statsView"); } }
        }
    ]
}
