/*
    Stockona is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Stockona is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with NewsFlow. If not, see <http://www.gnu.org/licenses/>.
*/

var sid = "";

// UI components
var waiting;
var error;

var model;
var pModel;

var itemsURL = "";
var gPfoLength;

// 0: US; 1: Asia
var quoteChgColorMode = 0;

function returnSid() {
    return sid;
}

function setComponents(mod, pos, wait, err) {
    model   = mod;
    pModel  = pos;
    waiting = wait;
    error   = err;
}

// This will parse a delimited string into an array of
// arrays. The default delimiter is the comma, but this
// can be overriden in the second argument.
function CSVToArray( strData, strDelimiter ){
// Check to see if the delimiter is defined. If not,
// then default to comma.
strDelimiter = (strDelimiter || ",");

// Create a regular expression to parse the CSV values.
var objPattern = new RegExp(
(
// Delimiters.
"(\\" + strDelimiter + "|\\r?\\n|\\r|^)" +

// Quoted fields.
"(?:\"([^\"]*(?:\"\"[^\"]*)*)\"|" +

// Standard fields.
"([^\"\\" + strDelimiter + "\\r\\n]*))"
),
"gi"
);

// Create an array to hold our data. Give the array
// a default empty first row.
var arrData = [[]];

// Create an array to hold our individual pattern
// matching groups.
var arrMatches = null;


// Keep looping over the regular expression matches
// until we can no longer find a match.
while (arrMatches = objPattern.exec( strData )){

// Get the delimiter that was found.
var strMatchedDelimiter = arrMatches[ 1 ];

// Check to see if the given delimiter has a length
// (is not the start of string) and if it matches
// field delimiter. If id does not, then we know
// that this delimiter is a row delimiter.
if (
strMatchedDelimiter.length &&
(strMatchedDelimiter != strDelimiter)
){

// Since we have reached a new row of data,
// add an empty row to our data array.
arrData.push( [] );

}


// Now that we have our delimiter out of the way,
// let's check to see which kind of value we
// captured (quoted or unquoted).
if (arrMatches[ 2 ]){

// We found a quoted value. When we capture
// this value, unescape any double quotes.
var strMatchedValue = arrMatches[ 2 ].replace(
new RegExp( "\"\"", "g" ),
"\""
);

} else {

// We found a non-quoted value.
var strMatchedValue = arrMatches[ 3 ];

}


// Now that we have our value string, let's add
// it to the data array.
arrData[ arrData.length - 1 ].push( strMatchedValue );
}

// Return the parsed data.
return( arrData );
}

function httpErrorHandling (status, statusText, show) {
    // Ignore 500 error
    if (status==404) {
        showError("Cannot load portfolio.\n\nCheck if you have portfolio selected.\nCreate one on Google if you don't have any.", show);
        errorState.state = "promotePfo";
    }
    else if (status!=200&&status!=500) {
        showError("API returned " + status + " " + statusText, show);
    }
}

/*
 params
   0: Google Auth, encrypted
   1: Google Query with auth key
   2: Yahoo Query
*/
function doWebRequest(method, url, params, callback, show) {
    var doc = new XMLHttpRequest();
    //console.log(method + " " + url);

    doc.onreadystatechange = function() {
        if (doc.readyState == XMLHttpRequest.HEADERS_RECEIVED) {
            var status = doc.status;
            var statusText = doc.statusText;
            httpErrorHandling(status, statusText, show);
        } else if (doc.readyState == XMLHttpRequest.DONE) {
            var data;
            var contentType = doc.getResponseHeader("Content-Type");

            if (params==1) {
                data = doc.responseXML.documentElement;
            }
            else {
                data = doc.responseText;
            }
//            var dbg = doc.responseText;
//            console.log(dbg);
            callback(data);
        }
    }

    doc.open(method, url);
    if(sid.length>0 && params==1) {
        // Google Finance ignore SID/LSID
        //console.log("Authorization GoogleLogin auth=" + sid);
        doc.setRequestHeader("Authorization", "GoogleLogin auth=" + sid);
        // Specifying a version
        doc.setRequestHeader("GData-Version", "2");
    }

    if(params<2) {
        //console.log("Sending: " + params);
        doc.setRequestHeader("Content-Type", "application/x-www-form-urlencoded");
        doc.send();
    } else {
        doc.send();
    }
}

function doWebRequestIdx(method, url, params, callback, idx, show) {
    var doc = new XMLHttpRequest();
    //console.log(method + " " + url);

    doc.onreadystatechange = function() {
        if (doc.readyState == XMLHttpRequest.HEADERS_RECEIVED) {
            var status = doc.status;
            var statusText = doc.statusText;
            httpErrorHandling(status, statusText, show);
        } else if (doc.readyState == XMLHttpRequest.DONE) {
            var data;
            var contentType = doc.getResponseHeader("Content-Type");
            if (params==1) {                
                data = doc.responseXML.documentElement;
            }
            else {
                data = doc.responseText;
            }
//            var dbg = doc.responseText;
//            console.log(dbg);
            callback(data, idx);
        }
    }

    doc.open(method, url);

    if(sid.length>0) {
        // Google Finance ignore SID/LSID
        doc.setRequestHeader("Authorization", "GoogleLogin auth=" + sid);
        doc.setRequestHeader("GData-Version", "2");
    }

    if(params<2) {
        //console.log("Sending: " + params);
        doc.setRequestHeader("Content-Type", "application/x-www-form-urlencoded");
        doc.send();
    } else {
        doc.send();
    }
}

//////////////////////////////////////
// Login
//////////////////////////////////////

function parseAuth(data, parameterName) {
    var parameterIndex = data.indexOf(parameterName + "=");
    if(parameterIndex<0) {
        console.log("Didn't find Auth");
        return "";
    }
    var equalIndex = data.indexOf("=", parameterIndex);
    if(equalIndex<0) {
        return "";
    }

    var lineBreakIndex = data.indexOf("\n", equalIndex+1)

    var value = "";
    value = data.substring(equalIndex+1, lineBreakIndex);
    return value;
}

function parseToken(data) {
    sid = parseAuth(data, "Auth");
    //console.log("Auth=" + sid);

    if(sid.length>0) {
        waiting.state = "portfolio";
    } else {
        //addError("Couldn't parse SID");
        showError("Login failed.\n\nCheck username, password, or network connection.", 1);
        waiting.state = "hidden";
    }
}

function login(email, password) {
    sid = "";
    error.reason = "";
    error.state  = "hidden";

    try {
        waiting.state = "shown";
        var url = "https://www.google.com/accounts/ClientLogin?Email=" + encodeURIComponent(email) + "&Passwd=" + encodeURIComponent(password) + "&service=finance" + "&source=" + encodeURIComponent("stockona");
        doWebRequest("POST", url, 0, parseToken, 1);
    }catch(err) {
        showError("Error while logging in: " + err, 1);
    }
}

function showError(msg, dbg) {
    console.log(msg);
    waiting.state = "hidden";

    if (dbg>0) {
        error.reason = msg;
        error.state = "shown";
    }
}

function addError(msg) {
    console.log(msg);
    error.reason = msg;
    error.state = "shown";
}

function loadAllPortfolios() {
    itemsURL = "https://finance.google.com/finance/feeds/default/portfolios";
//    console.log(itemsURL);
    loadPortfolio();
}

//////////////////////////////////////
// Loading
//////////////////////////////////////

function loadOnePosition(posFeedLink, pfoLocal, pfoIsYahoo, loadPosFromFile) {

    // activePos is not set upon first entry
    if (posFeedLink=="") {
        posFeedLink = "https://finance.google.com/finance/feeds/default/portfolios/1/positions";
    }

    // Replace http with https
    posFeedLink = posFeedLink.replace("http:", "https:");

    console.log("loadOnePosition: " + posFeedLink + " " + pfoLocal + " " + pfoIsYahoo + " " + loadPosFromFile);

    if (pfoLocal) {
        loadLocalPosition(posFeedLink, pfoIsYahoo, loadPosFromFile);
    }
    else {
        // Pull both return and transactions info
        // itemsURL = posFeedLink+"?returns=true&transactions=true";
        itemsURL = posFeedLink+"?returns=true";

        loadPosition(loadPosFromFile);
    }
}

function loadAllQuotes(quoteList) {
    itemsURL = "http://www.google.com/finance/info?client=ig&infotype=infoquoteall&q=" + quoteList;
    //console.log(itemsURL);
    try {
        waiting.state = "shown";
        doWebRequest("GET", itemsURL, 2, parseJSONAll, 3);
    } catch(err) {
        showError("Error while loading g_quote: " + err, 1);
    }
}

function loadOneQuote(symObj, idx) {
    itemsURL = "http://www.google.com/finance/info?client=ig&infotype=infoquoteall&q=" + symObj.symExg + ":" + symObj.symName;
    //console.log(itemsURL);
    loadQuote(symObj, idx);
}

function loadOnePortfolio(PortfolioName) {
    itemsURL = "https://finance.google.com/finance/feeds/default/portfolios/"+PortfolioName;
    loadPortfolio();
}

function loadPortfolio() {
    try {
        model.clear();
        waiting.state = "shown";
        doWebRequest("GET", itemsURL, 1, parsePortfolio, 1);
    } catch(err) {
        showError("Error while loading portfolio: " + err, 1);
    }
}

function loadPortfolioNum(pfoIdx) {
    try {
        waiting.state = "shown";
        doWebRequestIdx("GET", itemsURL, 1, parsePortfolioNum, pfoIdx, 1);
    } catch(err) {
        showError("Error while loading portfolio num: " + err, 1);
    }
}

function loadPosition(firstEntry) {
    try {
        if (firstEntry) { waiting.state = "shown"; }
        else            { waiting.state = "loadPosition"; }
        console.log(itemsURL);
        doWebRequest("GET", itemsURL, 1, parsePosition, 2);
    } catch(err) {
        showError("Error while loading position: " + err, 1);
    }
}

function loadQuoteHeader(symObj, idx) {
    if (idx >= pModel.count) {
        pModel.append({
                     //"idx":            idx,
                     //"id":             symObj.symId,
                     //"rtnYTD":         symObj.symRtnYTD,
                     "share":          symObj.symShare,
                     "shareCost":      symObj.shareCost,
                     "shareDayGain":   symObj.shareDayGain,
                     "shareGain":      symObj.shareGain,
                     "shareValue":     symObj.shareValue,
                     "shareGainPercent": symObj.shareGainPercent,

                     "exchange":       "-",
                     "fullName":       "-",
                     "name":           "Loading",

                     "quotePrice":     "-",

                     "quoteChgColor":  "green",
                     "quoteChg":       "-",
                     "quoteChgPtg":    "-",
                     "quoteVol":       "-",
                     "quoteAvgVol":    "-",
                     "quoteMktCap":    "-",

                     "quoteDayHi":     "-",
                     "quoteDayLo":     "-",
                     "quote52wHi":     "-",
                     "quote52wLo":     "-",

                     "quoteEps":       "-",
                     "quoteBeta":      "-",
                     "quotePe":        "-",
                     "quoteType":      "-",

                     "quoteAsk":       "-",
                     "quoteBid":       "-"
        });
    }
    else {
        pModel.set(idx, {
            "share":          symObj.symShare,
            "shareGain":      symObj.shareGain,
            "shareCost":      symObj.shareCost,
            "shareDayGain":   symObj.shareDayGain,
            "shareValue":     symObj.shareValue,
            "shareGainPercent": symObj.shareGainPercent
            //"rtnYTD":         symObj.symRtnYTD
        });
    }
}

function loadQuote(symObj, idx) {
    if (idx >= pModel.count) {
        pModel.append({
                     //"idx":            idx,
                     //"id":             symObj.symId,
                     //"rtnYTD":         symObj.symRtnYTD,
                     "share":          symObj.symShare,
                     "shareCost":      symObj.shareCost,
                     "shareDayGain":   symObj.shareDayGain,
                     "shareGain":      symObj.shareGain,
                     "shareValue":     symObj.shareValue,
                     "shareGainPercent": symObj.shareGainPercent,

                     "exchange":       "-",
                     "fullName":       "-",
                     "name":           "Loading",

                     "quotePrice":     "-",

                     "quoteChgColor":  "green",
                     "quoteChg":       "-",
                     "quoteChgPtg":    "-",
                     "quoteVol":       "-",
                     "quoteAvgVol":    "-",
                     "quoteMktCap":    "-",

                     "quoteDayHi":     "-",
                     "quoteDayLo":     "-",
                     "quote52wHi":     "-",
                     "quote52wLo":     "-",

                     "quoteEps":       "-",
                     "quoteBeta":      "-",
                     "quotePe":        "-",
                     "quoteType":      "-",

                     "quoteAsk":       "-",
                     "quoteBid":       "-"
        });
    }
    else {
        pModel.set(idx, {
            "share":          symObj.symShare,
            "shareGain":      symObj.shareGain,
            "shareCost":      symObj.shareCost,
            "shareDayGain":   symObj.shareDayGain,
            "shareValue":     symObj.shareValue,
            "shareGainPercent": symObj.shareGainPercent
            //"rtnYTD":         symObj.symRtnYTD
        });
    }

    try {
        waiting.state = "shown";
        doWebRequestIdx("GET", itemsURL, 2, parseJSON, idx, 3);
    } catch(err) {
        showError("Error while loading g_quote: " + err, 1);
    }
}

function loadYahooQuote(quoteList, isYahooQuote) {
    try {
        if (isYahooQuote)
            itemsURL =   "http://download.finance.yahoo.com/d/quotes.csv?s="+ quoteList + "&f=b2b3ghjkc6k2l1j3a2vr2e7xn";
        else
            itemsURL =   "http://download.finance.yahoo.com/d/quotes.csv?s="+ quoteList + "&f=b2b3";

        waiting.state = "shown";
        //console.log(itemsURL);
        doWebRequestIdx("GET", itemsURL, 2, parseCSV, isYahooQuote, 3);
    } catch(err) {
        showError("Error while loading y_quote: " + err, 1);
    }
}

//////////////////////////////////////
// Local
//////////////////////////////////////

function loadLocalPortfolio() {

    var fileOk = fileHandler.loadPfo();

    if (fileOk==0){
        for (var i=0; i < fileHandler.localPfoName.length; i++) {
            model.append({
                         "local": 1,
                         "name": fileHandler.localPfoName[i],
                         "feedLink": "",
                         "excerpt": fileHandler.localPfoDesc[i],
                         "isYahoo": fileHandler.localPfoIsYahoo[i],
                         "num": fileHandler.localPfoNum[i]
            });
        }
    }
    else if (fileOk==-2) {
        error.reason = "Cannot load local portfolio.";
        error.state = "shown";
    }
}

//////////////////////////////////////
// Parse Portfolio
//////////////////////////////////////

function parsePortfolio(data) {
    //try {
        //console.log("DATA: " + data);

        // 2) XML DOM
        var pfoLength = data.childNodes[8].firstChild.nodeValue;
        //console.log("pfo.length="+pfoLength);

        // <entry>
        for (var i=0; i<pfoLength; i++) {
            var pfoEtag        = data.childNodes[11+i].attributes[0].nodeValue;
            var pfoId          = data.childNodes[11+i].childNodes[0].firstChild.nodeValue;      // 1,2,3...

            var pfoName        = data.childNodes[11+i].childNodes[4].firstChild.nodeValue;      // Title
            var pfoFeedLink    = data.childNodes[11+i].childNodes[7].attributes[0].nodeValue;   // Link of position
            var currency       = data.childNodes[11+i].childNodes[8].attributes[0].nodeValue;
            var pfoGainPercent = data.childNodes[11+i].childNodes[8].attributes[1].nodeValue;
            var pfoRtnYTD      = data.childNodes[11+i].childNodes[8].attributes[9].nodeValue;

            // Push to pfoModel
            if (i<model.count) {
                model.set(i, {
                             "name": pfoName,
                             "feedLink": pfoFeedLink,
                             "excerpt": "-",
                             "num": 0
                });
            }
            else {
                //pfoCModel.append(false, false, 0, pfoName, pfoFeedLink, "-");
                model.append({
                             "local": 0,
                             "name": pfoName,
                             "feedLink": pfoFeedLink,
                             "isYahoo": false,
                             "excerpt": "-",
                             "num": 0
                });
            }

            // Retrieve number of quotes for each portfolio... this is kinda redundant but do with it for now.
            var idx = i + 1;
            itemsURL = pfoId + "/positions";
            loadPortfolioNum(i);

            /*
            console.log("<Portfolio.entry>");
            console.log("pfo.name"+pfoName);
            console.log("pfo.etag="+pfoEtag);
            console.log("pfo.Id="+pfoId);
            console.log("pfo.feedLink="+pfoFeedLink);
            console.log("currency="+currency);
            console.log("pfo.gainPercent="+pfoGainPercent);
            console.log("pfo.rtnYTD="+pfoRtnYTD);
            //*/
        }

    //}catch(err) {
    //    addError("Error: " + err);
    //}
        // Load local portfolio
        loadLocalPortfolio();

    waiting.state = "hidden";
}

function parsePortfolioNum(data, pfoIdx) {
    var posLength = data.childNodes[7].firstChild.nodeValue;
    var symName = new String("");

    for (var i=0; i< ((posLength>3) ? 3 : posLength) ; i++) {
        symName += data.childNodes[10+i].childNodes[7].attributes[2].nodeValue + ", ";
    }

    var lastComma = symName.lastIndexOf(",");
    symName = symName.substring(0, lastComma);
    symName +=  " ...";

    // Push to pfoModel
    //console.log("idx="+pfoIdx+" num="+posLength);
    if (pfoIdx<model.count) {       
        model.set(pfoIdx, {
                  "num": posLength,
                  "excerpt": symName
        });
    }
}

//////////////////////////////////////
// Parse Position
//////////////////////////////////////

function loadLocalPosition(pfoIdx, pfoIsYahoo, loadPosFromFile) {
    var fileOk = 0;
    var quoteListGoogle = "";
    var quoteListYahoo = "";

    // Only load from file when specified
    if (loadPosFromFile) {
        fileOk = fileHandler.loadPos(pfoIdx);
    }

    if (fileOk==0) {
        for (var i=0; i<fileHandler.localPosSymbol.length; i++) {
            //console.log(i + ":" + fileHandler.localPosSymbol[i]);

            var symObj = {
                "symName": fileHandler.localPosSymbol[i],
                "symExg": fileHandler.localPosExg[i],
                "symShare": fileHandler.localPosShare[i],
                "shareCost": fileHandler.localPosCost[i],
                "shareDayGain": 0,
                "shareGain": 0,
                "shareValue": 0,
                "shareGainPercent": 0
            };

            if (!pfoIsYahoo) {
                loadQuoteHeader(symObj, i);
            }

            if (symObj.symExg=="") {
                quoteListGoogle = quoteListGoogle + "," + symObj.symName;
            }
            else {
                quoteListGoogle = quoteListGoogle + "," + symObj.symName + ":" + symObj.symExg;
            }

            // Convert ^DJI to INDU to bypass yahoo limitation. Alternatively can use YQL but requires re-write.
            if (symObj.symName=="^DJI") {
                quoteListYahoo  = quoteListYahoo + "+" + "INDU";
            }
            else {
                quoteListYahoo  = quoteListYahoo + "+" + convertExchangeForYahoo(symObj.symName, symObj.symExg);
            }
        }

        // Take out leading +
        quoteListGoogle = quoteListGoogle.replace(/^\,/, "");
        quoteListYahoo = quoteListYahoo.replace(/^\+/, "");

        if (!pfoIsYahoo) {
            //console.log(quoteListGoogle);
            loadAllQuotes(quoteListGoogle);
        }

        loadYahooQuote(quoteListYahoo, pfoIsYahoo);
    }
    else if (fileOk==-2){
        error.state = "shown";
        error.reason = "Cannot load position file " + pfoIdx + ".pos";
    }

    waiting.state = "hidden";
}

function parsePosition(data) {
    //try {
        var posLength = data.childNodes[7].firstChild.nodeValue;

        console.log("pos.length="+posLength);
        var quoteListGoogle = "";
        var quoteListYahoo = "";

        // Improved error handling
        if (posLength==0) {
            showError("Cannot load portfolio.\n\nCheck if you have portfolio selected.\nCreate one on Google if you don't have any.", 1);
            errorState.state = "promotePfo";
        }
        else {
            for (var i=0; i<posLength; i++) {
                var symObj = {
                "shareGainPercent" : sprintf("%.2f", 100*data.childNodes[10+i].childNodes[6].attributes[0].nodeValue), // Same as returnOverall
                "symRtnYTD"      : sprintf("%.2f", 100*data.childNodes[10+i].childNodes[6].attributes[8].nodeValue),
                "symShare"       : data.childNodes[10+i].childNodes[6].attributes[9].nodeValue,
                "symExg"         : data.childNodes[10+i].childNodes[7].attributes[0].nodeValue,
                "symName"        : data.childNodes[10+i].childNodes[7].attributes[2].nodeValue,

                "shareCost"     : "0",
                "shareValue"    : "0",
                "shareDayGain"  : "0",
                "shareGain"     : "0"

                // Don't parse for now...
                //"symEtag"        : data.childNodes[10+i].attributes[0].nodeValue,
                //"symFeedLink"    : data.childNodes[10+i].childNodes[5].attributes[0].nodeValue,   // transaction link
                //"symFullName"    : data.childNodes[10+i].childNodes[7].attributes[1].nodeValue,
            };

            // costBasis, gain, mktValue. Only those who have TX records are available
            if (data.childNodes[10+i].childNodes[6].childNodes.length==4) {
                symObj.shareCost    = data.childNodes[10+i].childNodes[6].childNodes[0].firstChild.attributes[0].nodeValue;
                symObj.shareDayGain = data.childNodes[10+i].childNodes[6].childNodes[1].firstChild.attributes[0].nodeValue;
                symObj.shareGain   = data.childNodes[10+i].childNodes[6].childNodes[2].firstChild.attributes[0].nodeValue;
                symObj.shareValue   = data.childNodes[10+i].childNodes[6].childNodes[3].firstChild.attributes[0].nodeValue;
            }

            //console.log(symObj.symName + " cost=" + symObj.shareCost + " value=" + symObj.shareValue + " gain=" + symObj.shareGain);

            // JSON Parsing, parsed data stored in global jsonObj.
            // Push needs to be done in parseJSON to keep the program synchronous.
            loadQuoteHeader(symObj, i);
            quoteListGoogle = quoteListGoogle + "," + symObj.symName + ":" + symObj.symExg;
            quoteListYahoo  = quoteListYahoo + "+" + convertExchangeForYahoo(symObj.symName, symObj.symExg);

            /*
            console.log("<position.entry>");
            //console.log("pos.etag="+symObj.symEtag);
            console.log("pos.id="+symObj.symId);
            console.log("pos.feedLink="+symObj.symFeedLink);
            console.log("pos.gainPercent="+symObj.shareGainPercent);
            console.log("pos.rtnYTD="+symObj.symRtnYTD);
            console.log("pos.share="+symObj.symShare);
            console.log("pos.exchange="+symObj.symExg);
            console.log("pos.fullName="+symObj.symFullName);
            console.log("pos.name="+symObj.symName);
            //*/
        }

        // Take out leading +
        quoteListGoogle = quoteListGoogle.replace(/^\,/, "");
        quoteListYahoo = quoteListYahoo.replace(/^\+/, "");

        loadAllQuotes(quoteListGoogle);
        loadYahooQuote(quoteListYahoo, 0);

    //}catch(err) {
    //    addError("Error: " + err);
    //}
    waiting.state = "hidden";
        }
}

// Parse Yahoo feed
function parseCSV(data, isYahooQuote) {
    //console.log(data);
    var csvArray = fileHandler.parseCSV(data);
    //console.log(csvArray);
    //console.log("isYahoo="+isYahooQuote+"");

    // Array[count*i+j]
    //var csvObj = CSVToArray(data);

//    if ( csvArray.length != ( pModel.count*2 ))  {
//        console.log("yahoo feed info not shown: " + csvArray.length + ":" + pModel.count);
//        waiting.state = "hidden";
//        return;
//    }
//    else {
        // b2, b3: ask, bid (real-time)
        if (!isYahooQuote) {
            if ( csvArray.length != ( pModel.count*2 ))  {
                console.log("yahoo feed info not shown: " + csvArray.length + ":" + pModel.count);
                waiting.state = "hidden";
                return;
            }
            else {
                for (var i=0; i<pModel.count; i++) {
                    //console.log( i +":(Ask,Bid) = "+csvArray[2*i+0] + "/" + csvArray[2*i+1]);
                    pModel.set(i, {
                               "quoteAsk": (csvArray[2*i+0]=="N/A"||csvArray[2*i+0]=="") ? "-" : csvArray[2*i+0],
                               "quoteBid": (csvArray[2*i+1]=="N/A"||csvArray[2*i+1]=="") ? "-" : csvArray[2*i+1]
                    });
                }
            }
        }
        // b2, b3: ask, bid (real-time)
        // g,h,j,k (day lo, day hi, 52w lo 52w hi)
        // c6: change, k2: change percentage
        // l1: last trade price
        // j3: market cap
        // a2: avg volume (3m), v: volume (real-time)
        // r2: PE (real-time), r: PE
        // e7: EPS Estimate Current Year
        // x: stock exchange
        // n: name
        //itemsURL =   "http://download.finance.yahoo.com/d/quotes.csv?s="+ quoteList + "&f=b2b3ghjkc6k2l1j3a2vr2e7xn";
        else {
             for (var i=0; i<fileHandler.localPosSymbol.length; i=i+1) {
                 var offset = 16;
                 //var change = csvObj[i][7];
                 var change = csvArray[offset*i+6];
                 var changePtg = csvArray[offset*i+7];

                 //console.log("DBG: " + fileHandler.localPosSymbol[i] + ":" + change + ":" + changePtg);

                 var quoteChgColor;

                 if (change.indexOf("+")==0) {
                     quoteChgColor = (quoteChgColorMode) ? "red" : "green";
                 }
                 else {
                     quoteChgColor = (quoteChgColorMode) ? "green" : "red";
                 }

                 // Split change percentage
                 changePtg = changePtg.split(" - ");
                 changePtg[1] = changePtg[1].substr(0, changePtg[1].indexOf("%"));

                 // C function to compute value, gain, day-gain, and gain percentage.
                 var perfData = "";

                 //if (csvObj[i][8]!="N/A") {
                 if (csvArray[offset*i+8]!="N/A") {
                     perfData = fileHandler.calcPerformance(i, csvArray[offset*i+8]);
                 }

                 if (pModel.count==0) {
                        pModel.append({
                             "exchange":   (csvArray[offset*i+14]=="N/A") ? "-" : csvArray[offset*i+14],
                             "fullName":   (csvArray[offset*i+15]=="N/A") ? "-" : csvArray[offset*i+15],
                             "name":         fileHandler.localPosSymbol[i],
                             "share":        fileHandler.localPosShare[i],
                             "shareGain":    perfData[0],
                             "shareCost":    perfData[1],
                             "shareDayGain": "-",
                             "shareValue":   perfData[2],
                             "shareGainPercent": perfData[3],

                             "quoteAsk":   (csvArray[offset*i+0]=="N/A") ? "-" : csvArray[offset*i+0],
                             "quoteBid":   (csvArray[offset*i+1]=="N/A") ? "-" : csvArray[offset*i+1],
                             "quotePrice": (csvArray[offset*i+8]=="N/A") ? "-" : csvArray[offset*i+8],
                             "quoteChgColor": quoteChgColor,

                             "quoteChg":   (change=="N/A") ? "-" : change,
                             "quoteChgPtg":(changePtg[1]=="N/A") ? "-" : changePtg[1],
                             "quoteVol":   (csvArray[offset*i+11]=="N/A") ? "-" : csvArray[offset*i+11],
                             "quoteAvgVol":(csvArray[offset*i+10]=="N/A") ? "-" : csvArray[offset*i+10],
                             "quoteMktCap":(csvArray[offset*i+9]=="N/A") ? "-" : csvArray[offset*i+9],
                             "quoteDayHi": (csvArray[offset*i+3]=="N/A") ? "-" : csvArray[offset*i+3],
                             "quoteDayLo": (csvArray[offset*i+2]=="N/A") ? "-" : csvArray[offset*i+2],
                             "quote52wHi": (csvArray[offset*i+5]=="N/A") ? "-" : csvArray[offset*i+5],
                             "quote52wLo": (csvArray[offset*i+4]=="N/A") ? "-" : csvArray[offset*i+4],
                             "quoteEps":   (csvArray[offset*i+13]=="N/A") ? "-" : csvArray[offset*i+13],
                             "quoteBeta":   "-",
                             "quotePe":    (csvArray[offset*i+12]=="N/A") ? "-" : csvArray[offset*i+12],
                             "quoteType":   "-"
                        });
                 }
                 else {
                     pModel.set(i, {
                                "exchange":   (csvArray[offset*i+14]=="N/A") ? "-" : csvArray[offset*i+14],
                                "fullName":   (csvArray[offset*i+15]=="N/A") ? "-" : csvArray[offset*i+15],
                                "name":         fileHandler.localPosSymbol[i],
                                "share":        fileHandler.localPosShare[i],
                                "shareGain":    0,
                                "shareCost":    fileHandler.localPosCost[i],
                                "shareDayGain": 0,
                                "shareValue":   0,
                                "shareGainPercent": 0,

                                "quoteAsk":   (csvArray[offset*i+0]=="N/A") ? "-" : csvArray[offset*i+0],
                                "quoteBid":   (csvArray[offset*i+1]=="N/A") ? "-" : csvArray[offset*i+1],
                                "quotePrice": (csvArray[offset*i+8]=="N/A") ? "-" : csvArray[offset*i+8],
                                "quoteChgColor": quoteChgColor,

                                "quoteChg":   (change=="N/A") ? "-" : change,
                                "quoteChgPtg":(changePtg[1]=="N/A") ? "-" : changePtg[1],
                                "quoteVol":   (csvArray[offset*i+11]=="N/A") ? "-" : csvArray[offset*i+11],
                                "quoteAvgVol":(csvArray[offset*i+10]=="N/A") ? "-" : csvArray[offset*i+10],
                                "quoteMktCap":(csvArray[offset*i+9]=="N/A") ? "-" : csvArray[offset*i+9],
                                "quoteDayHi": (csvArray[offset*i+3]=="N/A") ? "-" : csvArray[offset*i+3],
                                "quoteDayLo": (csvArray[offset*i+2]=="N/A") ? "-" : csvArray[offset*i+2],
                                "quote52wHi": (csvArray[offset*i+5]=="N/A") ? "-" : csvArray[offset*i+5],
                                "quote52wLo": (csvArray[offset*i+4]=="N/A") ? "-" : csvArray[offset*i+4],
                                "quoteEps":   (csvArray[offset*i+13]=="N/A") ? "-" : csvArray[offset*i+13],
                                "quoteBeta":   "-",
                                "quotePe":    (csvArray[offset*i+12]=="N/A") ? "-" : csvArray[offset*i+12],
                                "quoteType":   "-"
                     });
                 }
             }
        }
//    }
}

function parseJSON(data, idx) {
    // Remove comment // at the beginning
    //data = data.replace(/\/\//, "");
    data = removeComment(data);
    //console.log(data);

    // eval is unsafe, but JSON.parse is prone to fail
    var jsonObj = eval("(" + data + ")");
    //var jsonObj = JSON.parse(data);

    if (jsonObj==null || typeof(jsonObj)==undefined) {
        waiting.state = "hidden";
        return;
    }

    var quoteChgColor;
    //var chgSymbol = jsonObj[0].c.match(/^\-/);
    if (jsonObj[0].c.indexOf("-")==0) { //chgSymbol != null) {
        quoteChgColor = (quoteChgColorMode) ? "green" : "red";
    }
    else {
        quoteChgColor = (quoteChgColorMode) ? "red" : "green";
    }

    // Check if entry already exists, if not append, if yes set.
    // Use set method to avoid flickering when updating content by timer
    //if (idx<pModel.count) {

        pModel.set(idx, {
                     "exchange":       (jsonObj[0].e   =="") ? "-" : jsonObj[0].e   ,    //symObj.symExg,
                     "fullName":       (jsonObj[0].name=="") ? "-" : jsonObj[0].name, //symObj.symFullName,
                     "name":           (jsonObj[0].t   =="") ? "-" : jsonObj[0].t   ,    //symObj.symName,
                     "quotePrice":     (jsonObj[0].l   =="") ? "-" : jsonObj[0].l   ,
                     "quoteChgColor":  (quoteChgColor  =="") ? "-" : quoteChgColor  ,
                     "quoteChg":       (jsonObj[0].c   =="") ? "-" : jsonObj[0].c   ,
                     "quoteChgPtg":    (jsonObj[0].cp  =="") ? "-" : jsonObj[0].cp  ,
                     "quoteVol":       (jsonObj[0].vo  =="") ? "-" : jsonObj[0].vo  ,
                     "quoteAvgVol":    (jsonObj[0].avvo=="") ? "-" : jsonObj[0].avvo,
                     "quoteMktCap":    (jsonObj[0].mc  =="") ? "-" : jsonObj[0].mc  ,
                     "quoteDayHi":     (jsonObj[0].hi  =="") ? "-" : jsonObj[0].hi  ,
                     "quoteDayLo":     (jsonObj[0].lo  =="") ? "-" : jsonObj[0].lo  ,
                     "quote52wHi":     (jsonObj[0].hi52=="") ? "-" : jsonObj[0].hi52,
                     "quote52wLo":     (jsonObj[0].lo52=="") ? "-" : jsonObj[0].lo52,
                     "quoteEps":       (jsonObj[0].eps =="") ? "-" : jsonObj[0].eps ,
                     "quoteBeta":      (jsonObj[0].beta=="") ? "-" : jsonObj[0].beta,
                     "quotePe":        (jsonObj[0].pe  =="") ? "-" : jsonObj[0].pe  ,
                     "quoteType":      (jsonObj[0].type=="") ? "-" : jsonObj[0].type
        });
    //}

    waiting.state = "hidden";
}

function parseJSONAll(data) {
    // Remove comment // at the beginning
    //data = data.replace(/\/\//, "");
    data = removeComment(data);
    //console.log(data);

    // eval is unsafe, but JSON.parse is prone to fail
    var jsonObj = eval("(" + data + ")");
    //var jsonObj = JSON.parse(data);

    if (jsonObj==null || typeof(jsonObj)==undefined) {
        waiting.state = "hidden";
        return;
    }

    for (var i=0; i<pModel.count; i++) {

        var quoteChgColor;
        //var chgSymbol = jsonObj[0].c.match(/^\-/);
        if (jsonObj[i].c.indexOf("-")==0) { //chgSymbol != null) {
            quoteChgColor = "red";
        }
        else {
            quoteChgColor = "green";
        }

        pModel.set(i, {
                     "exchange":       (jsonObj[i].e   =="") ? "-" : jsonObj[i].e   ,    //symObj.symExg,
                     "fullName":       (jsonObj[i].name=="") ? "-" : jsonObj[i].name, //symObj.symFullName,
                     "name":           (jsonObj[i].t   =="") ? "-" : jsonObj[i].t   ,    //symObj.symName,
                     "quotePrice":     (jsonObj[i].l   =="") ? "-" : jsonObj[i].l   ,
                     "quoteChgColor":  (quoteChgColor  =="") ? "-" : quoteChgColor  ,
                     "quoteChg":       (jsonObj[i].c   =="") ? "-" : jsonObj[i].c   ,
                     "quoteChgPtg":    (jsonObj[i].cp  =="") ? "-" : jsonObj[i].cp  ,
                     "quoteVol":       (jsonObj[i].vo  =="") ? "-" : jsonObj[i].vo  ,
                     "quoteAvgVol":    (jsonObj[i].avvo=="") ? "-" : jsonObj[i].avvo,
                     "quoteMktCap":    (jsonObj[i].mc  =="") ? "-" : jsonObj[i].mc  ,
                     "quoteDayHi":     (jsonObj[i].hi  =="") ? "-" : jsonObj[i].hi  ,
                     "quoteDayLo":     (jsonObj[i].lo  =="") ? "-" : jsonObj[i].lo  ,
                     "quote52wHi":     (jsonObj[i].hi52=="") ? "-" : jsonObj[i].hi52,
                     "quote52wLo":     (jsonObj[i].lo52=="") ? "-" : jsonObj[i].lo52,
                     "quoteEps":       (jsonObj[i].eps =="") ? "-" : jsonObj[i].eps ,
                     "quoteBeta":      (jsonObj[i].beta=="") ? "-" : jsonObj[i].beta,
                     "quotePe":        (jsonObj[i].pe  =="") ? "-" : jsonObj[i].pe  ,
                     "quoteType":      (jsonObj[i].type=="") ? "-" : jsonObj[i].type
        });
    }

    waiting.state = "hidden";
}

//////////////////////////////////////
// Misc
//////////////////////////////////////

// Yahoo return 0 for ask/bid for non-North American exchange
function convertExchangeForYahoo (name, exg) {
    // No need to do this replacement, Yahoo quote doesn't understand.
//    if (name.indexOf(".")==0)
//        return name.replace(".", "^");

    //console.log(name + exg);
    if (exg=="NYSE" || exg=="NASDAQ") { return name; }
    else if (exg=="OTC")              { return (name + "." + "OB"); }
    else if (exg=="AMEX")             { return (name + "." + "OB"); }
    else if (exg=="TSXV")             { return (name + "." + "V");  }
    else if (exg=="TSX")              { return (name + "." + "TO"); }
    else if (exg=="LON")              { return (name + "." + "L"); }
    // Australia
    else if (exg=="ASX")              { return (name + "." + "AX"); }
    // Asian
    else if (exg=="TPE")              { return (name + "." + "TW"); }
    else if (exg=="HKG")              { return (name + "." + "HK"); }
    else if (exg=="SHA")              { return (name + "." + "SS"); }
    else if (exg=="SHE")              { return (name + "." + "SZ"); }
    else if (exg=="SEO")              { return (name + "." + "KS"); }
    else                              { return name; }
}

function showData(data) {
    console.log("DONE: " + data);
}

function getNode(node, name) {
    for(var i=0; i<node.childNodes.length; i++) {
        var nodeName = node.childNodes[i].nodeName;
        if(nodeName==name) {
            return node.childNodes[i].firstChild;
        }
    }

    return node;
}

function cleanModel(model) {
    model.clear();
}

function removeComment(data) {
    data = data.replace(/\/\//, "");
    return data;
}

// function from http://forums.devshed.com/t39065/s84ded709f924610aa44fff827511aba3.html
// author appears to be Robert Pollard
function sprintf()
{
   if (!arguments || arguments.length < 1 || !RegExp)
   {
      return;
   }
   var str = arguments[0];
   var re = /([^%]*)%('.|0|\x20)?(-)?(\d+)?(\.\d+)?(%|b|c|d|u|f|o|s|x|X)(.*)/;
   var a = [];
   var b = [];
   var numSubstitutions = 0;
   var numMatches = 0;
   while (a = re.exec(str))
   {
      var leftpart = a[1], pPad = a[2], pJustify = a[3], pMinLength = a[4];
      var pPrecision = a[5], pType = a[6], rightPart = a[7];

      numMatches++;
      if (pType == '%')
      {
         subst = '%';
      }
      else
      {
         numSubstitutions++;
         if (numSubstitutions >= arguments.length)
         {
            alert('Error! Not enough function arguments (' + (arguments.length - 1)
               + ', excluding the string)\n'
               + 'for the number of substitution parameters in string ('
               + numSubstitutions + ' so far).');
         }
         var param = arguments[numSubstitutions];
         var pad = '';
                if (pPad && pPad.substr(0,1) == "'") pad = leftpart.substr(1,1);
           else if (pPad) pad = pPad;
         var justifyRight = true;
                if (pJustify && pJustify === "-") justifyRight = false;
         var minLength = -1;
                if (pMinLength) minLength = parseInt(pMinLength);
         var precision = -1;
                if (pPrecision && pType == 'f')
                   precision = parseInt(pPrecision.substring(1));
         var subst = param;
         switch (pType)
         {
         case 'b':
            subst = parseInt(param).toString(2);
            break;
         case 'c':
            subst = String.fromCharCode(parseInt(param));
            break;
         case 'd':
            subst = parseInt(param) ? parseInt(param) : 0;
            break;
         case 'u':
            subst = Math.abs(param);
            break;
         case 'f':
            subst = (precision > -1)
             ? Math.round(parseFloat(param) * Math.pow(10, precision))
              / Math.pow(10, precision)
             : parseFloat(param);
            break;
         case 'o':
            subst = parseInt(param).toString(8);
            break;
         case 's':
            subst = param;
            break;
         case 'x':
            subst = ('' + parseInt(param).toString(16)).toLowerCase();
            break;
         case 'X':
            subst = ('' + parseInt(param).toString(16)).toUpperCase();
            break;
         }
         var padLeft = minLength - subst.toString().length;
         if (padLeft > 0)
         {
            var arrTmp = new Array(padLeft+1);
            var padding = arrTmp.join(pad?pad:" ");
         }
         else
         {
            var padding = "";
         }
      }
      str = leftpart + padding + subst + rightPart;
   }
   return str;
}

/*
function getNodeValue(node, name) {
    var nodeValue = "";
    for(var i=0; i<node.childNodes.length; i++) {
        var nodeName = node.childNodes[i].nodeName;
        if(nodeName==name) {
            nodeValue = node.childNodes[i].firstChild.nodeValue;
        }
    }
    return nodeValue;
}

function formatNumber(data) {
    // > 1M: Display M
    // > 10k: Display K
    var tmp;
    if (Number(data)>1000000) {
        tmp = sprintf("%.2f", Number(data)/1000000);
        return tmp + "m";
    }
    else if (Number(data)>10000) {
        tmp = sprintf("%.2f", Number(data)/10000);
        return tmp + "k";
    }
    else {
        return sprintf("%.2f", Number(data));
    }
}

function removeLinks(original) {
    var txt = original;
    txt = txt.replace(/<a /g, "<span ");
    txt = txt.replace(/<\/a>/g, "</span>");
    return txt;
}

function prettyDate(date){
    try {
        var diff = (((new Date()).getTime() - date.getTime()) / 1000);
        var day_diff = Math.floor(diff / 86400);

        if ( isNaN(day_diff) || day_diff >= 31 ) {
            //console.log("Days: " + day_diff);
            return "some time ago";
        } else if (day_diff < 0) {
            //console.log("day_diff: " + day_diff);
            return "just now";
        }

        return day_diff == 0 && (
                    diff < 60 && "just now" ||
                    diff < 120 && "1 minute ago" ||
                    diff < 3600 && Math.floor( diff / 60 ) + " min ago" ||
                    diff < 7200 && "1 hour ago" ||
                    diff < 86400 && Math.floor( diff / 3600 ) + " hours ago") ||
                day_diff == 1 && "Yesterday" ||
                day_diff < 7 && day_diff + " days ago" ||
                day_diff < 31 && Math.ceil( day_diff / 7 ) + " weeks ago";
        day_diff >= 31 && Math.ceil( day_diff / 30 ) + " months ago";
    } catch(err) {
        console.log("Error: " + err);
        return "some time ago";
    }
}

// 2011-01-24T18:48:00Z
function parseDate(stamp)
{
    try {
        //console.log("stamp: " + stamp);
        var parts = stamp.split("T");
        var day;
        var time;
        var hours;
        var minutes;
        var seconds = 0;
        var year;
        var month;

        var dates = parts[0].split("-");
        year = parseInt(dates[0]);
        month = parseInt(dates[1])-1;
        day = parseInt(dates[2]);

        var times = parts[1].split(":");
        hours = parseInt(times[0]);
        minutes = parseInt(times[1]);

        var dt = new Date();
        dt.setUTCDate(day);
        dt.setYear(year);
        dt.setUTCMonth(month);
        dt.setUTCHours(hours);
        dt.setUTCMinutes(minutes);
        dt.setUTCSeconds(seconds);

        //console.log("day: " + day + " year: " + year + " month " + month + " hour " + hours);

        return dt;
    } catch(err) {
        console.log("Error while parsing date: " + err);
        return new Date();
    }
}


*/
