//-----------------------------------------------------------------------------
// $Id: squeezster.cpp 111 2011-12-04 08:38:13Z pakarlss $
//-----------------------------------------------------------------------------

#include "squeezster.h"
#include "ui_squeezster.h"
#include "ui_connectiondialog.h"
#include "ui_optionsdialog.h"
#include "ui_libraryviewdialog.h"
#include "squeezster_ui_definitions.h"

//----------------------------------------------
//----------------------------------------------
// squeezster(QWidget *parent)
// Main window
//----------------------------------------------
//----------------------------------------------
squeezster::squeezster(int debug,QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::squeezster)
{
    ui->setupUi(this);
    setDebugFlag(debug);
    core.setDebugFlag(debug);
#if defined(Q_WS_MAEMO_5) || defined(MEEGO_EDITION_HARMATTAN)
    setAttribute(Qt::WA_Maemo5AutoOrientation);
    grabZoomKeys(true);
    connect(this,SIGNAL(volumeUp()),this,SLOT(increaseVolume()));
    connect(this,SIGNAL(volumeDown()),this,SLOT(decreaseVolume()));
#endif
    //-------------------------------------------------
    // Do initialization
    //-------------------------------------------------

    // Init timers:
    progressTimer.setSingleShot(false);
    progressTimer.setInterval(PROGRESSINTERVAL);
    itemSelectTimer.setSingleShot(true);
    itemSelectTimer.setInterval(ITEMSELECTTIME);

    pixmapBusy = false;
    loadAlbumsBusy = false;
    albumsInitDone = false;
    musicFolderInitDone = false;

    // Setup signals & slots:
    connect(&progressTimer,SIGNAL(timeout()),this,SLOT(updateProgress()));
    connect(&itemSelectTimer,SIGNAL(timeout()),this,SLOT(itemUnselect()));
    connect(&core,SIGNAL(nowPlayingListChanged()),this,SLOT(nowPlayingListChangedSlot()));
    connect(&core,SIGNAL(albumsListChanged()),this,SLOT(albumsListChangedSlot()));
    connect(&core,SIGNAL(radiosListChanged()),this,SLOT(radiosListChangedSlot()));
    connect(&core,SIGNAL(appsListChanged()),this,SLOT(appsListChangedSlot()));
    connect(&core,SIGNAL(listListChanged()),this,SLOT(listListChangedSlot()));
    connect(&core,SIGNAL(treeListChanged()),this,SLOT(treeListChangedSlot()));
    connect(&core,SIGNAL(playersListChanged()),this,SLOT(playersListChangedSlot()));
    connect(&core,SIGNAL(serverListChanged()),this,SLOT(serverListChangedSlot()));
    connect(&core,SIGNAL(playerStatusChanged()),this,SLOT(updateStatus()));
    connect(&core,SIGNAL(activePlayerIndexChanged()),this,SLOT(activePlayerIndexChangedSlot()));
    connect(&core,SIGNAL(albumartUrlChanged()),this,SLOT(updateAlbumArtLabel()));

    //    connect(&core,SIGNAL(networkError()),this,SLOT(constructConnDialog()));
    //    connect(&core,SIGNAL(networkError()),this,SLOT(displayNetworkError()));
    connect(&core,SIGNAL(mailboxChanged()),this,SLOT(displayMessage()));
    //-------------------------------------------------------------------------

    // Setup the albums view:
    setAlbumsListView();

    // Create UI:
    initUI();
    // Start communication:
    _DEBUG_ << "Start core:";
    core.start();
}

//----------------------------------------------
// squeezster::updateStatus()
//----------------------------------------------
void squeezster::updateStatus() {
    _DEBUG_ << "updateStatus() Player" << core.playerStatus()[P_NAME].toString();

    playerPower=core.playerPower();
    playerPlay=core.playerPlay();
    playerPause=core.playerPause();
    playerStop=core.playerStop();
    playerShuffle=core.playerShuffle();
    playerRepeat=core.playerRepeat();
    currentSong=core.currentSong();

    ui->volumeSlider->setValue(core.playerVolume());
    ui->progressBar->setMaximum((core.playerDuration()?core.playerDuration():3600));
    ui->progressBar->setValue(core.playerTime());

    if(!core.playerStatus()[P_LOOP].toList().isEmpty())
        core.setCurrentSong(core.playerStatus()[P_LOOP].toList().first().toMap());

    _DEBUG_ << "updateStatus" << "playlistChanged:" << (core.playerStatus()[P_TIMESTAMP].toDouble() - core.playlistTimeStamp())  << core.playerChanged();
    if(core.playlistChanged() || core.playerChanged()) {
        _DEBUG_ << "updateStatus" << "playlistChanged! Updating";
        requestList("status",core.activePlayer());
        core.setPlayerChanged(false);
        core.setPlaylistTimeStamp(core.playerStatus()[P_TIMESTAMP].toDouble());
    }
    //---------------------------------------------
    // If valid album art, queue http request
    //---------------------------------------------
    _DEBUG_ << "Remote" << currentSong[SONG_REMOTE] << "CoverID:"<< currentSong[SONG_COVERID] << "CoverArt" << currentSong[SONG_COVERART];
    QString newUrl=currentSong[SONG_ARTWORK_URL].toString();

    if (!currentSong[SONG_REMOTE].toBool()) { // Local
        newUrl=core.serverHttpAddress();
        newUrl+="/music/";
        newUrl+=currentSong[SONG_COVERID].toString();
        newUrl+=LOCALALBUMARTSUFFIX;
    } else if(!newUrl.isEmpty()){ // Remote
        if(!newUrl.contains("http:")) {
            newUrl=QString(core.serverHttpAddress()+"/"+newUrl);
        }
    }

    if(newUrl!=core.albumartUrl()) {
        _DEBUG_ << "Setting albumart URL" << newUrl;
        core.setAlbumartUrl(newUrl);
    } else {
        _DEBUG_ << "Keeping albumart URL" << newUrl;
    }
    updateUI();
}

//----------------------------------------------
// squeezster::updateUI()
// Update the UI to reflect the player status
//----------------------------------------------
void squeezster::updateUI()
{
    _DEBUG_ << "updateUI()";

    QFontMetrics fontMetrics = ui->titleLabel->fontMetrics();
    ui->titleLabel->setText(fontMetrics.elidedText(currentSong.value("title").toByteArray(),Qt::ElideMiddle,ui->titleLabel->width()));
    fontMetrics = ui->artistLabel->fontMetrics();
    ui->artistLabel->setText(fontMetrics.elidedText(currentSong.value("artist").toByteArray(),Qt::ElideMiddle,ui->artistLabel->width()));
    fontMetrics = ui->albumLabel->fontMetrics();
    ui->albumLabel->setText(fontMetrics.elidedText(currentSong.value("album").toByteArray(),Qt::ElideMiddle,ui->albumLabel->width()));

    //---------------------------------------------
    // Update Icons to reflect status
    //---------------------------------------------
    if (playerPower)
        ui->powerButton->setIcon(QPixmap(POWERGREENPIXMAPFILE));
    else
        ui->powerButton->setIcon(QPixmap(POWERREDPIXMAPFILE));
    updateNowPlayingIcons();

    // Menu text for shuffle
    if(!playerShuffle) {
        ui->actionShuffle->setText("Shuffle: Turn ON");
        ui->actionShuffle->setChecked(false);
    } else {
        ui->actionShuffle->setText("Shuffle: Turn OFF");
        ui->actionShuffle->setChecked(true);
    }
    // Menu text for repeat
    if(!playerRepeat) {
        ui->actionRepeat->setText("Repeat: Turn ON");
        ui->actionRepeat->setChecked(false);
    } else {
        ui->actionRepeat->setText("Repeat: Turn OFF");
        ui->actionRepeat->setChecked(true);
    }


}

void squeezster::updateAlbumArtLabel() {
    _DEBUG_ << "updateAlbumArtLabel: Starting to download" << core.albumartUrl();
    downloadImage(ui->albumArtLabel,core.albumartUrl());
}

//----------------------------------------------
// squeezster::updateNowPlayingIcons()
// Update status icons of the current song
//----------------------------------------------
void squeezster::updateNowPlayingIcons()
{
    _DEBUG_ << "squeezster::updateNowPlayingIcons()";
    int i;

    // Set previous song icon to bullet:
    for(i=0; i<ui->nowPlayingListWidget->count(); i++) {
        ui->nowPlayingListWidget->item(i)->setIcon(smallBulletIcon);
    }
    // Update current song icon:
    if (ui->nowPlayingListWidget->count()>core.playerSongIndex())
    {
        if (playerPower) {
            if (playerPlay)
                ui->nowPlayingListWidget->item(core.playerSongIndex())->setIcon(playIcon);
            else if (playerPause)
                ui->nowPlayingListWidget->item(core.playerSongIndex())->setIcon(pauseIcon);
            else if (playerStop)
                ui->nowPlayingListWidget->item(core.playerSongIndex())->setIcon(stopIcon);
        } else {
            ui->nowPlayingListWidget->item(core.playerSongIndex())->setIcon(QIcon(POWERREDPIXMAPFILE));
        }
        //ui->nowPlayingListWidget->scrollToItem(ui->nowPlayingListWidget->item(core.playerSongIndex()));
    }
}

//----------------------------------------------
// squeezster::displayError()
// Present connection errors
//----------------------------------------------
void squeezster::displayError(QAbstractSocket::SocketError socketError)
{
    switch (socketError) {
    case QAbstractSocket::RemoteHostClosedError:
        break;
    case QAbstractSocket::HostNotFoundError:
        QMessageBox::information(this, tr("Squeezster"),
                                 tr("The host was not found. Please check the "
                                    "host name and port settings."));
        constructConnDialog();
        break;
    case QAbstractSocket::ConnectionRefusedError:
        QMessageBox::information(this, tr("Squeezster"),
                                 tr("The connection was refused by the peer. "
                                    "Make sure the server is running, "
                                    "and check that the host name and port "
                                    "settings are correct."));
        break;
    default:;
    }
}

//----------------------------------------------
// squeezster::displayMessage()
// Present core messages
//----------------------------------------------
void squeezster::displayMessage()
{
    QMessageBox::information(this, tr("Squeezster"),core.mailbox());
}

//----------------------------------------------
// squeezster::updateProgress()
//----------------------------------------------
void squeezster::updateProgress()
{
    quint16 time;
    time = ui->progressBar->value()+PROGRESSINTERVAL/1000;
    //qDebug() << "updateProgress()" << "New Time:" << QString::number(time) << "Max Time:" << QString::number(ui->progressBar->maximum());
    if ((time <= ui->progressBar->maximum()) && playerPlay) {
        ui->progressBar->setValue(time);
    } else if(playerStop){
        ui->progressBar->setValue(1);
    }
}

//----------------------------------------------
// squeezster::downloadImage()
// Issues a http download request of an image
//----------------------------------------------
void squeezster::downloadImage(MapTreeItem* item) {
    QString url = item->property("image");
    if(url.isEmpty())
        url = item->property("icon");
    if(!url.isEmpty()) {
        if(!url.startsWith("http")) // E.g. radio
            url=QString(core.serverHttpAddress() +"/" + url);
        //_DEBUG_ << "downloadImage(): Starting to download" << url;
        treeWidgetItemNetworkRequest treeWidgetItemRequest(item);
        treeWidgetItemRequest.setUrl(url);
        QNetworkReply* reply = core.networkManager()->get(treeWidgetItemRequest);
        connect(reply,SIGNAL(finished()),this,SLOT(pixmapDownloaded()));
    } else
        item->setIcon();
}

//----------------------------------------------
// squeezster::downloadImage()
//----------------------------------------------
void squeezster::downloadAlbumImage(MapListItem* item)
{
    //_DEBUG_ << "downloadAlbumImage()";
    MapListItem* mapItem = static_cast<MapListItem*>(item);
    if(!QString(mapItem->property("artwork_track_id")).isEmpty()) {
        downloadAlbumImage(item,mapItem->property("artwork_track_id"));
        //_DEBUG_ << "downloadImage(): Starting to download artwork_id:" << mapItem->property("artwork_track_id");
    }
    else {
        item->setIcon(emptyAlbumArtIcon);
    }
}

//----------------------------------------------
// squeezster::downloadImage()
//----------------------------------------------
void squeezster::downloadAlbumImage(MapListItem* item, QString id)
{
    //   _DEBUG_ << "downloadAlbumImage()";
    QString url =QString(core.serverHttpAddress()+"/music/"+id+albumartsuffix);
    //_DEBUG_ << "downloadAlbumImage()" << url;
    QNetworkRequest listWidgetItemRequest(url);
    QVariant variant;
    qVariantSetValue(variant,item);
    QObject* object=new QObject();

    object->setProperty("item",variant);
    listWidgetItemRequest.setOriginatingObject(object);

    QNetworkReply* reply = core.networkManager()->get(listWidgetItemRequest);
    connect(reply,SIGNAL(finished()),this,SLOT(pixmapDownloaded()));
}

//----------------------------------------------
// squeezster::downloadImage()
//----------------------------------------------
void squeezster::downloadImage(QPushButton* item, QString url) {

    //_DEBUG_ << "downloadImage(): Starting to download" << url;
    pushButtonNetworkRequest pushButtonRequest(item);
    pushButtonRequest.setUrl(url);
    QNetworkReply* reply = core.networkManager()->get(pushButtonRequest);
    connect(reply,SIGNAL(finished()),this,SLOT(pixmapDownloaded()));
}


//----------------------------------------------
// squeezster::pixmapDownloaded()
// Called when a http request has finished.
// Assigns the image to the correct GUI item
//----------------------------------------------
void squeezster::pixmapDownloaded()
{
    //_DEBUG_ << "pixmapDownloaded()";
    QNetworkReply* reply = static_cast<QNetworkReply*>(QObject::sender());
    QNetworkRequest request = reply->request();
    QVariant variant;
    if(request.originatingObject())
        variant = request.originatingObject()->property("item");
    else
        variant = request.attribute((QNetworkRequest::Attribute)(1000));
    //qDebug() << "pixmapDownloaded()" << reply->url().toString() << "Variant" << variant.typeName();
    QByteArray jpegData = QByteArray(reply->readAll());
    QPixmap downloadPixmap; //= new QPixmap();
    QIcon newIcon;

    if (reply->error() != QNetworkReply::NoError) {
        qWarning() << "Error in" << reply->url() << ":" << reply->errorString();
        //return;
        if (QString(variant.typeName()) == "QLabel*")
        {     // QLabel
            variant.value<QLabel*>()->setPixmap(QPixmap(LARGEEMPTYALBUMARTPIXMAPFILE));
        } else if (variant.userType()==QMetaType::type("QPushButton*"))
        {     // QPushButton
            if(variant.canConvert(variant.type()))
                variant.value<QPushButton*>()->setIcon(largeEmptyAlbumArtIcon);
            else {
                variant.value<QPushButton*>()->setIcon(largeEmptyAlbumArtIcon);
                qWarning() << "squeezster::pixmapDownloaded(QNetworkReply *reply) Cannot convert:" << variant.typeName();
                return; }
        } else if (variant.userType()==QMetaType::type("MapTreeItem*"))
        {
            variant.value<MapTreeItem*>()->setIcon(smallEmptyAlbumArtIcon);
        } else if (variant.userType()==QMetaType::type("MapListItem*"))
        {
            if(variant.canConvert(variant.type()))
                variant.value<MapListItem*>()->setIcon(emptyAlbumArtIcon);
            else {
                variant.value<MapListItem*>()->setIcon(emptyAlbumArtIcon);
                qWarning() << "squeezster::pixmapDownloaded(QNetworkReply *reply) Cannot convert:" << variant.typeName();
                return; }
        } else if (variant.userType()==QMetaType::type("QTreeWidgetItem*"))
        {
            variant.value<QTreeWidgetItem*>()->setIcon(0,smallEmptyAlbumArtIcon);
        } else if (variant.userType()==QMetaType::type("QListWidgetItem*"))
        {
            variant.value<QListWidgetItem*>()->setIcon(emptyAlbumArtIcon);
        }
    } else {
        if (!downloadPixmap.loadFromData(jpegData)) {
            qWarning() << "Error loadFromData()";
            downloadPixmap=QPixmap(LARGEEMPTYALBUMARTPIXMAPFILE);
        } else {
            newIcon= QIcon(downloadPixmap);
            if (variant.userType()==QMetaType::type("QPushButton*"))
            {     // QPushButton
                newIcon= QIcon(downloadPixmap.scaledToWidth(ALBUMART_WIDTH));
                variant.value<QPushButton*>()->setIcon(newIcon);
            } else if (variant.userType()==QMetaType::type("MapTreeItem*"))
            {
                MapTreeItem* newItem= variant.value<MapTreeItem*>();
                _DEBUG_ << "image" << newItem->property("image");
                newItem->setIcon(newIcon);
                //qDebug() << "pixmapDownloaded() Setting treeWidgetIcon";
            } else if (variant.userType()==QMetaType::type("MapListItem*"))
            {
                MapListItem* newItem= variant.value<MapListItem*>();
                newItem->setIcon(newIcon);
                //qDebug() << "pixmapDownloaded() Setting listWidgetIcon";
            } else if (variant.userType()==QMetaType::type("QLabel*"))
            {     // QLabel
                variant.value<QLabel*>()->setPixmap(downloadPixmap);
            } else if (variant.userType()==QMetaType::type("QTreeWidgetItem*"))
            {
                QTreeWidgetItem* newItem= variant.value<QTreeWidgetItem*>();
                newItem->setIcon(0,newIcon);
                //qDebug() << "pixmapDownloaded() Setting treeWidgetIcon" << variant.value<QTreeWidgetItem*>()->text(mapItemCol);
            } else if (variant.userType()==QMetaType::type("QListWidgetItem*"))
            {
                QListWidgetItem* newItem= variant.value<QListWidgetItem*>();
                newItem->setIcon(newIcon);
                //ui->albumsListWidget->update();
                //qDebug() << "pixmapDownloaded() Setting listWidgetIcon" << variant.value<QListWidgetItem*>()->text();
            } else {
                _DEBUG_ <<  "Couldn't find type for pixmap assignment!!!";
            }
        }
    }
    reply->deleteLater();
}

//----------------------------------------------
// squeezster::requestList
//----------------------------------------------
void squeezster::requestList(const QString& command, QByteArray playerID="")

{
    _DEBUG_ << "requestList()";
    if(command!="status")
        isBusy(command);
    core.requestList(command,playerID);
}
//----------------------------------------------
// squeezster::sendCometRequest
//----------------------------------------------
void squeezster::sendCometRequest(const QString& command, const QString& id="", const QString& search="", const QString& type="")
{
    if(command!="status")
        isBusy(command);
    core.sendQmlCometRequest(command, id, search, type);
}
//----------------------------------------------
// squeezster::sendRequest
//----------------------------------------------
void squeezster::sendRequest(QString command)
{
    core.sendQmlRequest(command);
}
//----------------------------------------------
// squeezster::initAlbumsList
//----------------------------------------------
void squeezster::initAlbumsList() {
    _DEBUG_ << "squeezster::initAlbumsList()";
    if(ui->albumsListWidget->count()==0 && !core.connectionSettings().value("mysb").toBool()) {
        ui->albumsListWidget->clear();
        requestList("albums");
    }
}

//----------------------------------------------
// squeezster::initPlaylistsList
//----------------------------------------------
void squeezster::initPlaylistsList() {
    _DEBUG_ << "squeezster::initPlaylistsList()";
    if(!core.connectionSettings().value("mysb").toBool()) {
        ui->playlistsListWidget->clear();
        requestList("playlists");
    }
}

//----------------------------------------------
// squeezster::initTitlesTree()
// Initializes the titlesTreeWidget
//----------------------------------------------
void squeezster::initTitlesTree() {
    _DEBUG_ << "squeezster::initTitlesTree()";
    if(ui->titlesTreeWidget->topLevelItemCount()==0 && !core.connectionSettings().value("mysb").toBool()) {
        ui->titlesTreeWidget->clear();
        MapTreeItem* newItem;
        newItem = new MapTreeItem();
        newItem->setText(mapItemCol,"Search");
        newItem->addProperty("cmd","titles");
        newItem->addProperty("type","search");
        ui->titlesTreeWidget->addTopLevelItem(newItem);
        newItem = new MapTreeItem();
        newItem->setText(mapItemCol,"All");
        newItem->addProperty("cmd","titles");
        newItem->addProperty("type","xmlbrowser");
        ui->titlesTreeWidget->addTopLevelItem(newItem);
    }
}

//----------------------------------------------
// squeezster::initArtistsTree()
// Initializes the artistsTreeWidget
//----------------------------------------------
void squeezster::initArtistsTree() {
    _DEBUG_ << "squeezster::initArtistsTree()";
    if(ui->artistsTreeWidget->topLevelItemCount()==0 && !core.connectionSettings().value("mysb").toBool()) {
        ui->artistsTreeWidget->clear();
        MapTreeItem* newItem;
        newItem = new MapTreeItem();
        newItem->setText(mapItemCol,"Search");
        newItem->addProperty("cmd","artists");
        newItem->addProperty("type","search");
        ui->artistsTreeWidget->addTopLevelItem(newItem);
        newItem = new MapTreeItem();
        newItem->setText(mapItemCol,"All");
        newItem->addProperty("cmd","artists");
        newItem->addProperty("type","xmlbrowser");
        ui->artistsTreeWidget->addTopLevelItem(newItem);
    }
}

//----------------------------------------------
// squeezster::savePlaylist()
// Saves the current playlist to the server
//----------------------------------------------
void squeezster::savePlaylist()
{
    _DEBUG_ << "squeezster::savePlaylist()";
    bool ok;
    QString playlistName = QInputDialog::getText(this,tr("Save Playlist"),tr("Name:"),QLineEdit::Normal,"",&ok);
    if(ok && !core.connectionSettings().value("mysb").toBool()) {
        sendRequest("playlist save "+playlistName);
        sendCometRequest("playlists");
    }
}

//----------------------------------------------
// squeezster::playlistItemClicked()
// Called when a song in the current playlist is
// clicked. Sends a request to play the song.
//----------------------------------------------
void squeezster::playlistItemClicked(QListWidgetItem* songItem)
{
    _DEBUG_ << "squeezster::playlistItemClicked()";
    MapListItem* item=static_cast<MapListItem*>(songItem);
    qVariantSetValue(listItemParent,static_cast<MapListItem*>(item));
    qVariantSetValue(itemSelected,static_cast<MapListItem*>(item));
    itemSelectTimer.start();
    sendPlaylistIndex(item->playlistIndex());
}

//----------------------------------------------
//----------------------------------------------
void squeezster::listItemPlay(QString cmd, MapListItem* item) {
    if(cmd=="playlists")
        sendRequest(QString("playlistcontrol cmd:load play_index:0 playlist_id:"+QUrl::toPercentEncoding(item->id())));
    else if(cmd=="nowplaying")
        sendRequest(QString("playlist index "+item->property("playlist index")));
    else if(cmd=="albums")
        sendRequest(QString("playlistcontrol cmd:load play_index:0 album_id:"+QUrl::toPercentEncoding(item->id())));
    else if(cmd=="artists")
        sendRequest(QString("playlistcontrol cmd:load play_index:0 artist_id:"+QUrl::toPercentEncoding(item->id())));
    else if(cmd=="titles")
        sendRequest(QString("playlistcontrol cmd:load track_id:"+QUrl::toPercentEncoding(item->id())));
    else if(cmd=="musicfolder")
        sendRequest(QString("playlistcontrol cmd:load play_index:0 "+QString((item->type()=="track"?"track_id:":"folder_id:"))+QUrl::toPercentEncoding(item->id())));
    //    else if(item->isAudio()!=0) {
    //        sendRequest(QString(cmd+" playlist play item_id:"+QUrl::toPercentEncoding(item->id())));
    //    }
}

//----------------------------------------------
//----------------------------------------------
void squeezster::treeItemPlay(QString cmd, MapTreeItem* item) {
    if(cmd=="playlists")
        sendRequest(QString("playlistcontrol cmd:load play_index:0 playlist_id:"+QUrl::toPercentEncoding(item->id())));
    else if(cmd=="nowplaying")
        sendRequest(QString("playlist index "+item->property("playlist index")));
    else if(cmd=="albums")
        sendRequest(QString("playlistcontrol cmd:load play_index:0 album_id:"+QUrl::toPercentEncoding(item->id())));
    else if(cmd=="artists")
        sendRequest(QString("playlistcontrol cmd:load play_index:0 artist_id:"+QUrl::toPercentEncoding(item->id())));
    else if(cmd=="titles")
        sendRequest(QString("playlistcontrol cmd:load track_id:"+QUrl::toPercentEncoding(item->id())));
    else if(cmd=="musicfolder")
        sendRequest(QString("playlistcontrol cmd:load play_index:0 "+QString((item->type()=="track"?"track_id:":"folder_id:"))+QUrl::toPercentEncoding(item->id())));
    else if(item->isAudio()!=0) {
        sendRequest(QString(cmd+" playlist play item_id:"+QUrl::toPercentEncoding(item->id())));
    }
}

//----------------------------------------------
//----------------------------------------------
void squeezster::treeItemAdd(QString cmd, MapTreeItem* item) {
    if(cmd=="playlists")
        sendRequest(QString("playlistcontrol cmd:add playlist_id:"+QUrl::toPercentEncoding(item->id())));
    else if(cmd=="albums")
        sendRequest(QString("playlistcontrol cmd:add album_id:"+QUrl::toPercentEncoding(item->id())));
    else if(cmd=="artists")
        sendRequest(QString("playlistcontrol cmd:add artist_id:"+QUrl::toPercentEncoding(item->id())));
    else if(cmd=="titles")
        sendRequest(QString("playlistcontrol cmd:add track_id:"+QUrl::toPercentEncoding(item->id())));
    else if(cmd=="musicfolder")
        sendRequest(QString("playlistcontrol cmd:add "+QString((item->type()=="track"?"track_id:":"folder_id:"))+QUrl::toPercentEncoding(item->id())));
    else if(item->isAudio()) {
        sendRequest(QString(cmd+" playlist add item_id:"+QUrl::toPercentEncoding(item->id())));
    }
}

//----------------------------------------------
//----------------------------------------------
void squeezster::listItemAdd(QString cmd, MapListItem* item) {
    if(cmd=="playlists")
        sendRequest(QString("playlistcontrol cmd:add playlist_id:"+QUrl::toPercentEncoding(item->id())));
    else if(cmd=="albums")
        sendRequest(QString("playlistcontrol cmd:add album_id:"+QUrl::toPercentEncoding(item->id())));
    else if(cmd=="artists")
        sendRequest(QString("playlistcontrol cmd:add artist_id:"+QUrl::toPercentEncoding(item->id())));
    else if(cmd=="titles")
        sendRequest(QString("playlistcontrol cmd:add track_id:"+QUrl::toPercentEncoding(item->id())));
    else if(cmd=="musicfolder")
        sendRequest(QString("playlistcontrol cmd:add "+QString(item->type()=="track"?"track_id:":"folder_id:")+QUrl::toPercentEncoding(item->id())));
}

//-------------------------------------------------
// squeezster::treeItemClicked()
// Called then an item in a QTreeWidget is clicked.
// Handles the action.
//-------------------------------------------------
void squeezster::treeItemClicked(QTreeWidgetItem* treeItem) {

    MapTreeItem* item = static_cast<MapTreeItem*>(treeItem);
    _DEBUG_ << "treeItemClicked()" << item->text() << item->command() << item->isXmlBrowser();
    QString cmd = item->command();
    bool expand = true;
    itemSelected.setValue(item);
    itemSelectTimer.start();

    if(!item->isExpanded()) {
        //_DEBUG_ << "treeItemClicked() not expanded" << item->isXmlBrowser() << item->hasItems();
        qVariantSetValue(treeItemParent,static_cast<MapTreeItem*>(item));

        if(item->isXmlBrowser()) {
            sendCometRequest(cmd);
        } else if((item->type()=="xmlbrowser_search") || (item->type()=="search")) {
            _DEBUG_ << "treeItemClicked() search";
            bool ok;
            QString searchString = QInputDialog::getText(this,tr(QByteArray("Search "+item->property("cmd").toUtf8()).data()),tr("Search string:"),QLineEdit::Normal,"",&ok);
            if (!ok) {
                searchString.clear();
                treeItemParent.clear();
            } else {
                sendCometRequest(cmd,QUrl::toPercentEncoding(item->id()),QUrl::toPercentEncoding(searchString),"");
            }
        } else if(item->hasItems()!=0) {
            if(item->type()=="textarea") {
                _DEBUG_ << "treeItemClicked() text";
                QString text = item->property("name") + item->property("text") + item->property("textarea");
                QMessageBox::information(this,item->property("cmd"),text);
                treeItemParent.clear();
            } else { //if (item->type()=="link"){
                sendCometRequest(cmd,QUrl::toPercentEncoding(item->id()),"","");
            }
        } else {
            if ((item->isAudio()!=0) || item->type()=="track") {
                _DEBUG_ << "treeItemClicked() isAudio/track isAudio:" << item->isAudio() << "type:" << item->type();
                treeItemPlay(cmd,item);
                treeItemParent.clear();
                expand=false;
            }
            else if ((item->type()=="text") || item->type()=="textarea") {
                _DEBUG_ << "treeItemClicked() text";
                QString text = item->property("name") + item->property("text") + item->property("textarea");
                QMessageBox::information(this,item->property("cmd"),text);
                treeItemParent.clear();
            } else if((cmd=="titles") || (cmd=="albums") || (cmd=="artists")) {
                if(!item->search().isEmpty()) {
                    if(!item->id().isEmpty())
                        sendCometRequest(cmd,QUrl::toPercentEncoding(item->id()),QUrl::toPercentEncoding(item->search()),"");
                    else
                        sendCometRequest(cmd,"",QUrl::toPercentEncoding(item->search()),"");
                } else {
                    if(cmd=="artists") { // Sub tree
                        sendCometRequest("albums",QUrl::toPercentEncoding(item->id()),"","");
                    }
                    else if (cmd=="albums") { // Sub tree
                        sendCometRequest("titles",QUrl::toPercentEncoding(item->id()),"","");
                    } else {
                        _DEBUG_ << "treeItemClicked() default: treeItemPlay";
                        treeItemPlay(cmd,item);
                        treeItemParent.clear();
                        expand=false;
                    }
                }
            } else { // E.g. folder
                sendCometRequest(cmd,QUrl::toPercentEncoding(item->id()),"","");
            }
        }
    }
    item->setExpanded(expand && !item->isExpanded());
}

//----------------------------------------------
// squeezster::listItemClicked()
// Called then an item in a QListWidget is clicked.
// Handles the action.
//----------------------------------------------
void squeezster::listItemClicked(QListWidgetItem* listItem) {
    MapListItem* item=static_cast<MapListItem*>(listItem);
    qVariantSetValue(itemSelected,static_cast<MapListItem*>(item));
    itemSelectTimer.start();
    QString parameters;
    _DEBUG_ << "listItemClicked()" << listItem->text();
    //if(cmd=="artists") {
    //lastName=mmodel.artist;
    //busyIndicator.running=true;
    //sendCometRequest("albums",QUrl::toPercentEncoding(item->value("id")),"","");
    //}
    //else if (cmd=="albums") {
    //lastName=mmodel.album;
    //busyIndicator.running=true;
    //sendListRequestSignal("titles",encodeURI(mmodel.id),"","")
    //}
    parameters += "playlistcontrol cmd:load ";
    if(!item->property("album").isEmpty())
        parameters += item->album_id();
    else if(!item->property("playlist").isEmpty())
        parameters += item->playlist_id();
    sendRequest(parameters);
}

//----------------------------------------------
// squeezster::itemUnselect()
// Called by a timer set when an item is clicked.
// Unselects the item, and thus gives feedback to
// the user that the click has been registered.
//----------------------------------------------
void squeezster::itemUnselect()
{
    _DEBUG_ << "squeezster::itemUnselect()";
    if(itemSelected.isValid()) {
        if (itemSelected.userType()==QMetaType::type("MapTreeItem*")) {
            itemSelected.value<MapTreeItem*>()->setSelected(false);
        }
        else if(itemSelected.userType()==QMetaType::type("QTreeWidgetItem*")) {
            itemSelected.value<QTreeWidgetItem*>()->setSelected(false);
        }
        else if (itemSelected.userType()==QMetaType::type("MapListItem*")) {
            itemSelected.value<MapListItem*>()->setSelected(false);
        }
        else if (itemSelected.userType()==QMetaType::type("QListWidgetItem*")) {
            itemSelected.value<QListWidgetItem*>()->setSelected(false);
        }
        itemSelected.clear();
    }
}

//----------------------------------------------
// squeezster::slotTabWidgetCurrentChanged(int currentIndex)
// Updates the contents of the album view on need basis.
//----------------------------------------------
void squeezster::slotTabWidgetCurrentChanged(int currentIndex)
{
    // Read Albums if not already done:
    if(currentIndex == 1) { // Favorites
        requestList(("favorites items"));
    } else if(currentIndex == 2) { // MusicLibraryView
        showMusicLibraryView();
        //if (ui->stackedWidget->currentIndex()==0) { // Albums
        //if (ui->stackedWidget->currentIndex()== 2) { // Titles
        //if (ui->stackedWidget->currentIndex()== 3) { // Artists
    } else if (currentIndex == 4) { // MusicMapView
        if(!musicFolderInitDone && !core.connectionSettings().value("mysb").toBool()) {
            ui->folderTreeWidget->clear();
            sendCometRequest("musicfolder");
            musicFolderInitDone=true;
        }
    }
}

//----------------------------------------------
// squeezster::showMusicLibraryView
//----------------------------------------------
void squeezster::showMusicLibraryView() {
    _DEBUG_ << "showMusicLibraryView()" << core.optionsSettings().value("libraryview").toString();
    if(core.optionsSettings().value("libraryview").toString()=="playlists") {
        showPlaylists();
    } else if(core.optionsSettings().value("libraryview").toString()=="titles") {
        showTitles();
    } else if(core.optionsSettings().value("libraryview").toString()=="artists") {
        showArtists();
    } else { // albums
        showAlbums();
    }
}

//----------------------------------------------
// squeezster::showAlbums() etc.
// Switches the active widget in the
// library presentation (a stackedWidget)
//----------------------------------------------
void squeezster::showAlbums() {
    initAlbumsList();
    ui->stackedWidget->setCurrentIndex(0);
    ui->stackedWidget->show();
    ui->squeezeTabWidget->setCurrentIndex(2);
    QVariantMap settings=core.optionsSettings();
    settings["libraryview"]=QString("albums");
    core.setOptionsSettings(settings);
    core.writeOptionsSettings();
}
//----------------------------------------------
void squeezster::showTitles() {
    initTitlesTree();
    ui->stackedWidget->setCurrentIndex(2);
    ui->stackedWidget->show();
    ui->squeezeTabWidget->setCurrentIndex(2);
    QVariantMap settings=core.optionsSettings();
    settings["libraryview"]=QString("titles");
    core.setOptionsSettings(settings);
    core.writeOptionsSettings();
}
//----------------------------------------------
void squeezster::showArtists() {
    initArtistsTree();
    ui->stackedWidget->setCurrentIndex(3);
    ui->stackedWidget->show();
    ui->squeezeTabWidget->setCurrentIndex(2);
    QVariantMap settings=core.optionsSettings();
    settings["libraryview"]=QString("artists");
    core.setOptionsSettings(settings);
    core.writeOptionsSettings();
}
//----------------------------------------------
void squeezster::showPlaylists() {
    initPlaylistsList();
    ui->stackedWidget->setCurrentIndex(1);
    ui->stackedWidget->show();
    ui->squeezeTabWidget->setCurrentIndex(2);
    QVariantMap settings=core.optionsSettings();
    settings["libraryview"]=QString("playlists");
    core.setOptionsSettings(settings);
    core.writeOptionsSettings();
}

//----------------------------------------------
// squeezster::isBusy(const QString busyString)
// Set busy indicator
//----------------------------------------------
void squeezster::isBusy(const QString &busyString)
{
    QString str;
    if (!busyString.isEmpty()) {
        squeezster::busyList<<busyString;
        QTimer::singleShot(10000, this, SLOT(isBusy())); // "Watchdog timer" clears busy symbol after 10 s
    }
    else
        if (!squeezster::busyList.isEmpty())
            str=squeezster::busyList.takeFirst();
    //qDebug() << "Busy:" << busyList.first() << QString::number(busyList.count());
#if defined(Q_WS_MAEMO_5) || defined(MEEGO_EDITION_HARMATTAN)
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, (busyList.isEmpty()?Qt::Unchecked:Qt::Checked));
#endif
}

//----------------------------------------------
// squeezster::constructConnDialog()
//----------------------------------------------
void squeezster::constructConnDialog()
{
    connectionUI=new connectionDialog(core.connectionSettings(),core.serverList(),this);
    connect(connectionUI,SIGNAL(connectionSettingsUpdated(QVariantMap)),this,SLOT(coreRestart(QVariantMap)));
    connectionUI->setServerList(core.serverList());
    connectionUI->exec();
}

//----------------------------------------------------------------
// squeezster::coreRestart(const QVariantMap& connectionSettings)
//----------------------------------------------------------------
void squeezster::coreRestart(const QVariantMap& connectionSettings) {
    _DEBUG_ << "coreRestart()";
    core.stop(core.serverHttpAddress());
    core.setConnectionSettings(connectionSettings);
    QTimer::singleShot(500,&core,SLOT(start()));
}

//----------------------------------------------
void squeezster::increaseVolume() {
    _DEBUG_ << "increaseVolume()";
    ui->volumeSlider->setValue(ui->volumeSlider->value()+5);
}
//----------------------------------------------
void squeezster::decreaseVolume() {
    _DEBUG_ << "decreaseVolume()";
    ui->volumeSlider->setValue(ui->volumeSlider->value()-5);
}

//----------------------------------------------
// squeezster::slotOptions()
//----------------------------------------------
void squeezster::slotOptions()
{
    _DEBUG_ << "squeezster::slotOptions()";
    OptionsDialog* myOptionsDialog =new OptionsDialog(core.optionsSettings(),core.squeezsterVersion());
    myOptionsDialog->show();
    connect(myOptionsDialog,SIGNAL(accepted(QVariantMap)),&core,SLOT(setOptionsSettings(QVariantMap)));
    connect(myOptionsDialog,SIGNAL(accepted(QVariantMap)),&core,SLOT(writeOptionsSettings()));
    connect(myOptionsDialog,SIGNAL(accepted(QVariantMap)),this,SLOT(setAlbumsListView()));
}

//----------------------------------------------
// squeezster::setAlbumsListView()
//----------------------------------------------
void squeezster::setAlbumsListView(bool loadList)
{
    _DEBUG_ << "squeezster::setAlbumsListView()";
    if (core.optionsSettings().value("albumicons").toBool()) { // iconview
        if (core.optionsSettings().value("largealbumicons").toBool()) {
            albumartsuffix = QString(LARGERALBUMARTSUFFIX);
            emptyAlbumArtIcon = QIcon(LARGEEMPTYALBUMARTPIXMAPFILE);
            ui->albumsListWidget->setGridSize(QSize(116,150));
            ui->albumsListWidget->setIconSize(QSize(112,112));
        } else {
            albumartsuffix = QString(SMALLALBUMARTSUFFIX);
            emptyAlbumArtIcon = QIcon(SMALLEMPTYALBUMARTPIXMAPFILE);
            ui->albumsListWidget->setGridSize(QSize(116,90));
            ui->albumsListWidget->setIconSize(QSize(50,50));
        }
        ui->albumsListWidget->setViewMode(QListView::IconMode);
    } else { // listview
        albumartsuffix = QString(SMALLALBUMARTSUFFIX);
        emptyAlbumArtIcon = QIcon(SMALLEMPTYALBUMARTPIXMAPFILE);
        ui->albumsListWidget->setGridSize(QSize(60,60));
        ui->albumsListWidget->setIconSize(QSize(50,50));
        ui->albumsListWidget->setViewMode(QListView::ListMode);
    }
    if (ui->stackedWidget->currentIndex()==0 && ui->squeezeTabWidget->currentIndex()==2) {
        ui->albumsListWidget->clear();
        initAlbumsList();
    }
}

//----------------------------------------------
// squeezster::slotMusicLibraryView()
//----------------------------------------------
void squeezster::slotMusicLibraryView()
{
    _DEBUG_ << "squeezster::slotMusicLibraryView()";
    libraryViewDialog* myLibraryViewDialog;
    myLibraryViewDialog = new libraryViewDialog();
    connect(myLibraryViewDialog, SIGNAL(albums()),this,SLOT(showAlbums()));
    connect(myLibraryViewDialog, SIGNAL(titles()),this,SLOT(showTitles()));
    connect(myLibraryViewDialog, SIGNAL(playlists()),this,SLOT(showPlaylists()));
    connect(myLibraryViewDialog, SIGNAL(artists()),this,SLOT(showArtists()));

    myLibraryViewDialog->initButtons(ui->stackedWidget->currentIndex());
    myLibraryViewDialog->show();
}

//----------------------------------------------
// Strip ID on first value
QString squeezster::favId(QString id)
{
    QString favID;
    int index = id.indexOf(".");
    if(index<0) {
        favID = id;
    } else
        favID=id.remove(0,index+1);
    _DEBUG_ << "favId() index:"+QString::number(index)+" id:"+id+" -> "+favID;
    return favID;
}
//----------------------------------------------
// Strip ID on last value
QString squeezster::favParent(QString id)
{
    QString favPar= favId(id);
    int index = favPar.lastIndexOf(".");
    if (index<0)
        favPar="";
    else
        favPar.truncate(index);
    _DEBUG_ << "favParent() index:" << index << "id:" << id <<"->"<<favPar;
    return favPar;
}
//----------------------------------------------
QString squeezster::treeItemUrl(MapTreeItem* item)
{
    QString url;
    if(!item->property("remote").isEmpty() || !item->property("url").isEmpty())
        url=QUrl::toPercentEncoding(item->property("url"));
    else if(!item->property("album").isEmpty())
        url=QUrl::toPercentEncoding("db:album.title="+item->property("album"));
    else //if("artist" in currentitem)
        url=QUrl::toPercentEncoding("db:contributor.name="+item->property("artist"));
    return url;
}
//----------------------------------------------
void squeezster::treeItemAddFav(MapTreeItem* item)
{
    QString name=QUrl::toPercentEncoding(item->title());
    sendRequest("favorites add url:"+item->property("url")+" title:"+name);
}
//----------------------------------------------
//void squeezster::checkFavoriteExist()
//{
//    sendRequest("favorites exists "+itemUrl(currentSong))
//}
//----------------------------------------------
void squeezster::treeItemDelFav(MapTreeItem* item)
{
    sendRequest("favorites delete item_id:"+QUrl::toPercentEncoding(favId(item->id())));
    sendCometRequest("favorites items",QUrl::toPercentEncoding(favParent(item->id())),"","");
}
//----------------------------------------------
void squeezster::treeItemRenameFav(MapTreeItem* item)
{
    bool ok;
    QString newName = QInputDialog::getText(this,tr("Rename favorite"),tr("Enter new name:"),QLineEdit::Normal,item->name(),&ok);
    if (!ok)
        newName.clear();
    else {
        sendRequest("favorites rename item_id:"+QUrl::toPercentEncoding(favId(item->id()))+" title:"+QUrl::toPercentEncoding(newName));
    }
    sendCometRequest("favorites items",QUrl::toPercentEncoding(favParent(item->id())),"","");
}

//----------------------------------------------
// squeezster::treeContextMenuHandler(QPoint point)
//----------------------------------------------
void squeezster::treeContextMenuHandler(QPoint point) {
    QTreeWidget* treeWidget = static_cast<QTreeWidget*>(this->sender());
    QString cmd;
    QPoint widgetPos = treeWidget->mapFromGlobal(point);
    // Adjust for N900 fullscreen:
    if(!this->isFullScreen())
        widgetPos.ry()+=54;
    QMenu contextMenu;
    contextMenu.addActions(treeWidget->actions());
    QAction* selectedAction = contextMenu.exec(point);

    _DEBUG_ << this->objectName() << this->sender()->objectName();
    _DEBUG_ << "treeContextMenuHandler(QPoint point). Pos:" << QString::number(point.y());
    _DEBUG_ << "treeContextMenuHandler(QPoint point). Adjusted Pos:" << QString::number(widgetPos.y());

    if (selectedAction) {
        MapTreeItem* item=static_cast<MapTreeItem*>(treeWidget->itemAt(widgetPos));
        cmd = item->command();
        if (item) {
            _DEBUG_ << item->name() << "Item isaudio" << QString::number(item->isAudio()) << "Command:" << item->command();
            QString command;
            QString parameters;
            if(selectedAction->text().toLower()==QString("play"))
                treeItemPlay(cmd,item);
            else if(selectedAction->text().toLower()==QString("add to playlist"))
                treeItemAdd(cmd,item);
            else if(selectedAction->text().toLower()==QString("add to favorites"))
                treeItemAddFav(item);
            else if(selectedAction->text().toLower()==QString("rename favorite"))
                treeItemRenameFav(item);
            else if(selectedAction->text().toLower()==QString("delete favorite"))
                treeItemDelFav(item);
            parameters += item->item_id();
        }
    }
}


//----------------------------------------------
// squeezster::listContextMenuHandler(QPoint point)
//----------------------------------------------
void squeezster::listContextMenuHandler(QPoint point) {
    QListWidget* listWidget;
    QString cmd;
    listWidget = static_cast<QListWidget*>(this->sender());
    QPoint widgetPos = listWidget->mapFromGlobal(point);
    QMenu contextMenu;
    contextMenu.addActions(listWidget->actions());

    if(!this->isFullScreen())
        widgetPos.ry()+=54;
    QAction* selectedAction = contextMenu.exec(point);
    MapListItem* item = static_cast<MapListItem*>(listWidget->itemAt(widgetPos));
    _DEBUG_ << this->objectName() << this->sender()->objectName();
    _DEBUG_ << "listContextMenuHandler(QPoint point). Pos:" << QString::number(point.y());
    _DEBUG_ << "listContextMenuHandler(QPoint point). Adjusted Pos:" << QString::number(widgetPos.y());

    QString idStr="";
    if(listWidget==ui->albumsListWidget) {
        cmd="albums";
        idStr=item->album_id();
    } else if(listWidget==ui->nowPlayingListWidget) {
        cmd="nowplaying";
    }else if(listWidget==ui->playlistsListWidget) {
        cmd="playlists";
        idStr=item->playlist_id();
    }

    if (selectedAction) {
        if(selectedAction->text().toLower()==QString("play"))
            sendRequest("playlistcontrol cmd:play "+QUrl::toPercentEncoding(idStr));
        else if(selectedAction->text().toLower()==QString("add to playlist"))  // albums
            sendRequest("playlistcontrol cmd:add "+QUrl::toPercentEncoding(idStr));
        else if(selectedAction->text().toLower()==QString("delete from playlist")) {
            _DEBUG_ << "playlist delete " << listWidget->row(item);
            sendRequest("playlist delete "+QString::number(listWidget->row(item)));  }
        else if(selectedAction->text().toLower()==QString("delete playlist")) {
            sendRequest("playlists delete "+QUrl::toPercentEncoding(idStr));
            sendCometRequest("playlists"); }
        else if(selectedAction->text().toLower()==QString("rename playlist")) {
            bool ok;
            QString newName = QInputDialog::getText(this,tr("Rename playlist"),tr("Enter new name:"),QLineEdit::Normal,item->playlist(),&ok);
            if (!ok)
                newName.clear();
            else {
                sendRequest("playlists rename "+QUrl::toPercentEncoding(idStr) + " newname:"+QUrl::toPercentEncoding(newName));
                sendCometRequest("playlists");
            }
        }
        else if(selectedAction->text().toLower()==QString("add to favorites")) {
            QString name=QUrl::toPercentEncoding(item->title());
            if(!item->property("url").isEmpty()) {
                sendRequest("favorites add url:"+item->property("url")+" title:"+name);
                sendCometRequest("favorites items");
            }
        }
        _DEBUG_ << "Action" << selectedAction->text() << listWidget->itemAt(widgetPos)->text();
        _DEBUG_ << "SelectedItems" << listWidget->selectedItems();
        _DEBUG_ << "Current Item" << listWidget->currentItem();
    }
}

//----------------------------------------------
// Init UI
//----------------------------------------------
void squeezster::initUI()
{
    _DEBUG_ << "initUI";
    screenMode = 0;

    connect(ui->albumArtLabel,SIGNAL(clicked()),this,SLOT(incrementScreenMode()));
    connect(ui->actionOptions,SIGNAL(triggered()),this,SLOT(slotOptions()));
    //connect(ui->actionClearCache,SIGNAL(triggered()),this,SLOT(clearCache()));
    connect(ui->actionShuffle,SIGNAL(triggered()),this,SLOT(slotShuffle()));
    connect(ui->actionRepeat,SIGNAL(triggered()),this,SLOT(slotRepeat()));
    connect(ui->actionConnectionSettings,SIGNAL(triggered()),this,SLOT(constructConnDialog()));
    connect(ui->playButton, SIGNAL(clicked()), this, SLOT(sendPlay()));
    connect(ui->powerButton, SIGNAL(clicked()), this, SLOT(sendPower()));
    connect(ui->pauseButton, SIGNAL(clicked()), this, SLOT(sendPause()));
    connect(ui->stopButton, SIGNAL(clicked()), this, SLOT(sendStop()));
    connect(ui->skipBackButton, SIGNAL(clicked()), this, SLOT(sendSkipBack()));
    connect(ui->skipFwdButton, SIGNAL(clicked()), this, SLOT(sendSkipForward()));
    connect(ui->volumeSlider, SIGNAL(valueChanged(int)), this, SLOT(setVolume(int)));
    connect(ui->progressBar, SIGNAL(sliderMoved(int)), this, SLOT(setNewProgress(int)));
    connect(ui->nowPlayingListWidget, SIGNAL(itemClicked(QListWidgetItem*)), this, SLOT(playlistItemClicked(QListWidgetItem*)));
    connect(ui->albumsListWidget, SIGNAL(itemClicked(QListWidgetItem*)), this, SLOT(listItemClicked(QListWidgetItem*)));
    connect(ui->playlistsListWidget, SIGNAL(itemClicked(QListWidgetItem*)), this, SLOT(listItemClicked(QListWidgetItem*)));
    connect(ui->radiosTreeWidget, SIGNAL(itemClicked(QTreeWidgetItem*,int)), this, SLOT(treeItemClicked(QTreeWidgetItem*)));
    connect(ui->appsTreeWidget, SIGNAL(itemClicked(QTreeWidgetItem*,int)), this, SLOT(treeItemClicked(QTreeWidgetItem*)));
    connect(ui->artistsTreeWidget, SIGNAL(itemClicked(QTreeWidgetItem*,int)), this, SLOT(treeItemClicked(QTreeWidgetItem*)));
    connect(ui->titlesTreeWidget, SIGNAL(itemClicked(QTreeWidgetItem*,int)), this, SLOT(treeItemClicked(QTreeWidgetItem*)));
    connect(ui->favoritesTreeWidget, SIGNAL(itemClicked(QTreeWidgetItem*,int)), this, SLOT(treeItemClicked(QTreeWidgetItem*)));
    connect(ui->folderTreeWidget, SIGNAL(itemClicked(QTreeWidgetItem*,int)), this, SLOT(treeItemClicked(QTreeWidgetItem*)));
    connect(ui->actionClearPlaylist,SIGNAL(triggered()),this,SLOT(sendPlaylistClear()));
    connect(ui->actionSavePlaylist,SIGNAL(triggered()),this,SLOT(savePlaylist()));
    connect(ui->actionMusicLibraryView, SIGNAL(triggered()),this,SLOT(slotMusicLibraryView()));
    connect(ui->actionRescanMusicLibrary,SIGNAL(triggered()),this,SLOT(slotRescan()));
    connect(ui->squeezeTabWidget, SIGNAL(currentChanged(int)),this,SLOT(slotTabWidgetCurrentChanged(int)));
    connect(ui->playerComboBox,SIGNAL(currentIndexChanged(int)),this,SLOT(playerChangedSlot()));

    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(setupScreenMode()));

    ui->radiosTreeWidget->setColumnWidth(0,470);
    ui->appsTreeWidget->setColumnWidth(0,470);
    ui->folderTreeWidget->setColumnWidth(0,470);
    ui->titlesTreeWidget->setColumnWidth(0,470);
    ui->artistsTreeWidget->setColumnWidth(0,470);
    ui->albumLabel->hide();

    bulletIcon = QIcon(BULLETPIXMAPFILE);
    smallBulletIcon = QIcon(SMALLBULLETPIXMAPFILE);
    plusIcon =  QIcon(PLUSPIXMAPFILE);
    minusIcon = QIcon(MINUSPIXMAPFILE);
    powerIcon = QIcon(POWERPIXMAPFILE);
    playIcon = QIcon(PLAYPIXMAPFILE);
    pauseIcon = QIcon(PAUSEPIXMAPFILE);
    stopIcon = QIcon(STOPPIXMAPFILE);
    rewIcon = QIcon(REWPIXMAPFILE);
    fwdIcon = QIcon(FWDPIXMAPFILE);
    noteIcon= QIcon(SMALLWHITENOTEPIXMAPFILE);
    largeEmptyAlbumArtIcon = QIcon(LARGEEMPTYALBUMARTPIXMAPFILE);
    smallEmptyAlbumArtIcon = QIcon(SMALLEMPTYALBUMARTPIXMAPFILE);


    //----------------------------------------------------------------------
    // Add context menus to  widgets
    //----------------------------------------------------------------------
    // albumsListWidget Actions
    ui->albumsListWidget->setContextMenuPolicy(Qt::CustomContextMenu);
    QAction* albumsPlayAction=new QAction(noteIcon,"Play",ui->albumsListWidget);
    QAction* albumsAddAction=new QAction(noteIcon,"Add to playlist",ui->albumsListWidget);
    QAction* albumsAddFavAction=new QAction(noteIcon,"Add to favorites",ui->albumsListWidget);
    //QAction* albumsDelFavAction=new QAction(noteIcon,"Delete favorite",ui->albumsListWidget);
    //QAction* albumsRenFavAction=new QAction(noteIcon,"Rename favorite",ui->albumsListWidget);
    ui->albumsListWidget->addAction(albumsPlayAction);
    ui->albumsListWidget->addAction(albumsAddAction);
    ui->albumsListWidget->addAction(albumsAddFavAction);
    //ui->albumsListWidget->addAction(albumsDelFavAction);
    //ui->albumsListWidget->addAction(albumsRenFavAction);
    connect(ui->albumsListWidget,SIGNAL(customContextMenuRequested(QPoint)),this,SLOT(listContextMenuHandler(QPoint)));

    // playlistsListWidget Actions
    ui->playlistsListWidget->setContextMenuPolicy(Qt::CustomContextMenu);
    QAction* playlistsPlayAction=new QAction(noteIcon,"Play",ui->playlistsListWidget);
    QAction* playlistsAddAction=new QAction(noteIcon,"Add to playlist",ui->playlistsListWidget);
    QAction* playlistsDelAction=new QAction(noteIcon,"Delete playlist",ui->playlistsListWidget);
    QAction* playlistsRenAction=new QAction(noteIcon,"Rename playlist",ui->playlistsListWidget);
    ui->playlistsListWidget->addAction(playlistsPlayAction);
    ui->playlistsListWidget->addAction(playlistsAddAction);
    ui->playlistsListWidget->addAction(playlistsDelAction);
    ui->playlistsListWidget->addAction(playlistsRenAction);
    connect(ui->playlistsListWidget,SIGNAL(customContextMenuRequested(QPoint)),this,SLOT(listContextMenuHandler(QPoint)));

    // nowPlayingListWidget Actions
    ui->nowPlayingListWidget->setContextMenuPolicy(Qt::CustomContextMenu);
    QAction* playlistAddFavAction=new QAction(bulletIcon,"Add to favorites",ui->nowPlayingListWidget);
    QAction* playlistDeleteAction=new QAction(bulletIcon,"Delete from playlist",ui->nowPlayingListWidget);
    ui->nowPlayingListWidget->addAction(playlistAddFavAction);
    ui->nowPlayingListWidget->addAction(playlistDeleteAction);
    connect(ui->nowPlayingListWidget,SIGNAL(customContextMenuRequested(QPoint)),this,SLOT(listContextMenuHandler(QPoint)));

    // radioTreeWidget Actions
    ui->radiosTreeWidget->setContextMenuPolicy(Qt::CustomContextMenu);
    QAction* radioPlayAction=new QAction(noteIcon,"Play",ui->radiosTreeWidget);
    QAction* radioAddAction=new QAction(noteIcon,"Add to playlist",ui->radiosTreeWidget);
    QAction* radioAddFavAction=new QAction(noteIcon,"Add to favorites",ui->radiosTreeWidget);
    radioPlayAction->setIconVisibleInMenu(true);
    radioAddAction->setIconVisibleInMenu(true);
    ui->radiosTreeWidget->addAction(radioPlayAction);
    ui->radiosTreeWidget->addAction(radioAddAction);
    ui->radiosTreeWidget->addAction(radioAddFavAction);
    connect(ui->radiosTreeWidget,SIGNAL(customContextMenuRequested(QPoint)),this,SLOT(treeContextMenuHandler(QPoint)));

    // appTreeWidget Actions
    ui->appsTreeWidget->setContextMenuPolicy(Qt::CustomContextMenu);
    QAction* appsPlayAction=new QAction(noteIcon,"Play",ui->appsTreeWidget);
    QAction* appsAddAction=new QAction(noteIcon,"Add to playlist",ui->appsTreeWidget);
    QAction* appsAddFavAction=new QAction(noteIcon,"Add to favorites",ui->appsTreeWidget);
    appsPlayAction->setIconVisibleInMenu(true);
    appsAddAction->setIconVisibleInMenu(true);
    appsAddFavAction->setIconVisibleInMenu(true);
    ui->appsTreeWidget->addAction(appsPlayAction);
    ui->appsTreeWidget->addAction(appsAddAction);
    ui->appsTreeWidget->addAction(appsAddFavAction);
    ui->appsTreeWidget->addAction(appsAddFavAction);
    connect(ui->appsTreeWidget,SIGNAL(customContextMenuRequested(QPoint)),this,SLOT(treeContextMenuHandler(QPoint)));

    // titlesTreeWidget Actions
    ui->titlesTreeWidget->setContextMenuPolicy(Qt::CustomContextMenu);
    QAction* titlesPlayAction=new QAction(noteIcon,"Play",ui->titlesTreeWidget);
    QAction* titlesAddAction=new QAction(noteIcon,"Add to playlist",ui->titlesTreeWidget);
    QAction* titlesAddFavAction=new QAction(noteIcon,"Add to favorites",ui->titlesTreeWidget);
    titlesPlayAction->setIconVisibleInMenu(true);
    titlesAddAction->setIconVisibleInMenu(true);
    ui->titlesTreeWidget->addAction(titlesPlayAction);
    ui->titlesTreeWidget->addAction(titlesAddAction);
    ui->titlesTreeWidget->addAction(titlesAddFavAction);
    connect(ui->titlesTreeWidget,SIGNAL(customContextMenuRequested(QPoint)),this,SLOT(treeContextMenuHandler(QPoint)));

    // folderTreeWidget Actions
    ui->folderTreeWidget->setContextMenuPolicy(Qt::CustomContextMenu);
    QAction* folderPlayAction=new QAction(noteIcon,"Play",ui->folderTreeWidget);
    QAction* folderAddAction=new QAction(noteIcon,"Add to playlist",ui->folderTreeWidget);
    QAction* folderAddFavAction=new QAction(noteIcon,"Add to favorites",ui->folderTreeWidget);
    folderPlayAction->setIconVisibleInMenu(true);
    folderAddAction->setIconVisibleInMenu(true);
    ui->folderTreeWidget->addAction(folderPlayAction);
    ui->folderTreeWidget->addAction(folderAddAction);
    ui->folderTreeWidget->addAction(folderAddFavAction);
    connect(ui->folderTreeWidget,SIGNAL(customContextMenuRequested(QPoint)),this,SLOT(treeContextMenuHandler(QPoint)));

    // artistsTreeWidget Actions
    ui->artistsTreeWidget->setContextMenuPolicy(Qt::CustomContextMenu);
    QAction* artistsPlayAction=new QAction(noteIcon,"Play",ui->artistsTreeWidget);
    QAction* artistsAddAction=new QAction(noteIcon,"Add to playlist",ui->artistsTreeWidget);
    artistsPlayAction->setIconVisibleInMenu(true);
    artistsAddAction->setIconVisibleInMenu(true);
    ui->artistsTreeWidget->addAction(artistsPlayAction);
    ui->artistsTreeWidget->addAction(artistsAddAction);
    connect(ui->artistsTreeWidget,SIGNAL(customContextMenuRequested(QPoint)),this,SLOT(treeContextMenuHandler(QPoint)));

    // favoritesTreeWidget Actions
    ui->favoritesTreeWidget->setContextMenuPolicy(Qt::CustomContextMenu);
    QAction* favPlayAction=new QAction(noteIcon,"Play",ui->favoritesTreeWidget);
    QAction* favAddAction=new QAction(noteIcon,"Add to playlist",ui->favoritesTreeWidget);
    QAction* favDelAction=new QAction(noteIcon,"Delete favorite",ui->favoritesTreeWidget);
    QAction* favRenAction=new QAction(noteIcon,"Rename favorite",ui->favoritesTreeWidget);
    ui->favoritesTreeWidget->addAction(favPlayAction);
    ui->favoritesTreeWidget->addAction(favAddAction);
    ui->favoritesTreeWidget->addAction(favDelAction);
    ui->favoritesTreeWidget->addAction(favRenAction);
    connect(ui->favoritesTreeWidget,SIGNAL(customContextMenuRequested(QPoint)),this,SLOT(treeContextMenuHandler(QPoint)));

}

void squeezster::nowPlayingListChangedSlot()
{
    _DEBUG_ << "nowPlayingListChangedSlot";
    updateListUI("nowplaying",core.nowPlayingList());
}
void squeezster::albumsListChangedSlot()
{
    _DEBUG_ << "albumsListChangedSlot";
    updateListUI("albums",core.albumsList());
}
void squeezster::radiosListChangedSlot()
{
    _DEBUG_ << "radiosListChangedSlot";
    updateTreeUI("radios",core.radiosList());
}
void squeezster::appsListChangedSlot()
{
    _DEBUG_ << "appsListChangedSlot";
    updateTreeUI("apps",core.appsList());
}
void squeezster::playersListChangedSlot()
{
    _DEBUG_ << "playersListChangedSlot activePlayer():" << core.activePlayer() << core.activePlayerIndex();
    disconnect(ui->playerComboBox,SIGNAL(currentIndexChanged(int)),this,SLOT(playerChangedSlot()));
    ui->playerComboBox->clear();
    for(int i=0;i<core.playersList().count();i++) {
        _DEBUG_ << "playersListChangedSlot" << i;
        ui->playerComboBox->addItem(core.playersList().at(i).toMap().value("name").toString(),core.playersList().at(i).toMap());
    }
    _DEBUG_ << "playersListChangedSlot -> set current index" << core.activePlayerIndex();
    ui->playerComboBox->setCurrentIndex(core.activePlayerIndex());
    connect(ui->playerComboBox,SIGNAL(currentIndexChanged(int)),this,SLOT(playerChangedSlot()));
}
void squeezster::playerChangedSlot() {
    _DEBUG_ << "playerChangedSlot()" << ui->playerComboBox->currentIndex() << ui->playerComboBox->itemData(ui->playerComboBox->currentIndex()).toMap().value("playerid").toByteArray();
    core.setActivePlayer(ui->playerComboBox->itemData(ui->playerComboBox->currentIndex()).toMap());
}

void squeezster::activePlayerIndexChangedSlot() {
    _DEBUG_ << "activePlayerIndexChangedSlot() index" << core.activePlayerIndex();
    ui->playerComboBox->setCurrentIndex(core.activePlayerIndex());
}

void squeezster::serverListChangedSlot()
{
    _DEBUG_ << "serverListChangedSlot";
    if(connectionUI)
        connectionUI->setServerList(core.serverList());
}
void squeezster::listListChangedSlot()
{
    _DEBUG_ << "listListChangedSlot";
    updateListUI(core.listCommand(),core.listList());
}
void squeezster::treeListChangedSlot()
{
    _DEBUG_ << "treeListChanged";
    updateTreeUI(core.treeCommand(),core.treeList());
}

//-----------------------------------------------------------------------------------
// squeezster::updateTreeUI
// Generic xml message decoder
//-----------------------------------------------------------------------------------
void squeezster::updateTreeUI(const QString& command, const QVariantList& resultList) {
    MapTreeItem* newMapTreeItem;
    MapTreeItem* parentTreeWidgetItem=0;
    QTreeWidget* parentTreeWidget=0;
    QList<QTreeWidgetItem*> *treeWidgetItemList= new QList<QTreeWidgetItem*>();
    bool isTreeItem = false;
    _DEBUG_ << "updateTreeUI() command:" << command << "resultList" << resultList;

    //---------------------------------------
    // Check if parent is treeWidget or
    // treeWidgetItem
    //---------------------------------------
    if (!treeItemParent.isValid()) {
        // Root level
        _DEBUG_ << "update() parent=0";
        if(command=="radios")
            parentTreeWidget=ui->radiosTreeWidget;
        else if(command=="favorites")
            parentTreeWidget=ui->favoritesTreeWidget;
        else if(command=="apps")
            parentTreeWidget=ui->appsTreeWidget;
        else if(command=="musicfolder")
            parentTreeWidget=ui->folderTreeWidget;
        else if(command=="titles")
            parentTreeWidget=ui->titlesTreeWidget;
        else if(command=="artists")
            parentTreeWidget=ui->artistsTreeWidget;
        else {
            return;
            treeItemParent.clear(); }
    } else {
        _DEBUG_ << "updateTreeUI() parent!=0" << QVariant::typeToName(treeItemParent.type());
        if (treeItemParent.userType()==QMetaType::type("QTreeWidget*")) {
            parentTreeWidget=treeItemParent.value<QTreeWidget*>();
            parentTreeWidgetItem=0;
            _DEBUG_ << "QTreeWidget:" << parentTreeWidget->objectName();
        } else {
            isTreeItem =true;
            parentTreeWidget=0;
            parentTreeWidgetItem=treeItemParent.value<MapTreeItem*>();
            _DEBUG_ << "MapTreeItem:" << parentTreeWidgetItem->text();
        }
    }

    isBusy();

    // Clean tree:
    if (isTreeItem) {
        // Delete?
        parentTreeWidgetItem->takeChildren();
    }
    else
        parentTreeWidget->clear();

    // Add new children
    _DEBUG_ << "updateTreeUI()";
    for(int i=0;i<resultList.count();i++)
    {
        newMapTreeItem=new MapTreeItem();
        newMapTreeItem->setMap(resultList.at(i).toMap());
        //_DEBUG_<< "Iterating..." << newMapTreeItem->property("name");
        if(newMapTreeItem->property("cmd").isEmpty()) {
            newMapTreeItem->addProperty("cmd",command); }
        if(!isTreeItem && newMapTreeItem->isXmlBrowser())  // Radio top level
            newMapTreeItem->addProperty("hasitems","1");
        //else if(isTreeItem && newMapTreeItem->property("hasitems").isEmpty())
        //    newMapTreeItem->addProperty("isaudio","1");
        newMapTreeItem->setTitle();
        newMapTreeItem->setIcon();
        downloadImage(newMapTreeItem);
        *treeWidgetItemList << newMapTreeItem;
    }
    // Insert tree list in parent:
    _DEBUG_ << "Inserting in parent...";
    if(isTreeItem) {
        parentTreeWidgetItem->addChildren(*treeWidgetItemList);
    } else {
        parentTreeWidget->addTopLevelItems(*treeWidgetItemList);
    }

    if(isTreeItem)
        (treeItemParent.value<MapTreeItem*>())->setExpanded(true);
    _DEBUG_ << "updateTreeUI() exit";
    treeItemParent.clear();
}

//-----------------------------------------------------------------------------------
// squeezster::updateListUI
//-----------------------------------------------------------------------------------
void squeezster::updateListUI(QString command, const QVariantList& resultList) {
    QListWidget* listParent;
    MapListItem* newMapListItem;
    _DEBUG_ << "updateListUI()" << command << treeItemParent.typeName();

    // Find parent widget of the list:
    if (treeItemParent.isValid() && command != "nowplaying" && command != "playlists") {
        updateTreeUI(command, resultList);
        return;
    } else {
        if(command=="nowplaying") {
            _DEBUG_ << "Status list received.";
            listParent=ui->nowPlayingListWidget;
        } else if(command=="albums") {
            _DEBUG_ << "Albums list received." << ui->albumsListWidget->isWindow() << ui->albumsListWidget->objectName();
            listParent=ui->albumsListWidget;
        } else if(command=="artists") {
            _DEBUG_ << "Artists list received.";
            updateTreeUI(command, resultList);
            return;
        } else if(command=="titles") {
            _DEBUG_ << "titles list received.";
            updateTreeUI(command, resultList);
            return;
        } else if(command=="playlists") {
            _DEBUG_ << "Playlists list received.";
            listParent=ui->playlistsListWidget;
        }
        else {
            _DEBUG_ << "Unknown list received.";
            return;
        }
    }
    isBusy();
    listParent->clear();
    for(int i=0;i<resultList.count();i++)
    {
        //_DEBUG_ << "Iterating...";
        newMapListItem=new MapListItem(listParent);
        newMapListItem->setMap(resultList.at(i).toMap());
        newMapListItem->addProperty("type",command);
        newMapListItem->setTitle();
        newMapListItem->setSizeHint(listParent->gridSize());
        if (command=="albums") {
            downloadAlbumImage(newMapListItem);
            //qDebug() << "property artist:" << newMapListItem->property("artist");
        } else {
            newMapListItem->setIcon();
        }
    }
}

//------------------------------------------------
//-----------------------------------------------
#ifdef Q_WS_MAEMO_5
void squeezster::grabZoomKeys(bool grab)
{
    if (!winId()) {
        qWarning("Can't grab keys unless we have a window id");
        return;
    }

    unsigned long val = (grab) ? 1 : 0;
    Atom atom = XInternAtom(QX11Info::display(), "_HILDON_ZOOM_KEY_ATOM", False);
    if (!atom) {
        qWarning("Unable to obtain _HILDON_ZOOM_KEY_ATOM. This example will only work "
                 "on a Maemo 5 device!");
        return;
    }

    XChangeProperty (QX11Info::display(),
                     winId(),
                     atom,
                     XA_INTEGER,
                     32,
                     PropModeReplace,
                     reinterpret_cast<unsigned char *>(&val),
                     1);
}

void squeezster::keyPressEvent(QKeyEvent* event)
{
    switch (event->key()) {
    case Qt::Key_F7:
        _DEBUG_ << "Volume down";
        event->accept();
        emit volumeDown();
        break;

    case Qt::Key_F8:
        _DEBUG_ << "Volume up";
        event->accept();
        emit volumeUp();
        break;
    }
    QWidget::keyPressEvent(event);
}
#endif

//----------------------------------------------
// Squeezster deconstructor
//----------------------------------------------
squeezster::~squeezster()
{
    _DEBUG_ << "Exit main()";
    //    delete nowPlayingDelegate;
    delete ui;
}

