#include "eventitem.h"
#include "ui_eventitem.h"
#include "profilewindow.h"

EventItem::EventItem(QWidget *parent, QString token, QNetworkAccessManager *nam) :
    QWidget(parent),
    ui(new Ui::EventItem),
    nam(nam),
    accessToken(token),
    qfacebook(new QFacebook(token, this))
{
    ui->setupUi(this);
    QPalette p = this->palette();
    p.setColor(QPalette::Foreground, p.highlight().color());
    ui->nameLabel->setPalette(p);
    connect(ui->rsvpBox, SIGNAL(activated(int)), this, SLOT(onRsvpChanged(int)));
    connect(ui->attendeesButton, SIGNAL(clicked()), this, SLOT(onAttendeesButtonClicked()));
    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));
    this->orientationChanged();
}

EventItem::~EventItem()
{
    delete ui;
}

void EventItem::setTime(QString time)
{
    QDateTime t = QDateTime::fromString(time, Qt::ISODate);
    t.setTimeSpec(Qt::UTC);
    QString date = QString(t.toLocalTime().toString("dddd, MMMM d %1 hh:mmap").arg(tr("at")));
    ui->timeLabel->setText(date);
}

void EventItem::setRsvp(QString rsvp)
{
    if (rsvp == "attending")
        ui->rsvpBox->setCurrentIndex(0);
    else if (rsvp == "maybe")
        ui->rsvpBox->setCurrentIndex(1);
    else if (rsvp == "declined")
        ui->rsvpBox->setCurrentIndex(2);
    else if (rsvp == "not_replied") {
        ui->rsvpBox->addItem(tr("Awaiting reply"));
        ui->rsvpBox->setCurrentIndex(3);
    }
}

void EventItem::setName(QString name)
{
    ui->nameLabel->setText(name);
}

void EventItem::setLocation(QString location)
{
    if (location.isEmpty())
        ui->locationLabel->hide();
    else
        ui->locationLabel->setText(location);
}

void EventItem::setUserId(QString id)
{
    m_userId = id;
}

void EventItem::updateAttendees()
{
    reply = qfacebook->getConnections(m_eventId, "invited");
    if (reply)
        connect(reply, SIGNAL(finished()), this, SLOT(onReplyReceived()));
}

void EventItem::onReplyReceived()
{
    if (reply->error() != QNetworkReply::NoError) {
        qDebug() << "Failed to attendee list!";
        reply->deleteLater();
        reply = 0;

        return;
    }

    QVariant jsonData = reply->data();
    qDebug() << jsonData;
    QVariantList listData;
    listData = jsonData.toMap().value("data").toList();
    foreach (QVariant jsonData, listData) {

    }
}

void EventItem::setEventId(QString id)
{
    m_eventId = id;
    this->setAvatarUrl(QString("https://graph.facebook.com/%1/picture&access_token=%2").arg(id, accessToken));
}

void EventItem::setAvatarUrl(QString url)
{
    avatarReply = nam->get(QNetworkRequest(url));
    if (avatarReply)
        connect(avatarReply, SIGNAL(finished()), this, SLOT(onAvatarReceived()));
}

void EventItem::appendAttendee(QString name, QString userId)
{
    attending.insert(name, userId);
}

void EventItem::onAvatarReceived()
{
    QUrl redir = avatarReply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl();
    if (!redir.isEmpty()) {
        avatarReply->deleteLater();
        avatarReply = nam->get(QNetworkRequest(redir));

        connect(avatarReply, SIGNAL(finished()), this, SLOT(onAvatarReceived()));
        return;
    }
    ui->avatarLabel->setPixmap(QPixmap::fromImage(QImage::fromData(avatarReply->readAll())));
    avatarReply->deleteLater();
}

void EventItem::onRsvpChanged(int rsvpOption)
{
    QString rsvp;
    switch (rsvpOption) {
    case 0:
        rsvp = "attending";
        break;
    case 1:
        rsvp = "maybe";
        break;
    case 2:
        rsvp = "declined";
        break;
    case 3:
        break;
    }

    ui->rsvpBox->setEnabled(false);

    rsvpReply = qfacebook->putObject(m_eventId, rsvp, 0);
    if (rsvpReply)
        connect(rsvpReply, SIGNAL(finished()), this, SLOT(onRsvpReplyReceived()));
}

void EventItem::onRsvpReplyReceived()
{
    ui->rsvpBox->setEnabled(true);
    if (ui->rsvpBox->count() == 4)
        ui->rsvpBox->removeItem(3);
}

void EventItem::onAttendeesButtonClicked()
{
    ui->attendeesButton->blockSignals(true);
    ui->attendeesButton->setDisabled(true);
    attendingReply = qfacebook->getObject(m_eventId + "/attending");
    if (attendingReply)
        connect(attendingReply, SIGNAL(finished()), this, SLOT(onAttendeesReplyReceived()));
}

void EventItem::onAttendeesReplyReceived()
{
    QDialog *dialog = new QDialog(this);
    QVBoxLayout *layout = new QVBoxLayout(dialog);
    QListWidget *listWidget = new QListWidget(this);
    connect(listWidget, SIGNAL(itemActivated(QListWidgetItem*)), this, SLOT(onAttendeeClicked(QListWidgetItem*)));
    layout->addWidget(listWidget);
    dialog->setLayout(layout);
#ifdef Q_WS_MAEMO_5
    dialog->setMinimumHeight(600);
#endif

    QVariant jsonData = attendingReply->data();
    QVariantList listData = jsonData.toMap().value("data").toList();
    foreach (jsonData, listData) {
        QListWidgetItem *item = new QListWidgetItem(listWidget);
        item->setData(Qt::UserRole, jsonData.toMap().value("id").toString());
        item->setText(jsonData.toMap().value("name").toString());
    }
    dialog->setWindowTitle(tr("Attendees"));
    dialog->setAttribute(Qt::WA_DeleteOnClose);
    dialog->show();
    ui->attendeesButton->setDisabled(false);
    ui->attendeesButton->blockSignals(false);
}

void EventItem::onAttendeeClicked(QListWidgetItem *item)
{
    ProfileWindow *profile = new ProfileWindow(this, accessToken);
    profile->setWindowTitle(item->text());
    profile->browseProfile(item->data(Qt::UserRole).toString());
#ifdef Q_WS_S60
    profile->showMaximized();
#else
    profile->show();
#endif
}

void EventItem::orientationChanged()
{
    QRect screenGeometry = QApplication::desktop()->screenGeometry();
    if (screenGeometry.width() > screenGeometry.height()) {
        ui->layout->setDirection(QBoxLayout::LeftToRight);
    } else {
        ui->layout->setDirection(QBoxLayout::TopToBottom);
    }
}
