#include "friendswindow.h"
#include "ui_friendswindow.h"

FriendsWindow::FriendsWindow(QWidget *parent, QString accessToken) :
    QMainWindow(parent),
    ui(new Ui::FriendsWindow),
    token(accessToken),
    qfacebook(new QFacebook(accessToken, this)),
    nam(new QNetworkAccessManager())
{
    ui->setupUi(this);
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5StackedWindow);
    ui->friendsList->setIconSize(QSize(64, 64));
#endif
    ui->searchWidget->hide();
#ifdef Q_WS_MAEMO_5
    ui->searchHideButton->setIcon(QIcon::fromTheme("general_close"));
#else
    ui->searchHideButton->setText("X");
#endif
    m_isDialog = false;
    m_shouldFetchAvatars = QSettings().value("friends/fetch-avatars", true).toBool();
    m_isRequestingFriendList = false;
    connect(nam, SIGNAL(finished(QNetworkReply*)), this, SLOT(onAvatarReplyFinished(QNetworkReply*)));
    connect(ui->friendsList, SIGNAL(itemActivated(QListWidgetItem*)), this, SLOT(onFriendClicked(QListWidgetItem*)));
    connect(ui->actionRefresh, SIGNAL(triggered()), this, SLOT(refresh()));
    connect(ui->searchEdit, SIGNAL(textChanged(QString)), this, SLOT(onSearchTextChanged(QString)));
    connect(ui->searchHideButton, SIGNAL(clicked()), ui->searchWidget, SLOT(hide()));
    connect(ui->searchHideButton, SIGNAL(clicked()), ui->searchEdit, SLOT(clear()));
    connect(ui->stackedWidget, SIGNAL(currentChanged(int)), this, SLOT(onStackedWidgetCurrentChanged(int)));
    connect(ui->requestsRadio, SIGNAL(clicked(bool)), this, SLOT(onRadioButtonClicked(bool)));
    connect(ui->friendsRadio, SIGNAL(clicked(bool)), this, SLOT(onRadioButtonClicked(bool)));
    connect(ui->pagesRadio, SIGNAL(clicked(bool)), this, SLOT(onRadioButtonClicked(bool)));
    connect(ui->requestsList, SIGNAL(itemActivated(QListWidgetItem*)), this, SLOT(onRequestClicked(QListWidgetItem*)));
    connect(ui->pagesList, SIGNAL(itemActivated(QListWidgetItem*)), this, SLOT(onPageClicked(QListWidgetItem*)));
}

FriendsWindow::~FriendsWindow()
{
    if (m_isRequestingFriendList)
        m_checkedIdList.clear();
    delete ui;
}

void FriendsWindow::onPageClicked(QListWidgetItem *item)
{
    ProfileWindow *profile = new ProfileWindow(this, token, ProfileWindow::GroupProfile);
    profile->browseProfile(item->data(Qt::UserRole).toString());
    profile->setWindowTitle(item->text());
#ifdef Q_WS_S60
    profile->showMaximized();
#else
    profile->show();
#endif
}

void FriendsWindow::refresh()
{
    int currentIndex = ui->stackedWidget->currentIndex();
    if (currentIndex == 0)
        updateFriends();
    else if (currentIndex == 1)
        getPagesList();
    else if (currentIndex == 2)
        getFriendRequests();
}

void FriendsWindow::onStackedWidgetCurrentChanged(int)
{
    ui->searchEdit->setText("");
    QList<QListWidget*> list;
    list << ui->friendsList;
    list << ui->pagesList;
    list << ui->requestsList;
    foreach (QListWidget *listWidget, list) {
        for (int i=0; i < listWidget->count(); i++)
            listWidget->item(i)->setHidden(false);
    }
}

void FriendsWindow::onRadioButtonClicked(bool)
{
    if (ui->friendsRadio->isChecked()) {
        ui->stackedWidget->setCurrentIndex(0);
    } else if (ui->pagesRadio->isChecked()) {
        ui->stackedWidget->setCurrentIndex(1);
        if (ui->pagesList->count() == 0)
            getPagesList();
    } else if (ui->requestsRadio->isChecked()) {
        ui->stackedWidget->setCurrentIndex(2);
        if (ui->requestsList->count() == 0)
            getFriendRequests();
    }
}

void FriendsWindow::getPagesList()
{
    pagesReply = qfacebook->getConnections("me", "groups");
    if (pagesReply)
        connect(pagesReply, SIGNAL(finished()), this, SLOT(onGotPagesList()));
}

void FriendsWindow::onGotPagesList()
{
    ui->pagesList->clear();

    QVariant jsonData = pagesReply->data();
    QVariantList jsonList = jsonData.toMap().value("data").toList();
    foreach (QVariant jsonItem, jsonList) {
        if (jsonItem.toMap().value("version").toInt() == 0) {
            QListWidgetItem *item = new QListWidgetItem(ui->pagesList);
            item->setData(Qt::UserRole, jsonItem.toMap().value("id").toString());
            item->setText(jsonItem.toMap().value("name").toString());
        }
    }

    pagesReply->deleteLater();
    pagesReply = 0;
}

void FriendsWindow::getFriendRequests()
{
    QString url = "https://api.facebook.com/method/notifications.get?format=JSON&access_token=" + token;
    requestsReply = nam->get(QNetworkRequest(url));
    if (requestsReply)
        connect(requestsReply, SIGNAL(finished()), this, SLOT(onGotFriendRequests()));
}

void FriendsWindow::onGotFriendRequests()
{
    ui->requestsList->clear();

    QJson::Parser parser;
    QVariant jsonData = parser.parse(requestsReply->readAll());
    QVariantList requestsList = jsonData.toMap().value("friend_requests").toList();
    foreach (QVariant userId, requestsList) {
        QFacebookReply *reply = qfacebook->getObject(userId.toString() + "&fields=name");
        connect(reply, SIGNAL(finished()), this, SLOT(onGotNameReply()));
        QListWidgetItem *item = new QListWidgetItem();
        Friend *friendObject = new Friend(this, userId.toString());
        connect(friendObject, SIGNAL(avatarProcessed(QString,QImage)), this, SLOT(onRequestAvatarProcessed(QString,QImage)));
        QString avatarUrl = QString("http://graph.facebook.com/%1/picture").arg(userId.toString());
        QNetworkReply *avatarReply = nam->get(QNetworkRequest(avatarUrl));
        connect(avatarReply, SIGNAL(finished()), friendObject, SLOT(avatarReceived()));
        item->setData(Qt::UserRole, userId.toString());
        item->setText(userId.toString());
        ui->requestsList->addItem(item);
    }

    requestsReply->deleteLater();
    requestsReply = 0;
}

void FriendsWindow::onRequestAvatarProcessed(QString id, QImage image)
{
    for (int i = 0; i < ui->requestsList->count(); i++) {
        qDebug() << id << ui->requestsList->item(i)->data(Qt::UserRole).toString();
        if (ui->requestsList->item(i)->data(Qt::UserRole).toString() == id)
            ui->requestsList->item(i)->setIcon(QIcon(QPixmap::fromImage(image)));
    }
}

void FriendsWindow::onGotNameReply()
{
    QFacebookReply *reply = qobject_cast<QFacebookReply*>(sender());
    QVariant jsonData = reply->data();
    QString name = jsonData.toMap().value("name").toString();
    QString id = jsonData.toMap().value("id").toString();
    for (int i=0; i < ui->requestsList->count(); i++) {
        if (ui->requestsList->item(i)->text() == id) {
            ui->requestsList->item(i)->setText(name);
            return;
        }
    }

    reply->deleteLater();
    reply = 0;
}

void FriendsWindow::onRequestClicked(QListWidgetItem *item)
{
    QString url = "http://www.facebook.com/dialog/friends/?id=" + item->data(Qt::UserRole).toString()
            + "&app_id=" + appId +
            "&redirect_uri=https://www.facebook.com/connect/login_success.html";
    qDebug() << url;
    browser = new FacebookBrowser(this);
    connect(browser->webView(), SIGNAL(urlChanged(QUrl)), this, SLOT(onUrlChanged(QUrl)));
    browser->openPage(url);
    browser->show();
}

void FriendsWindow::onUrlChanged(QUrl url)
{
    QString urlString = url.toString();
    if (urlString.contains("response/?action")) {
        QString response;
        if (urlString.endsWith("1"))
            response = tr("Friend request accepted");
        else if (urlString.endsWith("0"))
            response = tr("Friend request ignored");
#ifdef Q_WS_MAEMO_5
        QMaemo5InformationBox::information(this, response);
#else
        QMessageBox::information(this, tr("Friend request"), response);
#endif
        browser->close();
    }
}

void FriendsWindow::updateFriends()
{
    this->browseId(this->idToBrowse);
}

void FriendsWindow::browseId(QString id)
{
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
#endif
    this->idToBrowse = id;

    if (id == "me" && QSettings().contains("friends/list")) {
        qDebug() << "FriendsWindow: Cache exists, loading from there...";
        m_friendsList = QSettings().value("friends/list");
        this->parseData(m_friendsList);
    }

    this->reply = qfacebook->getConnections(this->idToBrowse, "friends");
    if (this->reply)
        connect(reply, SIGNAL(finished()), this, SLOT(onReplyFinished()));
}

void FriendsWindow::onReplyFinished()
{
    if (this->reply->error() != QNetworkReply::NoError) {
        qDebug() << "Failed to fetch friends!";
        this->reply->deleteLater();
        this->reply = 0;

        return;
    }

    QVariant jsonData = this->reply->data();
    if (idToBrowse == "me") {
        if (m_friendsList == jsonData) {
            qDebug() << "FriendsWindow: Friends list cache seems up to date...";
            return;
        } else {
            qDebug() << "FriendsWindow: Either there's no cache, or you gained or lost a friend, updating cache...";
            QSettings().setValue("friends/list", jsonData);
        }
    }

    this->parseData(jsonData);
}

void FriendsWindow::parseData(QVariant jsonData)
{
    ui->friendsList->clear();

    QThread *thread = new QThread;
    FriendParser *parser = new FriendParser(jsonData, m_isDialog, m_shouldFetchAvatars);
    parser->moveToThread(thread);

    connect(thread, SIGNAL(started()), parser, SLOT(start()));
    connect(thread, SIGNAL(finished()), parser, SLOT(deleteLater()));
    connect(thread, SIGNAL(finished()), thread, SLOT(deleteLater()));
    connect(parser, SIGNAL(avatarReceived(QString,QImage)), this, SLOT(onAvatarProcessed(QString,QImage)));
    connect(parser, SIGNAL(itemParsed(QListWidgetItem*)), this, SLOT(onItemParsed(QListWidgetItem*)));

    thread->start();

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
#endif
}

void FriendsWindow::onItemParsed(QListWidgetItem *item)
{
    ui->friendsList->addItem(item);
}

void FriendsWindow::onAvatarProcessed(QString id, QImage image)
{
    for (int i = 0; i < ui->friendsList->count(); i++) {
        if (ui->friendsList->item(i)->data(Qt::UserRole).toString() == id)
            ui->friendsList->item(i)->setIcon(QIcon(QPixmap::fromImage(image)));
    }
}

void FriendsWindow::onAvatarReplyFinished(QNetworkReply *reply)
{
    QString id = reply->objectName();
    for (int i = 0; i < ui->friendsList->count(); i++) {
        if (ui->friendsList->item(i)->data(Qt::UserRole).toString() == id)
            ui->friendsList->item(i)->setIcon(QIcon(QPixmap::fromImage(QImage::fromData(reply->readAll()))));
    }
}

void FriendsWindow::requestRecepientsList()
{
    m_isRequestingFriendList = true;
    ui->widget->hide();
    ui->friendsList->setSelectionMode(QListWidget::MultiSelection);
    disconnect(ui->friendsList, SIGNAL(itemActivated(QListWidgetItem*)), this, SLOT(onFriendClicked(QListWidgetItem*)));
    connect(ui->friendsList, SIGNAL(itemClicked(QListWidgetItem*)), this, SLOT(onFriendClicked(QListWidgetItem*)));
    QPushButton *doneButton = new QPushButton(tr("Done"), this);
    connect(doneButton, SIGNAL(clicked()), this, SLOT(onRecepientsSelected()));
    ui->verticalLayout->addWidget(doneButton);
    this->setWindowFlags(Qt::Dialog);
    m_isDialog = true;
    this->browseId("me");
#ifdef Q_WS_S60
    this->showMaximized();
#else
    this->show();
#endif
}

void FriendsWindow::requestFriend()
{
    m_isRequestingFriendList = true;
    ui->widget->hide();
    disconnect(ui->friendsList, SIGNAL(itemActivated(QListWidgetItem*)), this, SLOT(onFriendClicked(QListWidgetItem*)));
    connect(ui->friendsList, SIGNAL(itemActivated(QListWidgetItem*)), this, SLOT(onRecepientsSelected()));
    this->setWindowFlags(Qt::Dialog);
    m_isDialog = true;
    this->browseId("me");
#ifdef Q_WS_S60
    this->showMaximized();
#else
    this->show();
#endif
}

void FriendsWindow::onRecepientsSelected()
{
    emit recepientsSelected(ui->friendsList->selectedItems());
    this->close();
}

void FriendsWindow::onFriendClicked(QListWidgetItem *item)
{
    QString id = item->data(Qt::UserRole).toString();
    if (m_isRequestingFriendList) {
        if (item->isSelected()) {
            if (!m_checkedIdList.contains(id))
                m_checkedIdList.append(id);
        } else {
            if (m_checkedIdList.contains(id))
                m_checkedIdList.removeOne(id);
        }
    } else {
        ProfileWindow *window = new ProfileWindow(this, token);
        window->setAttribute(Qt::WA_DeleteOnClose);
        window->browseProfile(id);
        window->setWindowTitle(item->text());
#ifdef Q_WS_S60
        window->showMaximized();
#else
        window->show();
#endif
    }
}

void FriendsWindow::keyReleaseEvent(QKeyEvent *e)
{
    if (e->key() == Qt::Key_Enter || e->key() == Qt::Key_Left || e->key() == Qt::Key_Right || e->key() == Qt::Key_Backspace)
        return;
    else if ((e->key() == Qt::Key_Up || e->key() == Qt::Key_Down) && !ui->searchWidget->isHidden())
        ui->friendsList->setFocus();
    else {
        ui->friendsList->clearSelection();
        if (ui->searchWidget->isHidden())
            ui->searchWidget->show();
        if (!ui->searchEdit->hasFocus())
            ui->searchEdit->setText(ui->searchEdit->text() + e->text());
        ui->searchEdit->setFocus();
    }
    this->processCheckedList();
}

void FriendsWindow::onSearchTextChanged(QString text)
{
    QListWidget *listWidget = 0;
    int currentIndex = ui->stackedWidget->currentIndex();
    if (currentIndex == 0)
        listWidget = ui->friendsList;
    else if (currentIndex == 1)
        listWidget = ui->pagesList;
    else if (currentIndex == 2)
        listWidget = ui->requestsList;

    for (int i=0; i < listWidget->count(); i++) {
        if (listWidget->item(i)->text().toLower().indexOf(text.toLower()) == -1)
            listWidget->item(i)->setHidden(true);
        else
            listWidget->item(i)->setHidden(false);
    }

    if (text.isEmpty())
        ui->searchWidget->hide();

    if (currentIndex != 0)
        return;

    this->processCheckedList();
}

void FriendsWindow::processCheckedList()
{
    if (m_checkedIdList.isEmpty() || !m_isRequestingFriendList)
        return;

    /*
       Iterate over all friends in the list, check if the ID is in the list of checked IDs,
       if so, set the item as checked since setHidden(bool) clears QListWidgetItem::checkedState()
    */

    for (int i=0; i < ui->friendsList->count(); i++) {
        QString id = ui->friendsList->item(i)->data(Qt::UserRole).toString();
        if (m_checkedIdList.contains(id))
            ui->friendsList->item(i)->setSelected(true);
    }
}

Friend::Friend (QWidget *parent, QString id) :
    userId(id)
{
    this->setParent(parent);
}

Friend::~Friend() {}

void Friend::avatarReceived()
{
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());
    QUrl redir = reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl();
    if (!redir.isEmpty()) {
        reply->deleteLater();
        QString url = redir.toString();
        int pos = url.lastIndexOf("/");
        QString baseName = url.remove(0, pos);
        baseName.remove("/");

        QFile cachedImage(QDesktopServices::storageLocation(QDesktopServices::CacheLocation) + "/" + baseName);
        if (cachedImage.exists()) {
            cachedImage.open(QIODevice::ReadOnly);
            QByteArray data = cachedImage.readAll();
            cachedImage.close();
            emit avatarProcessed(this->userId, QImage::fromData(data));
            return;
        }
        QNetworkAccessManager *nam = qobject_cast<QNetworkAccessManager*>(sender()->parent());
        QNetworkReply *reply = nam->get(QNetworkRequest(redir));

        connect(reply, SIGNAL(finished()), this, SLOT(avatarReceived()));
        return;
    }
    QByteArray data = reply->readAll();
    emit avatarProcessed(this->userId, QImage::fromData(data));

    QString url = reply->url().toString();
    int pos = url.lastIndexOf("/");
    QString baseName = url.remove(0, pos);
    baseName.remove("/");
    QDir tempDir(QDesktopServices::storageLocation(QDesktopServices::CacheLocation));
    if (!tempDir.exists())
        tempDir.mkpath(QDesktopServices::storageLocation(QDesktopServices::CacheLocation));

    QString filename = QDesktopServices::storageLocation(QDesktopServices::CacheLocation) + "/" + baseName;
    QFile cachedImage(filename);
    cachedImage.open(QIODevice::WriteOnly);
    cachedImage.write(data);
    cachedImage.close();

    reply->deleteLater();
    this->deleteLater();
}

FriendParser::FriendParser(QVariant jsonData, bool isDialog, bool shouldFetchAvatars)
{
    nam = new QNetworkAccessManager(this);
    m_isDialog = isDialog;
    m_jsonData = jsonData;
    m_shouldFetchAvatars = shouldFetchAvatars;
}

void FriendParser::start()
{
    QVariant jsonData = m_jsonData;
    QVariantList listData;
    listData = jsonData.toMap().value("data").toList();
    foreach(jsonData, listData) {
        QListWidgetItem *item = new QListWidgetItem();
        item->setText(jsonData.toMap().value("name").toString());
        item->setData(Qt::UserRole, jsonData.toMap().value("id").toString());
        if ((m_isDialog && m_shouldFetchAvatars) || !m_isDialog) {
            Friend *user = new Friend(0, jsonData.toMap().value("id").toString());
            connect(user, SIGNAL(avatarProcessed(QString,QImage)), this, SLOT(onAvatarProcessed(QString,QImage)));
            QString avatarUrl = QString("http://graph.facebook.com/%1/picture").arg(jsonData.toMap().value("id").toString());
            QNetworkReply *reply = nam->get(QNetworkRequest(avatarUrl));
            connect(reply, SIGNAL(finished()), user, SLOT(avatarReceived()));
        }
        emit itemParsed(item);
    }
}

void FriendParser::onAvatarProcessed(QString id, QImage avatar)
{
    emit avatarReceived(id, avatar);
}
