#include "mainwindow.h"
#include "ui_mainwindow.h"
#ifdef Q_WS_MAEMO_5
    #include <X11/Xlib.h>
    #include <QX11Info>
#endif

QDataStream &operator<<(QDataStream &out, const FavoriteItem &item)
{
    out << item.text << item.avatar<< item.userId;
    return out;
}

QDataStream &operator>>(QDataStream &in, FavoriteItem &item)
{
    in >> item.text >> item.avatar >> item.userId;
    return in;
}

bool operator==(const FavoriteItem &item1, const FavoriteItem &item2)
{
    return item1.userId == item2.userId;
}

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow),
    newsFeedWindow(0),
    friendsWindow(0),
    albumsWindow(0),
    profileWindow(0),
    messagesWindow(0),
    groupsWindow(0),
    loginWindow(0),
    eventsWindow(0),
    chatWindow(0),
    notesWindow(0),
    notificationsDialog(0),
    nam(0)
{
    ui->setupUi(this);
    defaultItemCount = ui->listWidget->count();
    isQuitting = false;
#ifdef Q_WS_MAEMO_5
    if (!QDBusConnection::sessionBus().registerService(DBUS_SERVICE)) {
        qWarning("%s", qPrintable(QDBusConnection::sessionBus().lastError().message()));
    }

    if (!QDBusConnection::sessionBus().registerObject(DBUS_PATH, this, QDBusConnection::ExportScriptableSlots)) {
        qWarning("%s", qPrintable(QDBusConnection::sessionBus().lastError().message()));
    }
    ui->menuBar->addAction(tr("Quit"), this, SLOT(quit()));

    setAttribute(Qt::WA_Maemo5StackedWindow);
    setAttribute(Qt::WA_Maemo5AutoOrientation);
#endif
    ThumbnailItemDelegate *delegate = new ThumbnailItemDelegate(ui->listWidget);
    ui->listWidget->setItemDelegate(delegate);
    ui->listWidget->setContextMenuPolicy(Qt::CustomContextMenu);

    m_refreshTimer = new QTimer(this);

    this->connectSignals();
    if (!QSettings().contains("main/token") || !QSettings().contains("main/myId")) {
        this->logout();
        this->createLoginWindow();
    } else {
        this->accessToken = QSettings().value("main/token").toString();
        this->myName = QSettings().value("main/myName").toString();
        this->myId = QSettings().value("main/myId").toString();
        notificationsDialog = new NotificationsDialog(this, accessToken);
#ifdef Q_WS_S60
        connect(ui->pushButton, SIGNAL(clicked()), notificationsDialog, SLOT(showMaximized()));
#else
        connect(ui->pushButton, SIGNAL(clicked()), notificationsDialog, SLOT(show()));
#endif
        connect(notificationsDialog, SIGNAL(gotNotificationsCount(int)), this, SLOT(onGotNotificationCount(int)));
        connect(this, SIGNAL(refresh()), notificationsDialog, SLOT(updateNotifications()));
        connect(this, SIGNAL(refresh()), this, SLOT(getNotifications()));
        this->getNotifications();
    }
    int interval = QSettings().value("main/refresh-interval", 0).toInt();
    if (interval != 0) {
        m_refreshTimer->setInterval(interval);
        m_refreshTimer->start();
    }
    this->loadFavorites();
#ifdef Q_WS_MAEMO_5
    QTimer::singleShot(1200, this, SLOT(takeScreenshot()));
#endif
}

MainWindow::~MainWindow()
{
    this->saveFavorites();
    delete ui;
}

void MainWindow::refreshAll()
{
    if (QNetworkConfigurationManager(this).isOnline())
        emit refresh();
}

void MainWindow::saveFavorites()
{
    favorites.clear();

    if (ui->listWidget->count() <= defaultItemCount) {
        QSettings().remove("favorites/list");
        return;
    }

    for (int i = defaultItemCount; i < ui->listWidget->count(); i++) {
        FavoriteItem item;
        item.text = ui->listWidget->item(i)->text();
        item.avatar = ui->listWidget->item(i)->icon();
        item.userId = ui->listWidget->item(i)->data(Qt::UserRole).toString().remove("profile://");
        favorites.append(item);
    }

    if (!favorites.isEmpty()) {
        QByteArray array;
        QDataStream stream(&array, QIODevice::WriteOnly);
        stream << favorites;
        QSettings().setValue("favorites/list", array);
    }
}

void MainWindow::loadFavorites()
{
    if (!QSettings().contains("favorites/list"))
        return;

    QByteArray array = QSettings().value("favorites/list").toByteArray();
    QDataStream stream(&array, QIODevice::ReadOnly);
    stream >> favorites;

    if (favorites.isEmpty()) {
        QSettings().remove("favorites/list");
        return;
    }

    foreach (FavoriteItem item, favorites) {
        QListWidgetItem *favItem = new QListWidgetItem(ui->listWidget);
        favItem->setText(item.text);
        favItem->setData(Qt::UserRole, QString("profile://" + item.userId));
        favItem->setIcon(item.avatar);
        ui->listWidget->addItem(favItem);
    }
}

void MainWindow::connectSignals()
{
    connect(ui->actionAbout_Qt, SIGNAL(triggered()), qApp, SLOT(aboutQt()));
    connect(ui->actionAbout, SIGNAL(triggered()), this, SLOT(showAbout()));
    connect(ui->actionAdd_a_favorite, SIGNAL(triggered()), this, SLOT(onAddToFavoritesClicked()));
    connect(ui->actionApplication_settings, SIGNAL(triggered()), this, SLOT(showSettings()));
    connect(ui->actionLogout, SIGNAL(triggered()), this, SLOT(logout()));
    connect(ui->actionAccount_settings, SIGNAL(triggered()), this, SLOT(openAccountSettings()));
    connect(ui->actionHelp_Center, SIGNAL(triggered()), this, SLOT(openHelpCenter()));
    connect(ui->actionPrivacy_settings, SIGNAL(triggered()), this, SLOT(openPrivacySettings()));
    connect(ui->listWidget, SIGNAL(itemActivated(QListWidgetItem*)), this, SLOT(onItemActivated(QListWidgetItem*)));
    connect(ui->listWidget, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(onContextMenuRequested(QPoint)));
    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));
    connect(m_refreshTimer, SIGNAL(timeout()), this, SLOT(refreshAll()));
#ifdef Q_WS_MAEMO_5
    QDBusConnection::sessionBus().connect("com.nokia.HildonDesktop.AppMgr",
                                          "/com/nokia/HildonDesktop/AppMgr",
                                          "com.nokia.HildonDesktop.AppMgr",
                                          "LaunchApplication", this, SLOT(onHildonLaunch(const QDBusMessage &message)));
#endif
}

void MainWindow::showSettings()
{
    SettingsDialog *dialog = new SettingsDialog(this);
    dialog->exec();

    int interval = QSettings().value("main/refresh-interval", 0).toInt();
    if (interval != 0) {
        m_refreshTimer->setInterval(interval);
        m_refreshTimer->start();
    } else if (m_refreshTimer->isActive())
        m_refreshTimer->stop();
}

void MainWindow::onContextMenuRequested(QPoint point)
{
    // Don't show the menu for default icons
    if (ui->listWidget->currentRow() <= 8)
        return;

    QMenu *contextMenu = new QMenu(this);
    contextMenu->addAction("Delete favorite", this, SLOT(onFavoriteDeleteRequested()));
    contextMenu->exec(point);
    contextMenu->deleteLater();
}

void MainWindow::onFavoriteDeleteRequested()
{
    QListWidgetItem *item = ui->listWidget->currentItem();
    ui->listWidget->removeItemWidget(item);
    delete item;
}

void MainWindow::showAbout()
{
    AboutDialog *about = new AboutDialog(this);
#ifdef Q_WS_S60
    about->showMaximized();
#else
    about->show();
#endif
}

void MainWindow::onItemActivated(QListWidgetItem *item)
{
    if (item->text() == tr("News Feed")) {
        if (!newsFeedWindow) {
            newsFeedWindow = new NewsFeedWindow(this, this->accessToken);
            connect(this, SIGNAL(refresh()), newsFeedWindow, SLOT(updateNewsFeed()));
        }
#ifdef Q_WS_S60
        newsFeedWindow->showMaximized();
#else
        newsFeedWindow->show();
#endif
    } else if (item->text() == tr("Messages")) {
        if (!messagesWindow)
            messagesWindow = new MessagesWindow(this, this->accessToken);
#ifdef Q_WS_S60
        messagesWindow->showMaximized();
#else
        messagesWindow->show();
#endif
    } else if (item->text() == tr("Photos")) {
        if (!albumsWindow) {
            albumsWindow = new AlbumsWindow(this, this->accessToken);
            albumsWindow->browseAlbums("me");
        }
#ifdef Q_WS_S60
        albumsWindow->showMaximized();
#else
        albumsWindow->show();
#endif
    } else if (item->text() == tr("Profile")) {
        if (!profileWindow) {
            profileWindow = new ProfileWindow(this, this->accessToken);
            profileWindow->browseProfile("me");
            profileWindow->setWindowTitle(this->myName);
        }
#ifdef Q_WS_S60
        profileWindow->showMaximized();
#else
        profileWindow->show();
#endif
    }
    else if (item->text() == tr("Friends")) {
        if (!friendsWindow) {
            friendsWindow = new FriendsWindow(this, this->accessToken);
#ifdef Q_WS_S60
            friendsWindow->showMaximized();
#else
            friendsWindow->show();
#endif
            friendsWindow->browseId("me");
        } else {
#ifdef Q_WS_S60
        friendsWindow->showMaximized();
#else
        friendsWindow->show();
#endif
        }
    } else if (item->text() == tr("Groups")) {
        if (!groupsWindow) {
            groupsWindow = new GroupsWindow(this, this->accessToken);
            groupsWindow->browseGroups("me");
        }
#ifdef Q_WS_S60
        groupsWindow->showMaximized();
#else
        groupsWindow->show();
#endif
    } else if (item->text() == tr("Events")) {
        if (!eventsWindow) {
            eventsWindow = new EventsWindow(this, this->accessToken);
            eventsWindow->browseId("me");
        }
#ifdef Q_WS_S60
        eventsWindow->showMaximized();
#else
        eventsWindow->show();
#endif
    } else if (item->text() == tr("Places")) {
        PlacesWindow *placesWindow = new PlacesWindow(this, this->accessToken);
#ifdef Q_WS_S60
        placesWindow->showMaximized();
#else
        placesWindow->show();
#endif
    } else if (item->text() == tr("Chat")) {
        if (!chatWindow) {
            chatWindow = new ChatWindow(this);
            connect(chatWindow, SIGNAL(loginCanceled()), this, SLOT(onChatLoginCanceled()));
        } else {
#ifdef Q_WS_S60
            chatWindow->showMaximized();
#else
            chatWindow->show();
#endif
        }
    } else if (item->text() == tr("Notes")) {
        if (!notesWindow) {
            notesWindow = new NotesWindow(this, this->accessToken);
            notesWindow->browseNotes("me");
        }
#ifdef Q_WS_S60
        notesWindow->showMaximized();
#else
        notesWindow->show();
#endif
    } else if (item->data(Qt::UserRole).toString().startsWith("profile://")) {
        ProfileWindow *window = new ProfileWindow(this, this->accessToken);
        window->setWindowTitle(item->text());
        window->browseProfile(item->data(Qt::UserRole).toString().remove("profile://"));
#ifdef Q_WS_S60
        window->showMaximized();
#else
        window->show();
#endif
    }
}

void MainWindow::onChatLoginCanceled()
{
    chatWindow->deleteLater();
    chatWindow = 0;
}

void MainWindow::orientationChanged()
{
    ui->listWidget->setFlow(ui->listWidget->flow());
}

void MainWindow::getNotifications()
{
    if (!nam)
        nam = new QNetworkAccessManager(this);
    QString url = QString("https://api.facebook.com/method/notifications.get&format=JSON&access_token=%1").arg(accessToken);
    requestsReply = nam->get(QNetworkRequest(url));
    if (requestsReply)
        connect(requestsReply, SIGNAL(finished()), this, SLOT(onGotNotifications()));
}

void MainWindow::onGotNotifications()
{
    if (requestsReply->error() != QNetworkReply::NoError) {
        qDebug() << "Failed to fetch requests/notifications!";
        requestsReply->deleteLater();
        requestsReply = 0;
        return;
    }

    QByteArray reply = requestsReply->readAll();
    QJson::Parser parser;
    QVariant jsonData = parser.parse(reply);
    int newMessagesCount = jsonData.toMap().value("messages").toMap().value("unread").toInt();
    int pokes = jsonData.toMap().value("pokes").toMap().value("unread").toInt();
    int friendRequests = jsonData.toMap().value("friend_requests_counts").toMap().value("unread").toInt();
    int eventInvitesCount = jsonData.toMap().value("event_invites").toList().count();
    qDebug() << QString("%1 unread messages").arg(newMessagesCount);
    qDebug() << QString("%1 pokes").arg(pokes);
    qDebug() << QString("%1 friend requests").arg(friendRequests);
    qDebug() << QString("%1 event invitations").arg(eventInvitesCount);
    // Since positions are hardcoded for now...
    ui->listWidget->item(2)->setData(Qt::UserRole, friendRequests);
    ui->listWidget->item(3)->setData(Qt::UserRole, newMessagesCount);
    ui->listWidget->item(6)->setData(Qt::UserRole, eventInvitesCount);
}

void MainWindow::onGotNotificationCount(int count)
{
    if (count == 0)
        ui->pushButton->setText(tr("Notifications"));
    else
        ui->pushButton->setText(tr("Notifications (%1)").arg(count));
}

void MainWindow::createLoginWindow()
{
    if (!loginWindow) {
        loginWindow = new LoginWindow(this);
        connect(loginWindow, SIGNAL(loggedIn(QString)), this, SLOT(onLoggedIn(QString)));
    }
#ifdef Q_WS_S60
    loginWindow->showMaximized();
#else
    loginWindow->show();
#endif
    QTimer::singleShot(100, loginWindow, SLOT(raise()));
}

void MainWindow::onLoggedIn(QString token)
{
    this->accessToken = token;
    this->getNotifications();
    notificationsDialog = new NotificationsDialog(this, accessToken);
    connect(ui->pushButton, SIGNAL(clicked()), notificationsDialog, SLOT(show()));
    connect(notificationsDialog, SIGNAL(gotNotificationsCount(int)), this, SLOT(onGotNotificationCount(int)));
    this->getMyId();
    QSettings().setValue("main/token", token);
    loginWindow = 0;
}

void MainWindow::logout()
{
#ifdef Q_WS_MAEMO_5
    QMaemo5InformationBox::information(this, tr("Logging out..."));
#endif
    QSettings().remove("main/token");
    QSettings().remove("main/myName");
    QSettings().remove("main/myId");
    QSettings().remove("friends/list");
    QSettings().remove("favorites/list");
    QSettings().remove("chat/username");
    QString cookieFilename = QDesktopServices::storageLocation(QDesktopServices::DataLocation) + "/facebook.cookies";
    QFile cookieFile(cookieFilename);
    if (cookieFile.exists())
        cookieFile.remove();
    this->deleteWindows();
    QTimer::singleShot(10, this, SLOT(createLoginWindow()));
}

void MainWindow::getMyId()
{
    qfacebook = new QFacebook(accessToken, this);
    this->reply = qfacebook->getObject("me&fields=id,name");
    if (this->reply)
        connect(reply, SIGNAL(finished()), this, SLOT(onGetMyIdReply()));
}

void MainWindow::onGetMyIdReply()
{
    QVariant jsonData = this->reply->data();
    this->myName = jsonData.toMap().value("name").toString();
    this->myId = jsonData.toMap().value("id").toString();
#ifdef Q_WS_MAEMO_5
    QMaemo5InformationBox::information(this, tr("Logged in as %1").arg(myName));
#endif
    QSettings().setValue("main/myName", this->myName);
    QSettings().setValue("main/myId", this->myId);
    disconnect(reply, SIGNAL(finished()), this, SLOT(onGetMyIdReply()));
    qfacebook->deleteLater();
}

void MainWindow::onAddToFavoritesClicked()
{
    FriendsWindow *window = new FriendsWindow(this, accessToken);
    window->setAttribute(Qt::WA_DeleteOnClose);
    connect(window, SIGNAL(recepientsSelected(QList<QListWidgetItem*>)), this, SLOT(onFriendSelected(QList<QListWidgetItem*>)));
    window->requestFriend();
    window->setWindowFlags(Qt::Dialog);
#ifdef Q_WS_S60
    window->showMaximized();
#else
    window->show();
#endif
}

void MainWindow::onFriendSelected(QList<QListWidgetItem*> list)
{
    if (list.count() == 0)
        return;

    QListWidgetItem *item = list.first();
    QListWidgetItem *favItem = new QListWidgetItem(ui->listWidget);
    Friend *user = new Friend(ui->listWidget, item->data(Qt::UserRole).toString());
    connect(user, SIGNAL(avatarProcessed(QString,QImage)), this, SLOT(onAvatarProcessed(QString,QImage)));
    QString avatarUrl = QString("http://graph.facebook.com/%1/picture&type=large").arg(item->data(Qt::UserRole).toString());
    if (!nam)
        nam = new QNetworkAccessManager(this);
    QNetworkReply *reply =
            nam->get(QNetworkRequest(avatarUrl));
    connect(reply, SIGNAL(finished()), user, SLOT(avatarReceived()));

    favItem->setText(item->text());
    favItem->setData(Qt::UserRole, QString("profile://" + item->data(Qt::UserRole).toString()));
}

void MainWindow::onAvatarProcessed(QString id, QImage image)
{
    QPixmap expandedPixmap = QPixmap::fromImage(image);
    expandedPixmap = expandedPixmap.scaled(ui->listWidget->size(), Qt::KeepAspectRatioByExpanding, Qt::SmoothTransformation);
    for (int i = 0; i < ui->listWidget->count(); i++) {
        if (ui->listWidget->item(i)->data(Qt::UserRole).toString().endsWith(id))
            ui->listWidget->item(i)->setIcon(QIcon(expandedPixmap));
    }
}

void MainWindow::deleteWindows()
{
    if (newsFeedWindow) {
        delete newsFeedWindow;
        newsFeedWindow = 0;
    }
    if (friendsWindow) {
        delete friendsWindow;
        friendsWindow = 0;
    }
    if (albumsWindow) {
        delete albumsWindow;
        albumsWindow = 0;
    }
    if (profileWindow) {
        delete profileWindow;
        profileWindow = 0;
    }
    if (chatWindow) {
        delete chatWindow;
        chatWindow = 0;
    }
    if (groupsWindow) {
        delete groupsWindow;
        groupsWindow = 0;
    }
    if (notesWindow) {
        delete notesWindow;
        notesWindow = 0;
    }
    if (messagesWindow) {
        delete messagesWindow;
        messagesWindow = 0;
    }
    if (notificationsDialog) {
        delete notificationsDialog;
        notificationsDialog = 0;
        ui->pushButton->setText(tr("Notifications"));
    }

    ui->listWidget->item(2)->setData(Qt::UserRole, 0);
    ui->listWidget->item(3)->setData(Qt::UserRole, 0);
    ui->listWidget->item(6)->setData(Qt::UserRole, 0);
}

#ifdef Q_WS_MAEMO_5
void MainWindow::takeScreenshot()
{
    // True takes a screenshot, false destroys it
    // See http://maemo.org/api_refs/5.0/5.0-final/hildon/hildon-Additions-to-GTK+.html#hildon-gtk-window-take-screenshot
    bool take = true;
    XEvent xev = { 0 };

    xev.xclient.type = ClientMessage;
    xev.xclient.serial = 0;
    xev.xclient.send_event = True;
    xev.xclient.display = QX11Info::display();
    xev.xclient.window = XDefaultRootWindow (xev.xclient.display);
    xev.xclient.message_type = XInternAtom (xev.xclient.display, "_HILDON_LOADING_SCREENSHOT", False);
    xev.xclient.format = 32;
    xev.xclient.data.l[0] = take ? 0 : 1;
    xev.xclient.data.l[1] = this->winId();

    XSendEvent (xev.xclient.display,
                xev.xclient.window,
                False,
                SubstructureRedirectMask | SubstructureNotifyMask,
                &xev);

    XFlush (xev.xclient.display);
    XSync (xev.xclient.display, False);
}

void MainWindow::onHildonLaunch(const QDBusMessage &message)
{
    qDebug() << message.arguments();
    this->show();
    if (message.arguments()[0] == QApplication::applicationName())
        this->top_application();
}
#endif

void MainWindow::top_application()
{
    if (this->isHidden()) {
#ifdef Q_WS_S60
        this->showMaximized();
#else
        this->show();
#endif
    }
    this->activateWindow();
}

void MainWindow::openAccountSettings()
{
    FacebookBrowser *browser = new FacebookBrowser(this);
    browser->openPage(FacebookBrowser::FacebookAccountSettings);
    browser->show();
}

void MainWindow::openPrivacySettings()
{
    FacebookBrowser *browser = new FacebookBrowser(this);
    browser->openPage(FacebookBrowser::FacebookPrivacySettings);
    browser->show();
}

void MainWindow::openHelpCenter()
{
    FacebookBrowser *browser = new FacebookBrowser(this);
    browser->openPage(FacebookBrowser::FacebookHelpCenter);
    browser->show();
}

void MainWindow::quit()
{
    isQuitting = true;
    this->close();
}

void MainWindow::closeEvent(QCloseEvent *e)
{
#ifdef Q_WS_MAEMO_5
    if (QSettings().value("main/hide-on-exit", false).toBool() && !isQuitting) {
        e->ignore();
        this->hide();
    } else {
        e->accept();
    }
#else
    e->accept();
#endif
}
