//    smssend
//    Copyright (C) 2011 Paolo Iommarini
//    sakya_tg@yahoo.it
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
#include "smssender.h"
#include <iostream>
#include <QRegExp>
#include <QMessageService>
using namespace QtMobility;

SmsSender::SmsSender(int argc, char *argv[]) :
        QCoreApplication(argc,argv)
{
    m_LogFile = NULL;
    m_Argc = argc - 1;
    for (int i=1; i<argc; i++)
        m_Argv.append(argv[i]);
}

SmsSender::~SmsSender()
{
    if (m_LogFile != NULL){
        m_LogFile->close();
        delete m_LogFile;
    }
}

void SmsSender::consoleOutput(QString output, bool newline)
{
    std::cout << qPrintable(output);
    if (newline)
        std::cout << std::endl;
    else
        std::cout.flush();
}

void SmsSender::printInfo(){
    consoleOutput(QString("smssend version %1").arg(QCoreApplication::applicationVersion()));
    consoleOutput("copyright(c) 2011 by Paolo Iommarini <sakya_tg@yahoo.it>");
}

void SmsSender::printUsage()
{
    consoleOutput("Usage:");
    consoleOutput("  smssend [options] -n +39xxxxxx:+39xxxxxx -m \"Text\"");
    consoleOutput("Options:");
    consoleOutput("  -c         : compose message with conversation");
    consoleOutput("  -l logfile : log pnatd messages to logfile");
    consoleOutput("               This option is useless if used with -c");
}

int SmsSender::exec()
{
    int res = errNoError;
    m_Compose = false;

    printInfo();
    if (m_Argc == 0){
        printUsage();
        return errArguments;
    }

    args currArg = SmsSender::argNone;
    QStringList numbers;
    QString message;

    foreach (QString param, m_Argv){
        if (param.startsWith("-")){
            currArg = SmsSender::argNone;
            if (param == "-n")
                currArg = SmsSender::argNumber;
            else if(param == "-m")
                currArg = SmsSender::argText;
            else if(param == "-l")
                currArg = SmsSender::argLog;
            else if (param == "-c")
                m_Compose = true;
            else{
                consoleOutput(QString("ERR: unrecognized argument: %1").arg(param));
                return errArguments;
            }
        }else{
            switch (currArg){
                case SmsSender::argNumber:
                    numbers.append(param.split(":"));
                    break;
                case SmsSender::argText:
                    message = param;
                    break;
                case SmsSender::argLog:
                    {
                        m_LogFile = new QFile(param);
                        if (!m_LogFile->open(QIODevice::Truncate | QIODevice::WriteOnly)){
                            consoleOutput(QString("ERR: cannot open file: %1").arg(param));
                            return errLogfileOpenFailed;
                        }
                    }
                default:
                    break;
            }

            currArg = SmsSender::argNone;
        }
    }

    bool ok = true;
    if (numbers.isEmpty()){
        consoleOutput("ERR: No receiver number specified");
        ok = false;
        res = errArguments;
    }

    if (message.isEmpty()){
        consoleOutput("ERR: No message text specified");
        ok = false;
        res = errArguments;
    }

    if (ok){
        QStringList messageBlocks;
        if (m_Compose || message.length() <= 160)
            messageBlocks.append(message);
        else{
            int parts = message.length() / 160;
            if (message.length() % 160 > 0)
                parts++;
            consoleOutput("WARNING: SMS exceding 160 characters");
            consoleOutput(QString("         Splitted in %1 parts").arg(parts));
            for (int i=0; i<parts - 1; i++)
                messageBlocks.append(message.mid(160*i, 160));
        }

        consoleOutput("Sendind SMS:");
        bool sent;
        int totalMsg = numbers.count() * messageBlocks.count();
        int i = 0;
        foreach(QString number, numbers){
            i++;
            QRegExp exp("\\+?[0-9]*");
            if (!exp.exactMatch(number)){
                consoleOutput(QString("ERR: Invalid receiver number %1").arg(number));
                continue;
            }

            consoleOutput(QString("[%1/%2]").arg(i).arg(totalMsg), false);
            foreach (QString msg, messageBlocks){
                if (m_Compose)
                    sent = compose(number, msg);
                else
                    sent = send(number, msg);

                if (!sent)
                    res = errSmsFailed;
            }
        }
    }

    return res;
}


bool SmsSender::compose(QString number, QString messageText)
{
    //Implementation using QtMobility (opens conversation after the message is moved to the sent folder):
    QMessage message;
    QMessageService service;

    message.setType(QMessage::Sms);
    message.setTo(QMessageAddress(QMessageAddress::Phone, number));
    message.setBody(messageText);

    if (service.compose(message)){
        consoleOutput(".");
        //consoleOutput(QString("OK: sms queued for sending to %1").arg(number));
        return true;
    }

    consoleOutput(QString("ERR: Failed to queue sms for sending to %1"));
    return false;
}

bool SmsSender::send(QString number, QString messageText)
{
    //Implementation using pnatd (must be run as root)
    //to run it as user:
    // chmod 4555 /usr/bin/pnatd
    Pty* pty = new Pty();
    QStringList args;
    if (!pty->run("pnatd", args)){
        consoleOutput(QString("ERR: Failed to run pnatd"));
        return false;
    }

    if (!writeCmd(pty, "AT\r", "OK")){
        //consoleOutput(QString("ERR: Failed to send at command"));
        //return false;
    }
    consoleOutput(".", false);

    //AT+CPIN="0000" <-- TODO: send if the SIM is pin-protected
    if (!writeCmd(pty, "AT+CMGF=1\r", "OK")){
        consoleOutput(QString("ERR: Failed to send at+cmgf command"));
        //return false;
    }
    consoleOutput(".", false);

    if (!writeCmd(pty, QString("AT+CMGS=\"%1\"\r").arg(number).toLatin1(), ">")){
        consoleOutput(QString("ERR: Failed to send at+cmgs command"));
        //return false;
    }
    consoleOutput(".", false);

    if (!writeCmd(pty, messageText, "")){
        consoleOutput(QString("ERR: Failed to send text message"));
        //return false;
    }
    consoleOutput(".", false);

    bool res = writeCmd(pty, QString("%1").arg(QChar(26)).toLatin1(), "OK", -1);
    if (!res)
        consoleOutput("FAILED");
    else
        consoleOutput("OK");

    delete pty;
    return res;
}

bool SmsSender::writeCmd(Pty* p, QString cmd, QString okReply, int msectimeout)
{
    if (m_LogFile != NULL)
        m_LogFile->write(cmd.toLatin1());

    QString fullRes;
    if (!p->writeCmd(&cmd, &fullRes, msectimeout, &okReply)){
        consoleOutput(QString("ERR: failed to write to pty"));
        return false;
    }
    if (m_LogFile != NULL)
        m_LogFile->write(fullRes.toLatin1());

    //consoleOutput(QString("INFO: pry reply \"%1\"").arg(fullRes));

    if (!okReply.isEmpty()){
        QStringList lines = fullRes.split(QRegExp("\r|\r\n"), QString::SkipEmptyParts);
        QString res = lines.last().trimmed();
        bool ok = res == okReply;
        //if (!ok)
        //    consoleOutput(QString("ERR: pty reply \"%1\"").arg(fullRes));
        return ok;
    }else{
        return true;
    }
}
