/*
 *  slide2answer
 *  Copyright (C) 2011 Nicolai Hess
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <stdio.h>
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <libintl.h>
#include <string.h>
#include <locale.h>
#include <gdk/gdkx.h>
#include <libosso-abook/osso-abook.h>

#include <telepathy-glib/svc-generic.h>
#include <telepathy-glib/svc-client.h>
#include <telepathy-glib/defs.h>
#include <telepathy-glib/connection.h>
#include <telepathy-glib/channel.h>
#include <telepathy-glib/account.h>
#include <libmcclient/mc-account.h>
#include <libmcclient/mc-profile.h>
#include <libebook/e-book-view.h>
#include <libebook/e-book-query.h>

#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Xutil.h>

#include "slide2answer.h"
#include "slide2answer_marshal.h"


#define SLIDE2ANSWER_STATUS_PLUGIN_GET_PRIVATE(obj)(G_TYPE_INSTANCE_GET_PRIVATE(obj, TYPE_SLIDE2ANSWER_STATUS_PLUGIN, Slide2AnswerStatusPluginPrivate))


#define 	MCE_SERVICE   "com.nokia.mce"
#define 	MCE_PATH   "/com/nokia/mce/signal"
#define 	MCE_INTERFACE   "com.nokia.mce.signal"
#define 	MCE_STATE   "sig_call_state_ind"
#define     CALL_SERVICE "com.nokia.csd.Call"
#define     CALL_PATH "/com/nokia/csd/call/1"
#define     CALL_INTERFACE "com.nokia.csd.Call.Instance"
#define     CALL_END_PATH "/com/nokia/csd/call"
#define     CALL_END_INTERFACE "com.nokia.csd.Call"
#define     CALL_ANSWER "Answer"
#define     CALL_END "Release"

#define 	AVATAR_SIZE 216

struct _Slide2AnswerStatusPluginPrivate
{
  GtkWidget* call_window;
  GtkWidget* call_window_avatar;
  GtkWidget* call_window_name_label;
  GtkWidget* call_window_number_label;
  GtkWidget* call_window_slider;
  gboolean is_voip_call;
  DBusGConnection *dbus_system_conn;
  DBusGProxy *dbus_system_proxy;
  OssoABookAggregator* abook_aggregator;
  OssoABookAccountManager* abook_account_manager;
  TpChannel* active_channel;
  GHashTable* account_map;
};

HD_DEFINE_PLUGIN_MODULE(Slide2AnswerStatusPlugin, slide2answer_status_plugin, HD_TYPE_STATUS_MENU_ITEM);

static void
_reject_call(Slide2AnswerStatusPlugin* plugin)
{
  if(plugin->priv->active_channel)
  {
    if(plugin->priv->is_voip_call)
    {
      GArray* member = g_array_new (FALSE, FALSE, sizeof (guint));
      guint i = 1;
      g_array_append_val(member, i);
      tp_cli_channel_interface_group_call_remove_members_with_reason(plugin->priv->active_channel,
								     -1,
								     member,
								     "",
								     0,
								     NULL,
								     NULL, 
								     NULL,
								     NULL);
      g_array_free(member, TRUE);
    }
    else
    {
      if(plugin->priv->dbus_system_conn)
      {
	DBusGProxy* proxy = 
	  dbus_g_proxy_new_for_name(plugin->priv->dbus_system_conn, 
				    CALL_SERVICE,
				    CALL_END_PATH, 
				    CALL_END_INTERFACE); 
	dbus_g_proxy_call_no_reply(proxy,
				   CALL_END,  
				   G_TYPE_INVALID, G_TYPE_INVALID);
	g_object_unref(proxy);
      }
    }
  }
  g_object_unref(plugin->priv->active_channel);
  plugin->priv->active_channel = NULL;
}

static void
_accept_call(Slide2AnswerStatusPlugin* plugin)
{
  if(plugin->priv->is_voip_call)
  {
    if(plugin->priv->active_channel)
    {
      GArray* member = g_array_new (FALSE, FALSE, sizeof (guint));
      guint i = 1;
      g_array_append_val(member, i);
      tp_cli_channel_interface_group_call_add_members(plugin->priv->active_channel,
						      -1,
						      member,
						      "",
						      NULL,
						      NULL, 
						      NULL,
						      NULL);
      g_array_free(member, TRUE);
    }
  }
  else
  {
    if(plugin->priv->dbus_system_conn)
    {
      DBusGProxy* proxy = 
	dbus_g_proxy_new_for_name(plugin->priv->dbus_system_conn, 
				  CALL_SERVICE,
				  CALL_PATH,
				  CALL_INTERFACE); 
      dbus_g_proxy_call_no_reply(proxy,
				 CALL_ANSWER,  
				 G_TYPE_INVALID, G_TYPE_INVALID);
      g_object_unref(proxy);
    }
  }
  g_object_unref(plugin->priv->active_channel);
  plugin->priv->active_channel = NULL;
}

static gboolean
_slider_changed(GtkRange* range,
		GtkScrollType scroll,
		gdouble value,
		gpointer user_data)
{
  if(IS_SLIDE2ANSWER_STATUS_PLUGIN(user_data))
  {
    Slide2AnswerStatusPlugin* plugin = SLIDE2ANSWER_STATUS_PLUGIN(user_data);
    if(value>0.9)
    {
      gtk_widget_hide(GTK_WIDGET(plugin->priv->call_window));
      _accept_call(plugin);
    }
    else if(value<-0.9)
    {
      gtk_widget_hide(GTK_WIDGET(plugin->priv->call_window));
      _reject_call(plugin);
    }
  }
  return FALSE;
}

static gboolean
_slider_changed_stopped(GtkWidget *widget,
			GdkEventButton *event,
			gpointer user_data)
{
  if(GTK_IS_RANGE(user_data))
  {
    gtk_range_set_value(GTK_RANGE(user_data), 0.0);
  }
  return FALSE;
}

static void
_create_slide2answer_window(Slide2AnswerStatusPlugin* status_plugin)
{
  status_plugin->priv->call_window = hildon_window_new();//gtk_window_new(GTK_WINDOW_TOPLEVEL);
  hildon_gtk_window_set_portrait_flags(GTK_WINDOW(status_plugin->priv->call_window),
				       HILDON_PORTRAIT_MODE_SUPPORT);
  /*
  GdkPixbuf* pixbuf = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
					       "general_default_avatar",
					       AVATAR_SIZE, GTK_ICON_LOOKUP_NO_SVG, NULL);
  */
  //  status_plugin->priv->call_window_avatar = gtk_image_new_from_pixbuf(pixbuf);
  
  status_plugin->priv->call_window_avatar = osso_abook_avatar_image_new();

  osso_abook_avatar_image_set_fallback_icon(OSSO_ABOOK_AVATAR_IMAGE(status_plugin->priv->call_window_avatar),
				      "general_default_avatar");
  //  g_object_unref(pixbuf);
  status_plugin->priv->call_window_name_label = gtk_label_new("");
  hildon_helper_set_logical_font(status_plugin->priv->call_window_name_label, "LargeSystemFont");
  status_plugin->priv->call_window_number_label = gtk_label_new("");
  hildon_helper_set_logical_font(status_plugin->priv->call_window_number_label, "SmallSystemFont");
  
  status_plugin->priv->call_window_slider = hildon_gtk_hscale_new();
  //gtk_range_set_jump_to_position(GTK_RANGE(status_plugin->priv->call_window_slider), FALSE);
  gtk_widget_set_name(status_plugin->priv->call_window_slider,
		      "bidirectional-slider-horizontal");
  gtk_range_set_range(GTK_RANGE(status_plugin->priv->call_window_slider), -1.0, 1.0);              
  gtk_range_set_value(GTK_RANGE(status_plugin->priv->call_window_slider), 0.0);
  
  GtkWidget* accept_button = hildon_button_new(HILDON_SIZE_FINGER_HEIGHT,
					       HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  GtkWidget* reject_button = hildon_button_new(HILDON_SIZE_FINGER_HEIGHT,
					       HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title(HILDON_BUTTON(accept_button),
			  dgettext("rtcom-call-ui", "call_bd_incoming_call_answer"));
  hildon_button_set_title(HILDON_BUTTON(reject_button),
			  dgettext("rtcom-call-ui", "call_bd_incoming_call_reject"));
  gtk_widget_set_name(accept_button, "hildon-accept-button-thumb");
  gtk_widget_set_name(reject_button, "hildon-reject-button-thumb");
  gtk_widget_set_sensitive(accept_button, FALSE);
  gtk_widget_set_sensitive(reject_button, FALSE);
  /*
  gtk_range_set_update_policy(GTK_RANGE(status_plugin->priv->call_window_slider),
			      GTK_UPDATE_DELAYED);
  */
  g_signal_connect(status_plugin->priv->call_window_slider, "change-value", G_CALLBACK(_slider_changed), status_plugin);
  g_signal_connect(status_plugin->priv->call_window_slider, "button-release-event", G_CALLBACK(_slider_changed_stopped), status_plugin->priv->call_window_slider);

  GtkWidget* contact_box = gtk_hbox_new(FALSE, 0);
  GtkWidget* contact_info_box = gtk_vbox_new(FALSE, 0);
  GtkWidget* button_box = gtk_hbox_new(FALSE, 0);
  GtkWidget* content = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(status_plugin->priv->call_window), 
		    content);
  gtk_box_pack_start(GTK_BOX(contact_box), status_plugin->priv->call_window_avatar, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(contact_info_box), status_plugin->priv->call_window_name_label, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(contact_info_box), status_plugin->priv->call_window_number_label, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(contact_box), contact_info_box, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(button_box), reject_button, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(button_box), accept_button, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(content), contact_box, TRUE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(content), status_plugin->priv->call_window_slider, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(content), button_box, TRUE, FALSE, 0);
  gtk_widget_show_all(content);
}

static void
_show_slide2answer_window(Slide2AnswerStatusPlugin* plugin)
{
  if(plugin->priv->call_window == NULL)
  {
    _create_slide2answer_window(plugin);
    gtk_widget_show_all(GTK_WIDGET(plugin->priv->call_window));
    gtk_window_fullscreen(GTK_WINDOW(plugin->priv->call_window));
    Display* xdisplay = GDK_WINDOW_XDISPLAY(plugin->priv->call_window->window);
    guint32 enable = {9};
    Atom winStackingAtom = gdk_x11_get_xatom_by_name("_HILDON_STACKING_LAYER");
    XChangeProperty(xdisplay,
     		    GDK_WINDOW_XWINDOW(plugin->priv->call_window->window),
     		    winStackingAtom,
     		    XA_CARDINAL,
     		    32, PropModeReplace, (guchar*) &enable, 1);
  }
  else
  {
    gtk_range_set_value(GTK_RANGE(plugin->priv->call_window_slider), 0.0);
    gtk_widget_show_all(GTK_WIDGET(plugin->priv->call_window));
  }
}

static void
handle_incoming_call(DBusGProxy *object,
		     const char* state, 
		     const char* emstate, 
		     gpointer user_data)
{
  Slide2AnswerStatusPlugin* plugin = SLIDE2ANSWER_STATUS_PLUGIN(user_data);

  if(g_strcmp0(state, "ringing") == 0)
  {
    //    _show_slide2answer_window(plugin);
  }
  else
  {
    gtk_widget_hide(GTK_WIDGET(plugin->priv->call_window));
  }
}

static void
unregister_incoming_call_handler(Slide2AnswerStatusPlugin* plugin)
{
  if(plugin->priv->dbus_system_conn)
  {
    if(plugin->priv->dbus_system_proxy)
    {
      dbus_g_proxy_disconnect_signal(plugin->priv->dbus_system_proxy, 
				     MCE_STATE, 
				     G_CALLBACK(handle_incoming_call), 
				     plugin);
      g_object_unref(plugin->priv->dbus_system_proxy);
      plugin->priv->dbus_system_proxy = NULL;
    }
  }
}

static void
register_incoming_call_handler(Slide2AnswerStatusPlugin* plugin)
{
  if(plugin->priv->dbus_system_conn == NULL)
    plugin->priv->dbus_system_conn = 
      hd_status_plugin_item_get_dbus_g_connection(HD_STATUS_PLUGIN_ITEM(&plugin->parent),
						  DBUS_BUS_SYSTEM, 
						  NULL);
  if(plugin->priv->dbus_system_conn)
  {
    plugin->priv->dbus_system_proxy = dbus_g_proxy_new_for_name(plugin->priv->dbus_system_conn,
								MCE_SERVICE,
								MCE_PATH,
								MCE_INTERFACE);
    dbus_g_object_register_marshaller(_slide2answer_marshal_VOID__STRING_STRING,
				      G_TYPE_NONE,
				      G_TYPE_STRING,
				      G_TYPE_STRING,
				      G_TYPE_INVALID);   

    dbus_g_proxy_add_signal(plugin->priv->dbus_system_proxy,    
			    MCE_STATE,
			    G_TYPE_STRING,
			    G_TYPE_STRING,
			    G_TYPE_INVALID);
    dbus_g_proxy_connect_signal(plugin->priv->dbus_system_proxy,
				MCE_STATE,
				G_CALLBACK(handle_incoming_call), plugin, NULL);
  }
}

static void
_set_unknowncaller_data(Slide2AnswerStatusPlugin* status_plugin, const gchar* identifier)
{
  gtk_label_set_text(GTK_LABEL(status_plugin->priv->call_window_name_label),
		             identifier);
  gtk_label_set_text(GTK_LABEL(status_plugin->priv->call_window_number_label),
		             "");
  osso_abook_avatar_image_set_avatar(OSSO_ABOOK_AVATAR_IMAGE(status_plugin->priv->call_window_avatar),
				     NULL);
  /* GdkPixbuf* pixbuf = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(), */
  /* 					       "general_default_avatar", */
  /* 					       216, GTK_ICON_LOOKUP_NO_SVG, NULL); */
  /* osso_abook_avatar_image_set_pixbuf(OSSO_ABOOK_AVATAR_IMAGE(status_plugin->priv->call_window_avatar), */
  /* 				     pixbuf); */
  /* g_object_unref(pixbuf); */

}

static void
_set_contact_data(Slide2AnswerStatusPlugin* status_plugin, OssoABookContact* contact, const gchar* identifier)
{
  gtk_label_set_text(GTK_LABEL(status_plugin->priv->call_window_name_label),
		     osso_abook_contact_get_display_name(contact));
  gtk_label_set_text(GTK_LABEL(status_plugin->priv->call_window_number_label),
  		     identifier);      
  osso_abook_avatar_image_set_avatar(OSSO_ABOOK_AVATAR_IMAGE(status_plugin->priv->call_window_avatar),
				     OSSO_ABOOK_AVATAR(contact));
}

static void
handle_new_channel(TpConnection *proxy,
		   const gchar *arg_Object_Path,
		   const gchar *arg_Channel_Type,
		   guint arg_Handle_Type,
		   guint arg_Handle,
		   gboolean arg_Suppress_Handler,
		   gpointer user_data,
		   GObject *weak_object)
{
  if(arg_Suppress_Handler || 
     g_strcmp0(arg_Channel_Type, "org.freedesktop.Telepathy.Channel.Type.Text") == 0)
    return;
  
  Slide2AnswerStatusPlugin* plugin = SLIDE2ANSWER_STATUS_PLUGIN(user_data);
  plugin->priv->is_voip_call = !g_str_has_prefix(arg_Object_Path, "/org/freedesktop/Telepathy/Connection/ring/tel/ring");
  TpChannel* channel = tp_channel_new (proxy, 
				       arg_Object_Path, 
				       arg_Channel_Type,
				       arg_Handle_Type, 
				       arg_Handle, NULL);
  tp_channel_run_until_ready (channel, NULL, NULL);
  const gchar* identifier = tp_channel_get_identifier(channel);
  _show_slide2answer_window(plugin);
  if(strlen(identifier)>0)
  {
    GList* contacts = NULL;
    if(plugin->priv->is_voip_call)
      contacts = osso_abook_aggregator_find_contacts_for_im_contact(plugin->priv->abook_aggregator,
								    identifier,
								    NULL);
    else
      contacts = osso_abook_aggregator_find_contacts_for_phone_number(plugin->priv->abook_aggregator,
								      identifier,
								      TRUE);
    if(contacts)
    {
      GList* contact = contacts;
      if(contact)
      {
	OssoABookContact* abook_contact = OSSO_ABOOK_CONTACT(contact->data);
	_set_contact_data(plugin, abook_contact, identifier);
	while(contact)
	{
	  OssoABookContact* abook_contact = OSSO_ABOOK_CONTACT(contact->data);
	  contact = contact->next;
	}
      }
      g_list_free(contacts);
    }
    else
    {
      _set_unknowncaller_data(plugin, identifier);
    }
  }
  else
  {
    _set_unknowncaller_data(plugin, dgettext("rtcom-call-ui", "voip_fi_caller_information_unknown_caller"));
  }
  plugin->priv->active_channel = channel;
  //  g_object_unref(channel);
}

static void
_account_register_new_channel(Slide2AnswerStatusPlugin* status_plugin, McAccount* account)
{
  if(mc_account_get_connection_path(account)!=NULL)
  {
    GError* error = NULL;
    TpDBusDaemon* tpdbus = tp_dbus_daemon_dup(NULL);
    TpConnection* tpconnection = tp_connection_new(tpdbus, 
						   NULL, 
						   mc_account_get_connection_path(account), 
						   &error);
    if(error)
    {
      g_warning("error creating connection %s\n", error->message);
      g_error_free(error);
      error = NULL;
    }
    else
    {
      tp_cli_connection_connect_to_new_channel(tpconnection,
					       handle_new_channel,
					       status_plugin,
					       NULL,
					       NULL, 
					       NULL);
    }
  }
}

static void
_account_changed(OssoABookAccountManager *manager,
		 McAccount *account,
		 guint property,
		 GValue *value,
		 gpointer user_data)
{
  Slide2AnswerStatusPlugin* status_plugin = SLIDE2ANSWER_STATUS_PLUGIN(user_data);
  TpConnectionStatus stat = mc_account_get_connection_status(account);
  const gchar* connection = mc_account_get_connection_path(account);
  if((connection != NULL) && (stat == TP_CONNECTION_STATUS_CONNECTED))
  {
    if(g_hash_table_lookup(status_plugin->priv->account_map, connection) == NULL)
    {
      g_hash_table_insert(status_plugin->priv->account_map, g_strdup(connection), g_strdup(connection));
      _account_register_new_channel(status_plugin, account);
    }
  }
  else if((connection != NULL) && (stat == TP_CONNECTION_STATUS_DISCONNECTED))
  {
    if(g_hash_table_lookup(status_plugin->priv->account_map, connection) != NULL)
    {
      g_hash_table_remove(status_plugin->priv->account_map, connection);
    }
  }
}

static void
_account_created(OssoABookAccountManager *manager,
		 McAccount *account,
		 gpointer user_data)
{
  g_warning("slide2answer account created\n");
  g_warning("name %s\n", mc_account_get_connection_path(account));
}

static void
_account_removed(OssoABookAccountManager *manager,
		 McAccount *account,
		 gpointer user_data)
{
  g_warning("slide2answer account removed\n");
  g_warning("name %s\n", mc_account_get_connection_path(account));
}

static void
account_ready_cb(OssoABookAccountManager *manager,
		 const GError *xerror,
		 gpointer data)
{
  GList* accounts = osso_abook_account_manager_list_accounts(manager,
							     NULL, NULL);
  if(accounts)
  {
    GList* account;
    for(account = accounts; account; account = account->next)
    {
      McAccount* mc_account = MC_ACCOUNT(account->data);
      _account_register_new_channel(SLIDE2ANSWER_STATUS_PLUGIN(data), mc_account);
    }
    g_list_free(accounts);
  }
  g_signal_connect(OSSO_ABOOK_ACCOUNT_MANAGER(manager), "account-changed",
		   G_CALLBACK(_account_changed), data);
  
  g_signal_connect(OSSO_ABOOK_ACCOUNT_MANAGER(manager), "account-created",
		   G_CALLBACK(_account_created), data);
  
  g_signal_connect(OSSO_ABOOK_ACCOUNT_MANAGER(manager), "account-removed",
		   G_CALLBACK(_account_removed), data);
}

static void
slide2answer_status_plugin_init(Slide2AnswerStatusPlugin* status_plugin)
{
  status_plugin->priv = SLIDE2ANSWER_STATUS_PLUGIN_GET_PRIVATE(status_plugin);
  status_plugin->priv->dbus_system_conn = NULL;
  status_plugin->priv->dbus_system_proxy = NULL;
  status_plugin->priv->call_window = NULL;
  status_plugin->priv->is_voip_call = FALSE;
  status_plugin->priv->abook_account_manager = osso_abook_account_manager_new();
  status_plugin->priv->active_channel = NULL;
  status_plugin->priv->account_map = g_hash_table_new_full(g_str_hash, 
							   g_str_equal, 
							   g_free, 
							   g_free);

  EBookQuery* ebook_query = e_book_query_from_string("(not (is_vcard \"X-TELEPATHY-BLOCKED\"  \"yes\"))");
  osso_abook_account_manager_set_roster_query(status_plugin->priv->abook_account_manager, ebook_query);
  status_plugin->priv->abook_aggregator = OSSO_ABOOK_AGGREGATOR(osso_abook_aggregator_new_with_query(NULL, 
												     ebook_query, 
												     NULL, 
												     0, 
												     NULL));
  osso_abook_aggregator_set_roster_manager(OSSO_ABOOK_AGGREGATOR(status_plugin->priv->abook_aggregator),
					   OSSO_ABOOK_ROSTER_MANAGER(status_plugin->priv->abook_account_manager));
  e_book_query_unref(ebook_query);

  osso_abook_roster_start(OSSO_ABOOK_ROSTER(status_plugin->priv->abook_aggregator));
  osso_abook_account_manager_call_when_ready(status_plugin->priv->abook_account_manager,
					     account_ready_cb, status_plugin, NULL);
  g_signal_connect(status_plugin->priv->call_window, "delete-event", G_CALLBACK(&gtk_widget_hide_on_delete), NULL);
  register_incoming_call_handler(status_plugin);
}

static void
slide2answer_status_plugin_finalize(GObject* object)
{
  Slide2AnswerStatusPlugin* status_plugin = SLIDE2ANSWER_STATUS_PLUGIN(object);
  unregister_incoming_call_handler(status_plugin);
  if(status_plugin->priv->abook_account_manager)
  {
    g_object_unref(status_plugin->priv->abook_account_manager);
  }
  if(status_plugin->priv->abook_aggregator)
  {
    g_object_unref(status_plugin->priv->abook_aggregator);
  }
  g_hash_table_remove_all(status_plugin->priv->account_map);
  g_object_unref(status_plugin->priv->account_map);
  G_OBJECT_CLASS(slide2answer_status_plugin_parent_class)->finalize (object);
}

static void
slide2answer_status_plugin_class_finalize(Slide2AnswerStatusPluginClass* klass)
{
}

static void
slide2answer_status_plugin_class_init(Slide2AnswerStatusPluginClass* klass)
{
  g_type_class_add_private(klass, sizeof(Slide2AnswerStatusPluginPrivate));
  G_OBJECT_CLASS(klass)->finalize = (GObjectFinalizeFunc)slide2answer_status_plugin_finalize;
}
