#!/usr/bin/env python2.5
###########################################################################
## Name: 	       SleepAnalyser
## Description:   Sleep Analyser records your movement during your sleep. It is able to visualise it on a graph to show
##                      how much you move during your sleep. This can help to indicate how you sleep. It also has an alarm
##                      function. You can set the alarm and a time window (ex. 30 minutes). The Alarm will then go off
##                      sometimes during the time window (as soon as you move more), but latest at the set alarm.
##                      Old records can be visualised and you can load records from your friends. SleepAnalyser also has a
##                      test function. It will record and visualise much faster. Also it will make a beep when ever your
##                      movement goes over the trigger level.
## Copyright:     George Ruinelli, george@ruinelli.ch
## Licence:        GPL
###########################################################################

from PyQt4.QtCore import *
from PyQt4.QtGui import *
import os

import dbus
from dbus.mainloop.qt import DBusQtMainLoop

import data #own data
import records #record data

#app path has to be set before  first use!
data.app_path=os.path.dirname(__file__)
if(data.app_path==""): data.app_path="./"
else: data.app_path=data.app_path+"/"

#load language file
import i18n
_ = i18n.language.gettext

#own functions
from functions_general import *
from functions_file import *
from functions_record import *
from functions_alarm import *

#own c cfunctions (for speed improvements)
try:
    from lib import  mylib
    MyLib_Imported=True
except:
    MyLib_Imported=False


#GUIs
from Ui_StartWindow import *
from Ui_RecordWindow import *
from Ui_ViewWindow import *
from Ui_ConfigWindow import *
from Ui_HelpWindow import *
from Ui_DateWindow import *
from Ui_AlarmTimeWindow import *
from Ui_AskForKeepWindow import *
from Ui_AdditionalInfoWindow import *



###########################################################################
## Main function
##############o#############################################################
def main():
    global app,  StartWindow,  RecordWindow,  ViewWindow ,  ConfigWindow,  DateWindow,  AlarmTimeWindow,  AskForKeepWindow,  AdditionalInfoWindow,  HelpWindow
    global config

    try:
        if(sys.argv[1]=="debug"):
            print "Enable Debug mode"
            data.debug=1
    except:
        data.debug=0

    #if not on the N900, a acceleration sensor has to be simulated
    if(os.path.exists("/sys/class/i2c-adapter/i2c-3/3-001d/coord")): data.demo=False	#runs on real N900 hardware
    else: data.demo=True	#runs on other hardware/OS


    app = QApplication(sys.argv)


    if(data.demo==True):
        print "Runs not on a N900 => Use randomizer for acceleration sensor data"
        data.configFolder=data.developmentconfigFolder
        data.RecordFolder=data.developmentRecordFolder
    else: data.RecordFolder=data.defaultRecordFolder


    config=LoadConfig()

    #Load version file
    try:
        if(data.debug==1): print "Application path:",  data.app_path
        file = open(data.app_path+"version", 'r')
        data.version = file.readline()
        data.version=data.version[:-1]
        data.build = file.readline()
        print "Application version: "+str(data.version)+ "-"+str(data.build)
    except:
        print "Version file not found, please check your installation!"





    #chek if module is available and could get loaded
    if(MyLib_Imported==True): #module is loaded
        print "Mylib is compiled:", mylib.IsCompiled() #check if module is compiled
        if(mylib.IsCompiled()==False): #module is not compiled
            txt= "SleepAnalyser:\nThe compiled module mylib.so could not get loaded! Please check if there is a never version "+ \
                     "or inform the maintainer about it! SleepAnalyser will still work, " + \
                     "but some functions will work slow as we have to fall back to the unoptimised python code!"
            print txt
            cmd="dbus-send --type=method_call --dest=org.freedesktop.Notifications /org/freedesktop/Notifications org.freedesktop.Notifications.SystemNoteDialog " + \
                     "string:\""+txt+"\" uint32:0 string:\"OK\""
            os.popen(cmd)
    else: #module is available but broken
        txt="SleepAnalyser:\nSleepAnalyser could not load the module mylib.so\n" + \
                 "in "+data.app_path+"lib. It might be corrupted. Please check if there is a never version "+ \
                 "or inform the maintainer about it! You can get SleepAnalyser working by removing the file "+data.app_path+"lib/mylib.so, " + \
                 "how ever some functions will work slow as we have to fall back to the unoptimised python code!"
        print txt
        cmd="dbus-send --type=method_call --dest=org.freedesktop.Notifications /org/freedesktop/Notifications org.freedesktop.Notifications.SystemNoteDialog " + \
                 "string:\""+txt+"\" uint32:0 string:\"OK\""
        os.popen(cmd)
        exit()



    #Load Stylesheet for buttons
    try:
        file = open(data.configFolder+"UIStyle2.css", 'r')
        data.UIStyle=""
        for line in file: data.UIStyle=data.UIStyle+line+"\n"
        file.close()
        print "Loaded user defined style sheet"
    except:
#        print "User defined style sheet not existing or not readable"
#        file = open(data.app_path+"UIStyle.css", 'r')
        data.UIStyle=""
#        for line in file: data.UIStyle=data.UIStyle+line+"\n"
#        file.close()
#        try:
#            print "Create user defined style sheet in",  data.configFolder
#            file= open(data.configFolder+"UIStyle.css", 'w')
#            file.write(data.UIStyle)
#            file.close()
#        except:
#            print "Could not create user defined style sheet"





    StartWindow = frmStartWindow()
    StartWindow.show()
    StartWindow.UpdateScreenStart()

    # Create subwindows
    RecordWindow = frmRecordWindow(StartWindow)
    ConfigWindow = frmConfigWindow(StartWindow)
    ViewWindow = frmViewWindow(StartWindow)
    HelpWindow = frmHelpWindow(StartWindow)
    AlarmTimeWindow = frmAlarmTimeWindow(StartWindow)
    DateWindow = frmDateWindow(ViewWindow)
    AdditionalInfoWindow = frmAdditionalInfoWindow(ViewWindow)
    AskForKeepWindow = frmAskForKeepWindow(RecordWindow)


    # Hide SubWindows at first
    RecordWindow.hide()
    ViewWindow.hide()
    ConfigWindow.hide()
    AlarmTimeWindow.hide()
    DateWindow.hide()
    AdditionalInfoWindow.hide()
    AskForKeepWindow.hide()

    #get user name for in record files
    if(data.user==""):
        EditUserName()



    DisplayStatusDaemon()

    sys.exit(app.exec_())






###########################################################################
##
###########################################################################
def EditUserName():
#      r=QInputDialog.getText(None,  "Add Note",  "Add a note to this record:\n(leave empty for no note)",  QLineEdit.Normal,  records.Note)
#        if(r[1]==True): #OK pressed
    r = QInputDialog.getText(None, "SleepAnalyser", _("Please enter your name.\nIt will be used in the records file, this makes it easier\nwhen you want to share your sleep pattern with others."), 0,  data.user)
#    if(r != ""):
    if(r[1]==True): #OK pressed
        data.user=unicode(r[0])
        if(data.user==""): data.user="-"
        config.set('main', 'user', str(data.user.encode( "utf-8" )))

    print "User name:",  data.user






###########################################################################
##  Gets called every second
###########################################################################
def MeasurementTimerLoop(): #(interval: sampling*sample_smoothing*value_smoothing)
    r=GetSensorData()
    if(r==1): #new data got added to array
        ProcessData(False,  False)
        ProcessAlarm()

        WriteRecord()

#        GraphLen=records.Index/data.arrXscale[data.Xscale]
#        data.GraphStart=max(GraphLen-779, 0)

        data.ScrollToEnd=True
        if(data.DisplayIsOn==True):
            GenerateStatisticData()
            GenerateGraphBitmap(RecordWindow, False)

        if(records.Index==records.MaxArrayDataLen-1):
            txt=_('Memory is full!')
            print txt
            data.Status=txt
            QtGui.QMessageBox.warning(RecordWindow, txt, _("You reached the end of the recording capacity!\nThe recording stops now."), QtGui.QMessageBox.Ok)
#            RecordWindow.StopMeasurement()
            StopMeasurement() #stop but keep alarm active!
            SwitchModes(False) #change back to normal mode and general profile if requested

    ProcessAlarm()

    if(data.DisplayIsOn==True):
        RecordWindow.UpdateAnimation() #show turning clock
        if(data.debug==1): print "-> o Update screen"
        RecordWindow.UpdateScreen()
    else:
        if(data.debug==1): print "-> x Do not update screen, Display is off!"


#    print data.AlarmIsActive,  data.StopAfterAlarm
    if(data.AlarmIsActive==True): #alarm got activated
#        print "Alarm is active"
        if(data.StopAfterAlarm==True): #stop recording
#            print "Stop measurement, alarm is active"
            txt=_("Alarm got activated, stopped recording")
            print txt
            data.Status=txt
            StopMeasurement() #stop but keep alarm active!

            #change directly to view screen
#            DeleteAlarm() #delete Alarm if existing
#            records.AlarmIsSet=False
            data.LastRecord=data.RecordFilename #set to corrent file for viewwindow
            config.set('main', 'LastRecord', str(data.LastRecord))
            print "Show record in ViewWindow:",  data.LastRecord
            StartWindow.ShowViewWindow()
            RecordWindow.hide()

        else: #continue recording
            pass

        SwitchModes(False) #change back to normal mode and general profile if requested






###########################################################################
## Start the measurement
###########################################################################
def InitMeasurement():
    global app

    data.sample_index=0

    data.Animation=7
    RecordWindow.UpdateAnimation() #show turning clock

    if(data.QuickMode==False): data.Status=_("Recording...")
    else: data.Status=_("Recording in Test Mode...")

    records.User=data.user
    records.SWversion=data.version
    records.Sampling_interval=data.Sampling_interval
    records.Sample_smoothing=data.Sample_smoothing
    records.Value_smoothing=data.Value_smoothing
    records.Trigger=data.Trigger #save Trigger level on start
    records.statusID=_("OK")

    if(data.debug==1): PrintRecordData()

    #set scale and sampling interval
    if(data.QuickMode==False):
        data.interval=data.Sampling_interval #normal
        data.Xscale=1 #set to default zoom level
    else:
        data.interval=10 #Fast mode for testing
        data.Xscale=0 #set to max zoom level

#    print "#################### USE MAXIMUM SPEED FOR DEBUGGING!!!!!!!!!!!"
#    data.interval=1 #Debugging



    data.run=True #change to recording
    app.timer_GetData = QtCore.QTimer()
    QtCore.QObject.connect(app.timer_GetData, QtCore.SIGNAL("timeout()"), MeasurementTimerLoop)

    data.StartTime=time.localtime()
    data.StartTimeSec=Tuple2Seconds(data.StartTime)

#    if(data.debug==False): app.timer_GetData.start(data.interval)
#    else: app.timer_GetData.start(1)    #run very fast for testing
    app.timer_GetData.start(data.interval)

    CreateRecordFile(data.interval)
    GetFirstEntry()
    WriteRecord()

    GenerateGraphBitmap(RecordWindow, False)
#    RecordWindow.UpdateScreen()

    RecordWindow.ui.lblStatus.hide()
    RecordWindow.ui.CanvasWidget.show()
    RecordWindow.UpdateScreen()
    app.processEvents() #force ui update


    if(data.debug==1):
        print "Start timers (Interval:", data.interval ,"ms)..."
        print "Sampling interval:", data.interval, "ms"
        print "sample_smoothing:", data.Sample_smoothing, "x"
        print "value_smoothing:", data.Value_smoothing, "x"
        print "=> record interval:", data.interval*data.Value_smoothing, "ms"
        print "Derivation:", data.Derivation, "(substracts value", data.Derivation, "- value 0)"

    SwitchModes(True) #change to offline mode and silent profile if requested


###########################################################################
##
###########################################################################
def StopMeasurement():
    global app

    GenerateStatisticData()
    GenerateGraphBitmap(RecordWindow, False)

    RecordWindow.Canvas.update()
    RecordWindow.UpdateScreen()
    app.processEvents() #force ui update

#    DeleteAlarm() #delete Alarm if existing

#    print "Stop Measurement",  data.run
    if(data.run==True): #in recording
        if(data.debug==1): print "Stop Timer"
        app.timer_GetData.stop()
        StopRecordFile()
        UpdateFileHeader() #rewriterecord file (only do when recording is stopped!)

    data.run=False

    RecordWindow.UpdateScreenStop()







############################################################################
## Stretch graph to show more details
############################################################################
def Zoom(direction):
    if(direction==+1): #zoom in
        if(data.Xscale==0): return
        ViewWindow.ui.bZoomOut.setEnabled(True)
        data.Xscale=(data.Xscale-1) % data.arrXscaleLen
    else: # -1, zoom out
        if(data.Xscale==data.arrXscaleLen-1): return
        ViewWindow.ui.bZoomIn.setEnabled(True)
        data.Xscale=(data.Xscale+1) % data.arrXscaleLen





###########################################################################
## Raise or lower the Trigger in the graph + sleep phases
###########################################################################
def MoveTrigger(direction):
    if(direction==+1): #up
        data.Trigger=data.Trigger+1
        if(data.Trigger>data.maxTrigger): data.Trigger=data.maxTrigger
    else: #-1, down
        data.Trigger=data.Trigger-1
        if(data.Trigger<0): data.Trigger=0
    if(data.run==True): records.Trigger=data.Trigger




def About(ui):
    QtGui.QMessageBox.about(ui, _("About")+" SleepAnalyser", _("Copyright 2010 by George Ruinelli with great support from the MAEMO community")+"\n"+\
    _("Version:")+" "+str(data.version)+"-"+str(data.build)+"\n"+\
    _("Contact:")+" george@ruinelli.ch\n" +\
    _("Website:")+" http://www.ruinelli.ch/sleepanalyser/\n\n"+\
    _("Feedback and donations are very welcome!")+"\n\n"+\
    _("Translations:")+\
    "\nEnglish, Deutsch: George Ruinelli\n"+\
    "Francais: Fred Kwame Ayanful")























###########################################################################
##  Class for Main Window
###########################################################################
class frmStartWindow(QMainWindow):
    def __init__(self):
        QMainWindow.__init__(self)
        if(data.demo==False): self.setAttribute(Qt.WA_Maemo5StackedWindow) # This attribute makes the whole Stacked Window thing work
        self.ui = Ui_StartWindow()
        self.ui.setupUi(self)

        QtCore.QObject.connect(self.ui.bCollecting, QtCore.SIGNAL("clicked()"), self.StartMeasurementNormal)
        QtCore.QObject.connect(self.ui.bView, QtCore.SIGNAL("clicked()"), self.ShowViewWindow)
        QtCore.QObject.connect(self.ui.bSetAlarm, QtCore.SIGNAL("clicked()"), self.SetAlarm)

        #Add styles to buttons
        self.ui.bCollecting.setStyleSheet(data.UIStyle)
        self.ui.bSetAlarm.setStyleSheet(data.UIStyle)
        self.ui.bView.setStyleSheet(data.UIStyle)


        QtCore.QObject.connect(self.ui.actionTest_recording, QtCore.SIGNAL("triggered()"), self.StartMeasurementQuick)
        QtCore.QObject.connect(self.ui.actionSettings, QtCore.SIGNAL("triggered()"), self.ShowConfigWindow)
        QtCore.QObject.connect(self.ui.actionAbout, QtCore.SIGNAL("triggered()"), self.About)
        QtCore.QObject.connect(self.ui.actionHelp, QtCore.SIGNAL("triggered()"), self.ShowHelpWindow)


        if(data.app_path!="./"): #only needed if not started in app path
            self.ui.lblIcon.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"SleepAnalyser2.png"))
            self.ui.bSetAlarm.setIcon(QIcon(data.app_path+"img/"+"SleepAnalyser.png"))
#            self.ui.bQuick.setIcon(QIcon(data.app_path+"img/"+"krec.png"))
            self.ui.bCollecting.setIcon(QIcon(data.app_path+"img/"+"krec.png"))
            self.ui.bView.setIcon(QIcon(data.app_path+"img/"+"zoom.png"))
#            self.ui.bConfig.setIcon(QIcon(data.app_path+"img/"+"configure.png"))


        #set texts
        self.setWindowTitle("SleepAnalyser")
        self.ui.bView.setText(_("View records"))
        self.ui.bCollecting.setText(_("Start recording only"))
        self.ui.bSetAlarm.setText(_("Set alarm and\nstart recording"))
        self.ui.actionAbout.setText(_("About"))
        self.ui.actionSettings.setText(_("Settings"))
        self.ui.actionHelp.setText(_("Help"))
        self.ui.actionTest_recording.setText(_("Test recording"))

#        self.ui.lblNote.setText(_("Development version, might be unstable!"))





    ###########################################################################
    ## Show the collectring window (her the data gets visualised during the logging)
    ###########################################################################
    def StartMeasurementNormal(self): ##(Class StartWindow)
        if(data.debug==1): print "Measure in normal Mode"
        data.QuickMode=False #run in normal mode
        self.StartMeasurement()





    ###########################################################################
    ## Show the collectring window (her the data gets visualised during the logging)
    ###########################################################################
    def StartMeasurementQuick(self): ##(Class StartWindow)
        if(data.debug==1): print "Measure in QuickMode"
        data.QuickMode=True #run in quick mode (for testing only)
        data.Xscale=0 #default zoom level
        self.StartMeasurement()





    ###########################################################################
    ##
    ###########################################################################
    def StartMeasurement(self): ##(Class StartWindow)
        RecordWindow.ui.lblStatus.hide()
        RecordWindow.show()
        app.processEvents() #force ui update

        InitRecord() #clear all old data
        InitMeasurement()
        RecordWindow.UpdateScreenStart()
        RecordWindow.Canvas.show()


    ###########################################################################
    ## Show the log window (where the data from a file gets visualised)
    ###########################################################################
    def ShowViewWindow(self): ##(Class StartWindow)
        ViewWindow.ui.lblStart.setText("")
        ViewWindow.ui.lblEnd.setText("")
        ViewWindow.ui.lblNote.setText("")
        ViewWindow.Canvas.hide()
        ViewWindow.StatisticsCanvas.hide()
        ViewWindow.show()
        app.processEvents() #force ui update

        #reset to default record folder
        if(data.demo==True): data.RecordFolder=data.developmentRecordFolder #folder on devel PC
        else: data.RecordFolder=data.defaultRecordFolder

        LoadFileList()
        data.arrRecordFilesIndex=len(data.arrRecordFiles)-1 #no file selected, show latest

        if os.path.exists(data.RecordFolder+data.LastRecord):
            data.RecordFilename=data.LastRecord
        else:
            l=len(data.arrRecordFiles)
            if(l>0):
                data.RecordFilename=data.arrRecordFiles[l-1] #open last (youngest)record file
            else: #no RecordFiles in RecordFolder, show "open file" dialog
                tmp=ViewWindow.ShowFileDialog()
                if(not os.path.exists(tmp)): #only continue if valid record file selected
                    StartWindow.show()
                    ViewWindow.hide()
                    return #only continue if valid record file selected
                tmp2=tmp.split("/")
                l=len(tmp2)
                data.RecordFilename=tmp2[l-1]
                data.RecordFolder=string.replace(tmp,  data.RecordFilename,  "") #get folder of selected file
                print "Set folder=",  data.RecordFolder,  ", file=",  data.RecordFilename


        ViewWindow.ShowRecord()

#        if(data.RecordFilename!=""):
#            ViewWindow.show()
#        else:
        if(data.RecordFilename==""):
            print "No valid file selected"
            ViewWindow.hide()






    def ShowConfigWindow(self): ##(Class StartWindow)
        ConfigWindow.UpdateScreen()
        ConfigWindow.show()





    def ShowHelpWindow(self): ##(Class StartWindow)
        HelpWindow.show()




    def SetAlarm(self): ##(Class StartWindow)
        AlarmTimeWindow.UpdateScreen()
        AlarmTimeWindow.show()




    def UpdateScreenStart(self): ##(Class StartWindow)
        pass



    def About(self): ##(Class StartWindow)
        About(self)



    def closeEvent(self, event): ##(Class StartWindow)
        global config
        WriteConfig()

















###########################################################################
#  Class for Collecting Window
###########################################################################
class frmRecordWindow(QMainWindow):
    def __init__(self, parent=None):
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor
        if(data.demo==False): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack

        self.ui = Ui_RecordWindow()
        self.ui.setupUi(self)


        app.processEvents() #force ui update

#        scroller = self.ui.ScrollArea.property("kineticScroller").toPyObject()
#        scroller.setEnabled(True)

        self.Canvas = MyGraphCanvas(self.ui.CanvasWidget)

        self.ui.CanvasWidget.setMinimumSize(QtCore.QSize(1000, data.GraphBitMapHeight)) #needed?


        QtCore.QObject.connect(self.ui.bStop, QtCore.SIGNAL("clicked()"), self.StopMeasurement)
        QtCore.QObject.connect(self.ui.bRaiseTrigger, QtCore.SIGNAL("clicked()"), self.RaiseTrigger)
        QtCore.QObject.connect(self.ui.bLowerTrigger, QtCore.SIGNAL("clicked()"), self.LowerTrigger)
        QtCore.QObject.connect(self.ui.bZoomIn, QtCore.SIGNAL("clicked()"), self.ZoomIn)
        QtCore.QObject.connect(self.ui.bZoomOut, QtCore.SIGNAL("clicked()"), self.ZoomOut)
        QtCore.QObject.connect(self.ui.bEditNote, QtCore.SIGNAL("clicked()"), self.EditNote)

        QtCore.QObject.connect(self.ui.actionHelp, QtCore.SIGNAL("triggered()"), self.ShowHelpWindow)
        QtCore.QObject.connect(self.ui.actionAbout, QtCore.SIGNAL("triggered()"), self.About)

        #Add styles to buttons
        self.ui.bStop.setStyleSheet(data.UIStyle)
        self.ui.bRaiseTrigger.setStyleSheet(data.UIStyle)
        self.ui.bLowerTrigger.setStyleSheet(data.UIStyle)
        self.ui.bZoomIn.setStyleSheet(data.UIStyle)
        self.ui.bZoomOut.setStyleSheet(data.UIStyle)
        self.ui.bEditNote.setStyleSheet(data.UIStyle)

        if(data.app_path!="./"): #only needed if not started in app path
            self.ui.bZoomIn.setIcon(QIcon(data.app_path+"img/"+"zoom-in.png"))
            self.ui.bZoomOut.setIcon(QIcon(data.app_path+"img/"+"zoom-out.png"))
            self.ui.bRaiseTrigger.setIcon(QIcon(data.app_path+"img/"+"go-up.png"))
            self.ui.bLowerTrigger.setIcon(QIcon(data.app_path+"img/"+"go-down.png"))
            self.ui.bEditNote.setIcon(QIcon(data.app_path+"img/"+"kwrite.png"))
            self.ui.lblAnimation.setPixmap(QtGui.QPixmap(data.app_path+"img/anim/0.png"))

        self.setWindowTitle("SleepAnalyser ("+_("Recording")+")")
        self.ui.lblElapsedTime.setText(_("Elapsed time:")+" 0")
        self.ui.bStop.setText(_("Stop"))
        self.ui.lblAlarmTime.setText(_("Alarm time: (no alarm set)"))
        self.ui.lblStatus2.setText(_("Status:"))
        self.ui.lblStatus.setText(_("Loading..."))
        self.ui.menuMenu.setTitle(_("Menu"))
        self.ui.actionHelp.setText(_("Help"))
        self.ui.actionAbout.setText(_("About"))

#        app.processEvents() #force ui update



    def UpdateScreenStart(self): ##(Class RecordWindow)
#        t=strftime("%d. %b. %Y %H:%M:%S", data.StartTime)
        t=strftime("%a. %x %X", data.StartTime)
#        self.ui.lblStarttime.setText("Start: "+t)
        self.ui.lblElapsedTime.setText(_("Elapsed time:"))
#        self.ui.lblIcon.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"SleepAnalyser.png"))
        self.ui.bStop.setIcon(QIcon(data.app_path+"img/"+"krec2.png"))
        self.ui.bStop.setText(_("Stop"))

        if(records.AlarmIsSet==True): #alarm active
            m=str(data.AlarmMinute)
            if(len(m)==1): m="0"+m
#            self.ui.lblAlarmTime.setText(str(data.AlarmHour)+":"+m+" or up to "+str(data.AlarmTimeWindow)+" minutes before that")
            self.ui.lblAlarmTime.setText(_("Alarm time:")+" "+strftime("%H:%M", records.AlarmTimeWindowStart_tuple)+" - "+strftime("%H:%M", records.AlarmTimeWindowEnd_tuple))
        else: #Do only recording, no alarm
            self.ui.lblAlarmTime.setText(_("Alarm time: (no alarm set)"))

#        self.ui.lblNote.setText("Note: ")
        self.UpdateScreen()




    def UpdateScreen(self): ##(Class RecordWindow)
#        if(data.DisplayIsOn==False):
#            if(data.debug==1): print "-> x Do not update screen, Display is off!"
#            return
#        else:
#            if(data.debug==1): print "-> o Update screen"

        if(data.run==True): #recording
            t=time.mktime(time.localtime())
            time_difference=t-data.StartTimeSec
            d=time.strftime("%Hh %Mm", time.gmtime(time_difference))
#            self.ui.lblElapsedTime.setText("Elapsed time: "+d + " ("+str(records.Index)+" entries)")
            self.ui.lblElapsedTime.setText(_("Elapsed time:")+" "+d)
        else: self.UpdateScreenStop()

#        self.ui.lblStatus2.setText("Status: "+data.Status)
        self.ui.lblStatus2.setText(data.Status)

        if(data.Xscale==0): self.ui.bZoomIn.setEnabled(False)
        else: self.ui.bZoomIn.setEnabled(True)
        if(data.Xscale==data.arrXscaleLen-1): self.ui.bZoomOut.setEnabled(False)
        else: self.ui.bZoomOut.setEnabled(True)

        if(data.Trigger==data.maxTrigger): self.ui.bRaiseTrigger.setEnabled(False)
        else: self.ui.bRaiseTrigger.setEnabled(True)
        if(data.Trigger==0): self.ui.bLowerTrigger.setEnabled(False)
        else: self.ui.bLowerTrigger.setEnabled(True)






    def UpdateScreenStop(self): ##(Class RecordWindow)
        self.ui.bStop.setText(_("Close"))
#        self.ui.lblStatus2.setText("Status: "+data.Status)
        self.ui.lblStatus2.setText(data.Status)



    def UpdateAnimation(self):
        data.Animation=(data.Animation+1) % 8
        self.ui.lblAnimation.setPixmap(QtGui.QPixmap(data.app_path+"img/anim/"+str(data.Animation)+".png"))
#        print data.Animation


    def StopMeasurement(self): ##(Class RecordWindow)
            self.closeEvent(0)



    def ZoomIn(self): ##(Class RecordWindow)
        self.ui.lblStatus.setText(_("Zoom in..."))
        self.ui.lblStatus.show()
        app.processEvents() #force ui update
        Zoom(+1)
        GenerateGraphBitmap(self, False)
        self.ui.lblStatus.hide()



    def ZoomOut(self): ##(Class RecordWindow)
        self.ui.lblStatus.setText(_("Zoom out..."))
        self.ui.lblStatus.show()
        app.processEvents() #force ui update
        Zoom(-1)
        GenerateGraphBitmap(self, False)
        self.ui.lblStatus.hide()



    def RaiseTrigger(self): ##(Class RecordWindow)
        self.ui.lblStatus.setText(_("Raise trigger level..."))
        self.ui.lblStatus.show()
        app.processEvents() #force ui update
        MoveTrigger(+1)
        self.ui.bLowerTrigger.setEnabled(True)
        if(data.Trigger==data.maxTrigger): self.ui.bRaiseTrigger.setEnabled(False)
        else: self.ui.bRaiseTrigger.setEnabled(True)

        GenerateGraphBitmap(self, False)
        self.ui.lblStatus.hide()



    def LowerTrigger(self): ##(Class RecordWindow)
        self.ui.lblStatus.setText(_("Lower trigger level..."))
        self.ui.lblStatus.show()
        app.processEvents() #force ui update
        MoveTrigger(-1)
        self.ui.bRaiseTrigger.setEnabled(True)
        if(data.Trigger==0): self.ui.bLowerTrigger.setEnabled(False)
        else: self.ui.bLowerTrigger.setEnabled(True)

        GenerateGraphBitmap(self, False)
        self.ui.lblStatus.hide()



    def closeEvent(self, event): ##(Class RecordWindow)
        global AskForKeepWindow,  StartWindow
#        close=True
        if(data.QuickMode==False ): #running in normal mode
            try:
                event.ignore()  #ignore closing
            except:
                pass
            AskForKeepWindow.UpdateScreen()
            AskForKeepWindow.show()
#            RecordWindow.hide()
#        else:
#            pass #close
        else: #quick mode, delete file anyway and close
            StopMeasurement()
            print "Delete RecordFile: "+data.RecordFilename
            os.remove(data.RecordFolder+data.RecordFilename)
            data.QuickMode=False
            StartWindow.show()
            RecordWindow.hide()
            self.hide()
            SwitchModes(False) #change back to normal mode and general profile if requested



    def EditNote(self): ##(Class RecordWindow)
        r=QInputDialog.getText(None,  _("Note about this record"),  _("Add a note to this record\n(leave empty for no note):"),  QLineEdit.Normal,  records.Note)
        if(r[1]==True): #OK pressed
            records.Note=unicode(r[0])
#            self.ui.lblNote.setText("Note: "+records.Note)
            if(data.run==True): UpdateFileHeader() #rewrite record file (only do when recording is stopped!)



    def About(self): ##(Class RecordWindow)
        About(self)


    def ShowHelpWindow(self): ##(Class RecordWindow)
        HelpWindow.show()








###########################################################################
## Class for Log Window
###########################################################################
class frmViewWindow(QMainWindow):
    def __init__(self, parent=None):
        global app
#        global Canvas
        QMainWindow.__init__(self, parent)
        if(data.demo==False): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack

        self.ui = Ui_ViewWindow()
        self.ui.setupUi(self)

#        scroller = self.ui.ScrollArea.property("kineticScroller").toPyObject()
#        scroller.setEnabled(True)

        self.Canvas = MyGraphCanvas(self.ui.CanvasWidget)
        self.StatisticsCanvas = MyStatCanvas(self)

        self.ui.CanvasWidget.setMinimumSize(QtCore.QSize(1000, data.GraphBitMapHeight)) #needed?


        QtCore.QObject.connect(self.ui.bRaiseTrigger, QtCore.SIGNAL("clicked()"), self.RaiseTrigger)
        QtCore.QObject.connect(self.ui.bLowerTrigger, QtCore.SIGNAL("clicked()"), self.LowerTrigger)
        QtCore.QObject.connect(self.ui.bZoomIn, QtCore.SIGNAL("clicked()"), self.ZoomIn)
        QtCore.QObject.connect(self.ui.bZoomOut, QtCore.SIGNAL("clicked()"), self.ZoomOut)
        QtCore.QObject.connect(self.ui.bNext, QtCore.SIGNAL("clicked()"), self.NextRecord)
        QtCore.QObject.connect(self.ui.bLast, QtCore.SIGNAL("clicked()"), self.LastRecord)
        QtCore.QObject.connect(self.ui.bDelete, QtCore.SIGNAL("clicked()"), self.DeleteRecord)
        QtCore.QObject.connect(self.ui.bOpen, QtCore.SIGNAL("clicked()"), self.OpenRecord)
        QtCore.QObject.connect(self.ui.bEditNote, QtCore.SIGNAL("clicked()"), self.EditNote)
        QtCore.QObject.connect(self.ui.bOpenDate, QtCore.SIGNAL("clicked()"), self.OpenDate)
        QtCore.QObject.connect(self.ui.bStatistics, QtCore.SIGNAL("clicked()"), self.ShowRecordData)

#        QtCore.QObject.connect(self.ui.actionHelp, QtCore.SIGNAL("triggered()"), self.ShowHelpWindow)
        QtCore.QObject.connect(self.ui.actionAbout, QtCore.SIGNAL("triggered()"), self.About)
        QtCore.QObject.connect(self.ui.actionShowRecordData, QtCore.SIGNAL("triggered()"), self.ShowRecordData)
        QtCore.QObject.connect(self.ui.actionExport, QtCore.SIGNAL("triggered()"), self.Export)


        #Add styles to buttons
        self.ui.bRaiseTrigger.setStyleSheet(data.UIStyle)
        self.ui.bLowerTrigger.setStyleSheet(data.UIStyle)
        self.ui.bZoomIn.setStyleSheet(data.UIStyle)
        self.ui.bZoomOut.setStyleSheet(data.UIStyle)
        self.ui.bNext.setStyleSheet(data.UIStyle)
        self.ui.bLast.setStyleSheet(data.UIStyle)
        self.ui.bDelete.setStyleSheet(data.UIStyle)
        self.ui.bOpen.setStyleSheet(data.UIStyle)
        self.ui.bOpenDate.setStyleSheet(data.UIStyle)
        self.ui.bEditNote.setStyleSheet(data.UIStyle)
        self.ui.bStatistics.setStyleSheet(data.UIStyle)




        if(data.app_path!="./"): #only needed if not started in app path
            self.ui.bLast.setIcon(QIcon(data.app_path+"img/"+"arrow-left-double.png"))
            self.ui.bNext.setIcon(QIcon(data.app_path+"img/"+"arrow-right-double.png"))
            self.ui.bDelete.setIcon(QIcon(data.app_path+"img/"+"edittrash.png"))
            self.ui.bOpen.setIcon(QIcon(data.app_path+"img/"+"document-open-folder.png"))
            self.ui.bOpenDate.setIcon(QIcon(data.app_path+"img/"+"date.png"))
            self.ui.bZoomIn.setIcon(QIcon(data.app_path+"img/"+"zoom-in.png"))
            self.ui.bZoomOut.setIcon(QIcon(data.app_path+"img/"+"zoom-out.png"))
            self.ui.bRaiseTrigger.setIcon(QIcon(data.app_path+"img/"+"go-up.png"))
            self.ui.bLowerTrigger.setIcon(QIcon(data.app_path+"img/"+"go-down.png"))
            self.ui.bEditNote.setIcon(QIcon(data.app_path+"img/"+"kwrite.png"))
            self.ui.bStatistics.setIcon(QIcon(data.app_path+"img/"+"klipper_dock.png"))


        self.setWindowTitle("SleepAnalyser ("+_("View")+")")
        self.ui.lblstate1.setText(_("Awake"))
        self.ui.lblstate2.setText(_("Light sleep"))
        self.ui.lblstate3.setText(_("Deep sleep"))
        self.ui.actionShowRecordData.setText(_("Additional information"))
        self.ui.actionAbout.setText(_("About"))
        self.ui.actionExport.setText(_("Export as image"))




    def UpdateScreen(self): ##(Class ViewWindow)
#        d0=strftime("%a. %x %X", records.StartTime_tuple)
        day=strftime("%A %d. %B %Y", records.StartTime_tuple)
        d0=strftime("%H:%M", records.StartTime_tuple)
#        d1=strftime("%a. %x %X", records.EndTime_tuple)
        d1=strftime("%H:%M", records.EndTime_tuple)
        diff_sec=records.EndTime_seconds-records.StartTime_seconds
#        diff=time.strftime("%H:%M:%S", time.gmtime(diff_sec))
        diff=time.strftime("%Hh %Mm", time.gmtime(diff_sec))
##        The following lines are disabled, currently i dont care if it could record for more than 24h
#        if(diff_sec>86400): d="1 day, " #longer than 24h
#        else: d=""
#        print day
        self.ui.lblStart.setText(str(day))
#        self.ui.lblEnd.setText(str(d1)+" (Duration: "+d+str(diff)+")")
#        self.ui.lblEnd.setText(_("Start:")+" "+d0+", "+_("End:")+" "+d1+", "+_("Duration:")+" "+diff)
#        print d0+" - "+d1+"   ("+diff+")"
        self.ui.lblEnd.setText(d0+" - "+d1+"   ("+diff+")")

#        self.ui.lblNote.setText("Note: "+records.Note)
        self.ui.lblNote.setText(records.Note)

        if(data.Xscale==0): self.ui.bZoomIn.setEnabled(False)
        else: self.ui.bZoomIn.setEnabled(True)
        if(data.Xscale==data.arrXscaleLen-1): self.ui.bZoomOut.setEnabled(False)
        else: self.ui.bZoomOut.setEnabled(True)

        if(data.Trigger==data.maxTrigger): self.ui.bRaiseTrigger.setEnabled(False)
        else: self.ui.bRaiseTrigger.setEnabled(True)
        if(data.Trigger==0): self.ui.bLowerTrigger.setEnabled(False)
        else: self.ui.bLowerTrigger.setEnabled(True)

        if(len(data.arrRecordFiles)<2): #disable last/next buttons if only one file in folder
            self.ui.bNext.setEnabled(False)
            self.ui.bLast.setEnabled(False)
        else:
            self.ui.bNext.setEnabled(True)
            self.ui.bLast.setEnabled(True)

#        ScrollbarStartPercent,ScrollbarEndPercent= ScrollbarData()
#        ScrollbarStartPercent=round(ScrollbarStartPercent,  0)
#        ScrollbarEndPercent=round(ScrollbarEndPercent,  0)
##        print "UpdateScreen,  Scrollbar percent:",  ScrollbarStartPercent,  ScrollbarEndPercent
#        if(ScrollbarStartPercent>0): self.ui.bLeft.setEnabled(True)
#        else: self.ui.bLeft.setEnabled(False)
#        if(ScrollbarEndPercent<100): self.ui.bRight.setEnabled(True)
#        else: self.ui.bRight.setEnabled(False)




    ###########################################################################
    ##
    ###########################################################################
    def ShowRecord(self): ##(Class ViewWindow)
        global app

        self.ui.lblStart.setText("")
        self.ui.lblEnd.setText("")
        self.ui.lblNote.setText("")
        self.Canvas.hide()
        self.StatisticsCanvas.hide()

        self.ui.lblStatus.setText(_("Loading record..."))
        self.ui.lblStatus.show()
        app.processEvents() #force ui update

        r=LoadRecord()
        if(r==1): #file too long
            QtGui.QMessageBox.warning(self, _("Record too long!"), _("The record is too long. The end will be cut off!"), QtGui.QMessageBox.Ok)
        elif(r==2): #file does not exist
            self.OpenRecord()
            return
#        elif(r==3): #file is corrupted => draw empty graph
#            QtGui.QMessageBox.warning(self, 'File corrupded!', "The data in the record file\n"+data.RecordFilename+"\nis invalid!\nNo data loaded.", QtGui.QMessageBox.Ok)
#        elif(r==8): #file header is corrupted
#            QtGui.QMessageBox.warning(self, 'File header corrupted!', "The header data in the record file\n"+data.RecordFilename+"\nis invalid!\nNo data loaded.", QtGui.QMessageBox.Ok)
#        elif(r==4): #file is empty
#            QtGui.QMessageBox.warning(self, 'File empty!', "The data in the record file\n"+data.RecordFilename+"\ncontains no data!", QtGui.QMessageBox.Ok)
#        elif(r==9): #Record too old
#            QtGui.QMessageBox.warning(self, 'File version too old!', "Can not load record \n"+data.RecordFilename+"\nas it got created from SleepAnalyser version <1.6 and is not compatible with the current SleepAnalyser version.", QtGui.QMessageBox.Ok)
#        LoadFileList()

        if(data.demo==False): #We are on a N900
            #It gives miss behavior on PC, so only do on N900
            self.UpdateScreen()
            app.processEvents() #force ui update


        self.ui.lblStatus.setText(_("Processessing data..."))
        app.processEvents() #force ui update
        ProcessData(True, True)

        self.ui.lblStatus.setText(_("Rendering data..."))
        app.processEvents() #force ui update

        FilterData()

        self.ui.lblStatus.setText(_("Generating statistic..."))
        app.processEvents() #force ui update

        GenerateStatisticData()
#        data.Xscale=1 #default zoom level
        data.Xscale=2
#        data.GraphStart=max(records.LastIndex-data.arrXscale[data.Xscale]*775, 0)

#        data.ChangesInGraph=True #Make sure it only gets refreshed once


        self.ui.lblStatus.hide()
        self.Canvas.show()
        self.StatisticsCanvas.show()
        GenerateGraphBitmap(self, True)
        GenerateStatisticsBitmap(self)

    #Note: On the PC, the lable texts disappears here again until the window looses the focus, but it works fine on the N900, so I ignore it




    ###########################################################################
    ##
    ###########################################################################
    def NextRecord(self): ##(Class ViewWindow)
        data.arrRecordFilesIndex=(data.arrRecordFilesIndex+1)  % len(data.arrRecordFiles)
        data.RecordFilename=data.arrRecordFiles[data.arrRecordFilesIndex]
        self.ShowRecord()

    ###########################################################################
    ##
    ###########################################################################
    def LastRecord(self): ##(Class ViewWindow)
        data.arrRecordFilesIndex=(data.arrRecordFilesIndex-1)  % len(data.arrRecordFiles)
        data.RecordFilename=data.arrRecordFiles[data.arrRecordFilesIndex]
        self.ShowRecord()


    ###########################################################################
    ##
    ###########################################################################
    def ZoomIn(self): ##(Class ViewWindow)
        self.ui.lblStatus.setText(_("Zoom in..."))
        self.ui.lblStatus.show()
        app.processEvents() #force ui update
        Zoom(+1)

        self.ui.bZoomOut.setEnabled(True)
        if(data.Xscale==0): self.ui.bZoomIn.setEnabled(False)
        else: self.ui.bZoomIn.setEnabled(True)

        GenerateGraphBitmap(self, True)

        self.ui.lblStatus.hide()








    ###########################################################################
    ##
    ###########################################################################
    def ZoomOut(self): ##(Class ViewWindow)
        self.ui.lblStatus.setText(_("Zoom out..."))
        self.ui.lblStatus.show()
        app.processEvents() #force ui update
        Zoom(-1)
        self.ui.bZoomIn.setEnabled(True)
        if(data.Xscale==data.arrXscaleLen-1): self.ui.bZoomOut.setEnabled(False)
        else: self.ui.bZoomOut.setEnabled(True)

        GenerateGraphBitmap(self, True)

        self.ui.lblStatus.hide()


    ###########################################################################
    ##
    ###########################################################################
    def RaiseTrigger(self): ##(Class ViewWindow)
        self.ui.lblStatus.setText(_("Raise trigger level..."))
        self.ui.lblStatus.show()
        MoveTrigger(+1)
        app.processEvents() #force ui update

        GenerateGraphBitmap(self, True)

        GenerateStatisticData()
        GenerateStatisticsBitmap(self)
        self.ui.lblStatus.hide()


    ###########################################################################
    ##
    ###########################################################################
    def LowerTrigger(self): ##(Class ViewWindow)
        self.ui.lblStatus.setText(_("Lower trigger level..."))
        self.ui.lblStatus.show()
        MoveTrigger(-1)
        app.processEvents() #force ui update

        GenerateGraphBitmap(self, True)

        GenerateStatisticData()
        GenerateStatisticsBitmap(self)
        self.ui.lblStatus.hide()





    ###########################################################################
    ##
    ###########################################################################
    def DeleteRecord(self): ##(Class ViewWindow)
        self.ui.lblStatus.setText(_("Delete record..."))
        self.ui.lblStatus.show()
        reply = QtGui.QMessageBox.question(self, _("Delete record?"), _("Are you sure you want to delete this record:")+"\n"+data.RecordFilename, QtGui.QMessageBox.Yes, QtGui.QMessageBox.No)
        if reply == QtGui.QMessageBox.Yes:
            print "Delete RecordFile: "+data.RecordFilename
            os.remove(data.RecordFolder+data.RecordFilename)

            index=data.arrRecordFilesIndex
            LoadFileList()
            if(len(data.arrRecordFiles)>0):
#                data.RecordFilename=data.arrRecordFiles[l-1] #open last (youngest) record file
                data.arrRecordFilesIndex=index-1
                data.RecordFilename=data.arrRecordFiles[data.arrRecordFilesIndex]
            else: #no RecordFiles in RecordFolder, show "open file" dialog
                tmp=self.ShowFileDialog()
                if(tmp==""): #only continue if valid record file selected
                    self.closeEvent("") #no file selected, close
                    return
                tmp2=tmp.split("/")
                l=len(tmp2)
                data.RecordFilename=tmp2[l-1]
                data.RecordFolder=string.replace(tmp,  data.RecordFilename,  "") #get folder of selected file
                print "Set folder=",  data.RecordFolder,  ", file=",  data.RecordFilename
            self.ShowRecord()
            self.UpdateScreen()
        self.ui.lblStatus.hide()
        self.StatisticsCanvas.show()



    ###########################################################################
    ##
    ###########################################################################
    def OpenRecord(self): ##(Class ViewWindow)

        self.ui.lblStart.setText("")
        self.ui.lblEnd.setText("")
        self.ui.lblNote.setText("")
        self.Canvas.hide()
        self.StatisticsCanvas.hide()
        self.ui.lblStatus.setText(_("Opening file dialog..."))
        self.ui.lblStatus.show()


        tmp=self.ShowFileDialog()
        if(not os.path.exists(tmp)): #only continue if valid record file selected
            self.ui.lblStatus.hide()
            return
        tmp2=tmp.split("/")
        l=len(tmp2)
        data.RecordFilename=tmp2[l-1]
        data.RecordFolder=string.replace(tmp,  data.RecordFilename,  "") #get folder of selected file
        print "Set folder=",  data.RecordFolder,  ", file=",  data.RecordFilename
        LoadFileList()
        self.ShowRecord()
        self.ui.lblStatus.hide()
        self.StatisticsCanvas.show()


    ###########################################################################
    ##
    ###########################################################################
    def ShowFileDialog(self): ##(Class ViewWindow)
        data.RecordFolder=string.replace(data.RecordFolder, "~",  data.homedir) #workaround
        if(data.debug==1): print "Open file dialog in:",  data.RecordFolder
        tmp= QFileDialog.getOpenFileName(None,_("Open SleepAnalyser record"), data.RecordFolder, _("Record Files")+" (*.csv)")
        print "Selected file:",  tmp
        return tmp


    ###########################################################################
    ##
    ###########################################################################
    def OpenDate(self): ##(Class ViewWindow)
        DateWindow.show()
        DateWindow.DateChanged()


    ###########################################################################
    ##
    ###########################################################################
    def closeEvent(self, event): ##(Class ViewWindow)
        self.close()
        StartWindow.show()


    ###########################################################################
    ##
    ###########################################################################
    def EditNote(self): ##(Class ViewWindow)
        r=QInputDialog.getText(None,  _("Note about this record"),  _("Add a note to this record\n(leave empty for no note):"),  QLineEdit.Normal,  records.Note)
        if(r[1]==True): #OK pressed
            tmp=r[0]
            tmp=unicode(tmp)
            records.Note=tmp
#            self.ui.lblNote.setText("Note: "+records.Note)
            self.ui.lblNote.setText(records.Note)
            UpdateFileHeader() #rewrite record file (only do when recording is stopped!)



    ###########################################################################
    ##
    ###########################################################################
    def About(self): ##(Class ViewWindow)
        About(self)


    ###########################################################################
    ##
    ###########################################################################
    def ShowRecordData(self): ##(Class ViewWindow)
        AdditionalInfoWindow.UpdateScreen()
        self.hide()
        AdditionalInfoWindow.show()

    def ShowHelpWindow(self): ##(Class ViewWindow)
        HelpWindow.show()
        self.hide()



    def Export(self): ##(Class ViewWindow)
        filename=string.replace(data.RecordFilename, ".csv",  "")
        tmp=QFileDialog.getSaveFileName(None,_("Export graph as"), data.defaultRecordFolder+filename, _("PNG-Image")+" (*.png)") #;;JPEG-Image (*.jpg)
        if(tmp!=""):
            print "Export graph as :",  tmp
            r=data.GraphBitMap.save(tmp,  "png",  -1)
            if(r==True): print "Export was successful"
            else: QtGui.QMessageBox.warning(self, _("Could not export file"), _("Failed to save graph as image:")+"\n"+tmp, QtGui.QMessageBox.Ok)
#        print "beep2"""
#        self.show()
#        HelpWindow.show()
#        ViewWindow.show()




    def UpdateStatus(self, text): ##(Class ViewWindow)
        self.ui.lblStatus.setText(text)
        self.ui.lblStatus.show()
        app.processEvents() #force ui update





###########################################################################
## Class for Help Window
###########################################################################
class frmHelpWindow(QMainWindow):
    def __init__(self, parent):
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor
        if(data.demo==False): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack

        self.ui = Ui_HelpWindow()
        self.ui.setupUi(self)

#        TODO: support multilingual info file
        file = open(data.app_path+"Info.htm", 'r')
        txt=""
        for line in file: txt=txt+line+"\n"
        file.close()
        self.ui.textBrowser.setHtml(txt)

        self.setWindowTitle("SleepAnalyser ("+_("Help")+")")


    def closeEvent(self, event): ##(Class HelpWindow)
        self.close()
        StartWindow.show()





###########################################################################
## Class for Configuration Window
###########################################################################
class frmConfigWindow(QMainWindow):
    def __init__(self, parent):
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor
        if(data.demo==False): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack

        self.ui = Ui_ConfigWindow()
        self.ui.setupUi(self)
        self.ui.lblIcon.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"SleepAnalyser.png"))

        QtCore.QObject.connect(self.ui.bEditUserName, QtCore.SIGNAL("clicked()"), self.EditUserName)
        QtCore.QObject.connect(self.ui.SlrDelayStart,SIGNAL("valueChanged(int)"),self.DelaySliderMoved)
        QtCore.QObject.connect(self.ui.SlrYscale_Movement,SIGNAL("valueChanged(int)"),self.YscaleSliderMoved_Movement)
        QtCore.QObject.connect(self.ui.SlrYscale_Sleep,SIGNAL("valueChanged(int)"),self.YscaleSliderMoved_Sleep)


        QtCore.QObject.connect(self.ui.bOK, QtCore.SIGNAL("clicked()"), self.OK)
#        QtCore.QObject.connect(self.ui.bCancel, QtCore.SIGNAL("clicked()"), self.Cancel)


        #Add styles to buttons
        self.ui.bEditUserName.setStyleSheet(data.UIStyle)
        self.ui.SlrDelayStart.setStyleSheet(data.UIStyle)
        self.ui.SlrYscale_Movement.setStyleSheet(data.UIStyle)

        #Add styles to tick boxes
        self.ui.cOfflineMode.setStyleSheet(data.UIStyle)
        self.ui.cStopAfterAlarm.setStyleSheet(data.UIStyle)
        self.ui.cSilentProfile.setStyleSheet(data.UIStyle)

        self.setWindowTitle("SleepAnalyser ("+_("Settings")+")")
        self.ui.cOfflineMode.setText(_("Change to Offline Mode while recording"))
        self.ui.cSilentProfile.setText(_("Change to Silent Profile while recording"))
        self.ui.cStopAfterAlarm.setText(_("Stop recording after alarm occured"))
        self.ui.lblUser.setText(_("User name:"))
        self.ui.bEditUserName.setText(_("Change"))
        self.ui.tabWidget.setTabText(self.ui.tabWidget.indexOf(self.ui.tab_General), _("General"))
        self.ui.lblDelayStart.setText(_("Start delay:"))
        self.ui.lblYscale_Movement.setText(_("Y scaling movement (yellow):"))
        self.ui.lblYscale_Sleep.setText(_("Y scaling sleep graph (blue):"))
        self.ui.tabWidget.setTabText(self.ui.tabWidget.indexOf(self.ui.tab_Advanced), _("Advanced"))
        self.ui.lblMore.setText(_("More settings can be changed at your own risk in the file\n~/.SleepAnalyer/SleepAnalyser.conf."))
        self.ui.tabWidget.setTabText(self.ui.tabWidget.indexOf(self.ui.tab_More), _("More"))
        self.ui.bOK.setText(_("OK"))


    ###########################################################################
    ##
    ###########################################################################
    def UpdateScreen(self): ##(Class ConfigWindow)
        if(data.OfflineMode==True): self.ui.cOfflineMode.setChecked(True)
        else: self.ui.cOfflineMode.setChecked(False)
        if(data.SilentProfile==True): self.ui.cSilentProfile.setChecked(True)
        else: self.ui.cSilentProfile.setChecked(False)
        if(data.StopAfterAlarm==True): self.ui.cStopAfterAlarm.setChecked(True)
        else: self.ui.cStopAfterAlarm.setChecked(False)
        self.ui.lblUser.setText(_("User name:")+" "+data.user)
        self.ui.lblDelayStart.setText(_("Start delay:")+" "+str(data.DelayStartTime)+" seconds")
        self.ui.SlrDelayStart.setValue(int(data.DelayStartTime))

        self.ui.lblYscale_Movement.setText(_("Y scaling movement (yellow):")+" "+str(data.Yscale))
        self.ui.SlrYscale_Movement.setValue(int(data.Yscale))


        self.ui.lblYscale_Sleep.setText(_("Y scaling sleep graph (blue):")+" "+str(data.FilterScale))
        self.ui.SlrYscale_Sleep.setValue(int(data.FilterScale))



#    def Cancel(self):
#        self.hide()
#        StartWindow.show()



    def OK(self):
#        self.save()
        self.close()
#        StartWindow.show()



    ###########################################################################
    ##
    ###########################################################################
    def closeEvent(self, event):  ##(Class ConfigWindow)
        self.save() #closing the window is equal to saving
#        self.close()
#        event.ignore() #do not close
        StartWindow.show()





    def save(self):
#        print "save"
        if(self.ui.cOfflineMode.isChecked()==True):
            data.OfflineMode=True
        else:
            data.OfflineMode=False
        config.set('main', 'OfflineMode', str(data.OfflineMode))

        if(self.ui.cSilentProfile.isChecked()==True):
            data.SilentProfile=True
        else:
            data.SilentProfile=False
        config.set('main', 'SilentProfile', str(data.SilentProfile))

        if(self.ui.cStopAfterAlarm.isChecked()==True):
            data.StopAfterAlarm=True
        else:
            data.StopAfterAlarm=False
        config.set('main', 'StopAfterAlarm', str(data.StopAfterAlarm))

        config.set('main', 'user', str(data.user.encode( "utf-8" )))
        config.set('main', 'DelayStartTime', str(data.DelayStartTime))
        config.set('main', 'Yscale', str(data.Yscale))
        config.set('main', 'FilterScale', str(data.FilterScale))



    ###########################################################################
    ##
    ###########################################################################
    def EditUserName(self): ##(Class ConfigWindow)
        EditUserName()
        self.ui.lblUser.setText(_("User name:")+" "+data.user)


    ###########################################################################
    ##
    ###########################################################################
    def DelaySliderMoved(self): ##(Class ConfigWindow)
        x=self.ui.SlrDelayStart.value()
        data.DelayStartTime=x
        self.ui.lblDelayStart.setText(_("Start delay:")+" "+str(data.DelayStartTime)+" "+_("sec"))


    ###########################################################################
    ##
    ###########################################################################
    def YscaleSliderMoved_Movement(self): ##(Class ConfigWindow)
        x=self.ui.SlrYscale_Movement.value()
        data.Yscale=int(x)
        self.ui.lblYscale_Movement.setText(_("Y scaling movement (yellow):")+" "+str(data.Yscale))


    ###########################################################################
    ##
    ###########################################################################
    def YscaleSliderMoved_Sleep(self): ##(Class ConfigWindow)
        x=self.ui.SlrYscale_Sleep.value()
        data.FilterScale=int(x)
        self.ui.lblYscale_Sleep.setText(_("Y scaling sleep graph (blue):")+" "+str(data.FilterScale))



###########################################################################
## Class for Date Window
###########################################################################
class frmDateWindow(QMainWindow):
    def __init__(self, parent=None):
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor
        if(data.demo==False): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack

        self.ui = Ui_DateWindow()
        self.ui.setupUi(self)

        if(data.app_path!="./"): #only needed if not started in app path
            self.ui.bOpen.setIcon(QIcon(data.app_path+"img/"+"zoom.png"))

        QtCore.QObject.connect(self.ui.cal, QtCore.SIGNAL('selectionChanged()'), self.DateChanged)
        QtCore.QObject.connect(self.ui.bOpen, QtCore.SIGNAL("clicked()"), self.OpenRecord)

        self.ui.cal.showToday()

        self.setWindowTitle("SleepAnalyser ("+_("Select date)")+")")


    ###########################################################################
    ##
    ###########################################################################
    def DateChanged(self): ##(Class ConfigWindow)
        date = self.ui.cal.selectedDate()
        date=str(date.toPyDate())

#        print "Load list of all csv files in folder:",  data.RecordFolder + " with date " + date
        output = os.popen("cd "+str(data.RecordFolder)+"; ls -C1 *"+date+"*.csv").read() # run ls and make the output look like a file I can read
        arr=output.split("\n")

        amount=len(arr)-1

#TODO: visualise dates with records

        txt=string.replace(_("Found % Record(s) for this date."),"%", str(amount))
        self.ui.lblResult.setText(txt)
        if(amount>0):
            self.ui.bOpen.setEnabled(True)
            data.RecordFileSelectedDate=arr[0] #select first file of that date for opening

            for i in range(0,  len(data.arrRecordFiles)-1):
                if(data.arrRecordFiles[i]==data.RecordFileSelectedDate):
#                    print "found:",  i
                    data.arrRecordFilesIndex=i
                    break

        else: self.ui.bOpen.setEnabled(False)


    ###########################################################################
    ##
    ###########################################################################
    def OpenRecord(self): ##(Class ConfigWindow)
        data.RecordFilename=data.RecordFileSelectedDate
        print "Set folder=",  data.RecordFolder,  ", file=",  data.RecordFilename
        DateWindow.hide()
        ViewWindow.ShowRecord()








###########################################################################
## Class for Date Window
###########################################################################
class frmAdditionalInfoWindow(QMainWindow):
    def __init__(self, parent=None):
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor
        if(data.demo==False): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack

        self.ui = Ui_AdditionalInfoWindow()
        self.ui.setupUi(self)

        self.setWindowTitle("SleepAnalyser ("+_("Record Info\'s")+")")


    ###########################################################################
    ##
    ###########################################################################
    def UpdateScreen(self): ##(Class AdditionalInfoWindow)
        shortfolder=string.replace(data.RecordFolder, data.homedir, "~/")
        shortfolder=string.replace(shortfolder, "//", "/") #workaround

        d0=strftime("%a. %d.%m.%Y  %H:%M:%S", records.StartTime_tuple)
#        EndTime_seconds=Tuple2Seconds(records.StartTime_tuple)+records.Index*records.Sampling_interval*records.Value_smoothing/1000
#        EndTime_tuple=Seconds2Tuple(records.EndTime_seconds)
        if(strftime("%d.%m.%Y", records.StartTime_tuple)==strftime("%d.%m.%Y", records.EndTime_tuple)):
            d1=strftime("%H:%M:%S", records.EndTime_tuple) #on same day, only show date once
        else:
            d1=strftime("%a. %d.%m.%Y  %H:%M:%S", records.EndTime_tuple)

        diff_sec=records.Index*records.Sampling_interval*records.Value_smoothing/1000
#        diff=time.strftime("%X", time.gmtime(diff_sec))
        diff=time.strftime("%H:%M:%S", time.gmtime(diff_sec))

        if(diff_sec>86400): diff="1 day, "+diff #longer than 24h

        if(records.AlarmIsSet==True):
            a0=strftime("%a. %d.%m.%Y  %H:%M:%S", records.AlarmTimeWindowStart_tuple)
            if(strftime("%d.%m.%Y", records.AlarmTimeWindowStart_tuple)==strftime("%d.%m.%Y", records.AlarmTimeWindowEnd_tuple)):
                a1=strftime("%H:%M:%S", records.AlarmTimeWindowEnd_tuple) #on same day, only show date once
            else:
                a1=strftime("%a. %d.%m.%Y  %H:%M:%S", records.AlarmTimeWindowEnd_tuple)
            alarm="<b>"+_("Alarm Time Window:")+"</b> "+a0+" - "+a1 + "<br>"
            if(records.AlarmTime_seconds!=-1): #alarm was set
                if(records.EndTime_seconds>=records.AlarmTime_seconds-60): #alarm got set off during recording (-60 is workaround)
                    alarm=alarm+"<b>"+_("Alarm:")+"</b> "+_("Alarm went off on:")+" "+strftime("%a. %d.%m.%Y  %H:%M:%S", records.AlarmTime_tuple)+"<br>"
                    if(records.AlarmTimeWindowEnd_seconds<=records.AlarmTime_seconds):
                        alarm=alarm+"<b>"+_("Alarm reason:")+"</b> "+_("End of alarm time window")+"<br>"
                    else:
                        alarm=alarm+"<b>"+_("Alarm reason:")+"</b> "+_("Movements over trigger level")+"<br>"
                else:
                    alarm=alarm+"<b>"+_("Alarm:")+"</b> "+_("Alarm never went off")+"<br>"
            else:
                alarm=alarm+"<b>"+_("Alarm:")+"</b> "+_("Alarm never went off")+"<br>"
        else:
            alarm="<b>"+_("Alarm:")+"</b> "+_("No alarm was set")+"<br>"

        txt= "<b>"+_("File:")+"</b> "+str(shortfolder)+str(data.RecordFilename+"<br>") + \
        "<b>"+_("Date/Time:")+"</b> "+d0+" - "+d1+"<br>" + \
        "<b>"+_("Duration:")+"</b> "+ diff+"<br>" + \
        "<b>"+_("Record entries:")+"</b> "+str(records.Index)+" (1/"+str(int(records.Sampling_interval*records.Value_smoothing/1000))+" "+_("seconds")+")<br>" +\
        "<b>"+_("Record version:")+"</b> "+str(records.SWversion) + "<br>" + \
        alarm +\
        "<b>"+_("Trigger level:")+"</b> "+str(records.Trigger) + "<br>" + \
        "<b>"+_("Condition of record:")+"</b> "+str(records.status)
#        "<b>"+_("User:")+"</b> "+str(records.User) +"<br>" + \ ##Disabled due charset errors
#        "<b>"+_("Note:")+"</b> "+str(records.Note)+"<br>" +\ ##Disabled due charset errors



#        t_total=(records.EndTime_seconds-records.StartTime_seconds)
#        print records.EndTime_seconds,  records.StartTime_seconds,  t_total
#        if(t_total==0):  t_total=1 #division by zerro prevention
#        pa=records.arrStatData[0][0]*100/t_total
#        pb=records.arrStatData[1][0]*100/t_total
#        pc=records.arrStatData[2][0]*100/t_total
##        pd=records.arrStatData[3][0]*100/t_total
#        pl=100-pa-pb-pc

        txt=txt+"<hr>"
        txt=txt+"<b>"+_("Statistics:")+"</b><table>\n" + \
                "<tr bgcolor=#DCDCDC><td><b>"+_("Time without<br>movement:")+" </b></td>   <td><b>"+_("Total:")+" </b></td>                                     <td><b>"+_("Maximum:")+"</b></td>                                    <td><b>"+_("Count:")+" </b></td>                     <td colspan=2><b>"+_("Time percent:")+"</b></td>\n" + \
                "<tr><td bgcolor=#DCDCDC><b>"+"&gt; 7 "+_("minutes:")+"</b></td>         <td>"+strftime("%Hh %Mm ", time.gmtime(records.arrStatData[3][0]))+"</td>    <td>-</td>    <td>-</td>    <td>"+str(records.arrStatData[3][3])+"%</td><td><img src=\""+data.app_path+"img/"+"State4.png"+"\" width="+str(records.arrStatData[3][3]*2)+" height=25</td></tr>\n" + \
                "<tr><td bgcolor=#DCDCDC><b>"+"&lt; 7 "+_("minutes:")+"</b></td>         <td>"+strftime("%Hh %Mm ", time.gmtime(records.arrStatData[2][0]))+"</td>    <td>"+strftime(" %Hh %Mm ", time.gmtime(records.arrStatData[2][2]))+"</td>    <td>"+str(records.arrStatData[2][1])+"</td>    <td>"+str(records.arrStatData[2][3])+"%</td><td><img src=\""+data.app_path+"img/"+"State3.png"+"\" width="+str(records.arrStatData[2][3]*2)+" height=25</td></tr>\n" + \
                "<tr><td bgcolor=#DCDCDC><b>"+"&lt; 15 "+_("minutes:")+"</b></td>        <td>"+strftime("%Hh %Mm ", time.gmtime(records.arrStatData[1][0]))+"</td>    <td>"+strftime(" %Hh %Mm ", time.gmtime(records.arrStatData[1][2]))+"</td>    <td>"+str(records.arrStatData[1][1])+"</td>    <td>"+str(records.arrStatData[1][3])+"%</td><td><img src=\""+data.app_path+"img/"+"State2.png"+"\" width="+str(records.arrStatData[1][3]*2)+" height=25</td></tr>\n" + \
                "<tr><td bgcolor=#DCDCDC><b>"+"&lt; 30 "+_("minutes:")+"</b></td>        <td>"+strftime("%Hh %Mm ", time.gmtime(records.arrStatData[0][0]))+"</td>    <td>"+strftime(" %Hh %Mm ", time.gmtime(records.arrStatData[0][2]))+"</td>    <td>"+str(records.arrStatData[0][1])+"</td>    <td>"+str(records.arrStatData[0][3])+"%</td><td><img src=\""+data.app_path+"img/"+"State1.png"+"\" width="+str(records.arrStatData[0][3]*2)+" height=25</td></tr>\n" + \
              "</table>\n"

        txt=txt+"<hr>"
        txt=txt+"<b>"+_("Sampling interval:")+"</b> "+str(records.Sampling_interval) + " ms<br>" + \
        "<b>"+_("Sampling smoothing:")+"</b> "+str(records.Sample_smoothing) + "<br>" + \
        "<b>"+_("Value smoothing:")+"</b> "+str(records.Value_smoothing)

        self.ui.textBrowser.setHtml("<html><body>"+txt+"</html></body>")


    ###########################################################################
    ##
    ###########################################################################
    def closeEvent(self, event): ##(Class AdditionalInfoWindow)
        self.close()
        ViewWindow.show()







###########################################################################
## Class for Setting the alarm time Window
###########################################################################
class frmAlarmTimeWindow(QMainWindow):
    def __init__(self, parent=None):
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor

        if(data.demo==False): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack
#        self.setWindowFlags(Qt.SplashScreen)
        self.ui = Ui_AlarmTimeWindow()
        self.ui.setupUi(self)
        self.ui.lblIcon.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"SleepAnalyser.png"))

        QtCore.QObject.connect(self.ui.bSetAlarm, QtCore.SIGNAL("clicked()"), self.SetAlarm)

        #Add styles to buttons
        self.ui.bSetAlarm.setStyleSheet(data.UIStyle)


        if(data.app_path!="./"): #only needed if not started in app path
            self.ui.bSetAlarm.setIcon(QIcon(data.app_path+"img/"+"SleepAnalyser.png"))

        self.setWindowTitle("SleepAnalyser ("+_("Set alarm")+")")
        txt=_("The Alarm will go off sometimes during the set time window (as soon as your movement is over the trigger level), but latest at the set alarm time.")
        self.ui.lblNote.setText(txt)
        self.ui.bSetAlarm.setText(_("Set alarm"))
        self.ui.label_7.setText(_("Alarm Time:"))
        self.ui.label_9.setText(_("Time Window:"))
        self.ui.label_8.setText(_("Hour:"))
        self.ui.label_10.setText(_("Minutes:"))
        self.ui.label_11.setText(_("Minutes:"))


    ###########################################################################
    ##
    ###########################################################################
    def UpdateScreen(self): ##(Class AlarmTimeWindow)
        if(data.AlarmHour==-1 or data.AlarmMinute==-1):
            data.AlarmHour=7
            data.AlarmMinute=0
        if(data.debug==1): print "Alarm time:",  data.AlarmHour, ":",  data.AlarmMinute
        self.ui.lstAlarmHour.setCurrentRow(int(data.AlarmHour))
        self.ui.lstAlarmMinute.setCurrentRow(int(data.AlarmMinute)/5)
        self.ui.lstAlarmTimeWindow.setCurrentRow(int(data.AlarmTimeWindow)/5-1)




    ###########################################################################
    ##
    ###########################################################################
    def SetAlarm(self): ##(Class AlarmTimeWindow)

        InitRecord() #clear all old data

#        print "Set Alarm"
        data.QuickMode=False #run in normal mode

#        RecordWindowCanvas = RecordWindowCanvas(RecordWindow)

        try:
            data.AlarmHour=int(self.ui.lstAlarmHour.currentItem().text())
            data.AlarmMinute=int(self.ui.lstAlarmMinute.currentItem().text())
            data.AlarmTimeWindow=int(self.ui.lstAlarmTimeWindow.currentItem().text())
            config.set('main', 'AlarmHour', str(data.AlarmHour))
            config.set('main', 'AlarmMinute', str(data.AlarmMinute))
            config.set('main', 'AlarmTimeWindow', str(data.AlarmTimeWindow))

            now=int(strftime("%H", time.localtime()))*3600+int(strftime("%M", time.localtime()))*60
            al=int(data.AlarmHour)*3600+int(data.AlarmMinute)*60
            alarmdiff=al-now
            if(alarmdiff<0):  #set to tomorrow if time already is elapsed today
                alarmdiff=alarmdiff+86400

            records.AlarmTime_seconds=int((records.StartTime_seconds+alarmdiff)/60)*60 #set seconds to zero
            records.AlarmTime_tuple=Seconds2Tuple(records.AlarmTime_seconds)

            records.AlarmTimeWindowEnd_tuple=records.AlarmTime_tuple
            records.AlarmTimeWindowEnd_seconds=records.AlarmTime_seconds

            records.AlarmTimeWindow_minutes=int(self.ui.lstAlarmTimeWindow.currentItem().text())

            records.AlarmTimeWindowStart_seconds=records.AlarmTimeWindowEnd_seconds-records.AlarmTimeWindow_minutes*60
            records.AlarmTimeWindowStart_tuple=Seconds2Tuple(records.AlarmTimeWindowStart_seconds)

#            print records.AlarmTime_tuple,  records.AlarmTimeWindow_minutes
#            print records.AlarmTimeWindowStart_tuple
#            print records.AlarmTimeWindowEnd_tuple

        except: #If nothing selected (workaround)
            print "Selected time not valid!"
#            print sys.exc_info()
            StartWindow.show()
            return

        DeleteAlarm() #delete old alarm if existing
        records.AlarmIsSet=True
        SetAlarm() #set alarm for the last possible time
        InitMeasurement()

        RecordWindow.show()
        AlarmTimeWindow.hide()
        RecordWindow.UpdateScreenStart()
#        RecordWindowCanvas.show()


    def closeEvent(self, event): ##(Class AlarmTimeWindow)
        self.close()
        StartWindow.show()





###########################################################################
## Class for Setting the alarm time Window
###########################################################################
class frmAskForKeepWindow(QMainWindow):
    def __init__(self, parent=None):
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor

        if(data.demo==False): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack

        self.ui = Ui_AskForKeepWindow()
        self.ui.setupUi(self)

        QtCore.QObject.connect(self.ui.bstop_keepRecord, QtCore.SIGNAL("clicked()"), self.KeepRecord)
        QtCore.QObject.connect(self.ui.bstop_removeRecord, QtCore.SIGNAL("clicked()"), self.RemoveRecord)
        QtCore.QObject.connect(self.ui.bcontinue, QtCore.SIGNAL("clicked()"), self.ContinueRecording)

        #Add styles to buttons
        self.ui.bstop_keepRecord.setStyleSheet(data.UIStyle)
        self.ui.bstop_removeRecord.setStyleSheet(data.UIStyle)
        self.ui.bcontinue.setStyleSheet(data.UIStyle)

        if(data.app_path!="./"): #only needed if not started in app path
            self.ui.bstop_keepRecord.setIcon(QIcon(data.app_path+"img/"+"zoom.png"))
            self.ui.bstop_removeRecord.setIcon(QIcon(data.app_path+"img/"+"edittrash.png"))
            self.ui.bcontinue.setIcon(QIcon(data.app_path+"img/"+"krec.png"))

        self.setWindowTitle("SleepAnalyser ("+_("Recording")+")")
        self.ui.bcontinue.setText(_("Continue recording"))
        self.ui.bstop_keepRecord.setText(_("Stop recording and keep the record"))
        self.ui.bstop_removeRecord.setText(_("Stop recording and delete the record"))



    ###########################################################################
    ##
    ###########################################################################
    def UpdateScreen(self): ##(Class AskForKeepWindow)
#        s="Do you really want to stop recording?"
#        if(records.AlarmIsSet==True): s=s+"\nThis will also remove the set alarm!"
#        s=s+"\n\nAnd do you want to keep the record?"
#        self.ui.lblNote.setText(s)
        pass


    ###########################################################################
    ##
    ###########################################################################
    def KeepRecord(self): ##(Class AskForKeepWindow)
        print "Stop recording and keep record"
#        global ViewWindow
        StopMeasurement()
        DeleteAlarm() #delete Alarm if existing
        records.AlarmIsSet=False
        data.LastRecord=data.RecordFilename #set to corrent file for viewwindow
        config.set('main', 'LastRecord', str(data.LastRecord))
        print "Show record in ViewWindow:",  data.LastRecord
        RecordWindow.hide()
        self.hide()
        StartWindow.ShowViewWindow()
        app.processEvents() #force ui update
        SwitchModes(False) #change back to normal mode and general profile if requested



    ###########################################################################
    ##
    ###########################################################################
    def RemoveRecord(self): ##(Class AskForKeepWindow)
        print "Stop recording and remove record"
#        global StartWindow,  RecordWindow
        StopMeasurement()
        DeleteAlarm() #delete Alarm if existing
        records.AlarmIsSet=False
        print "Delete RecordFile: "+data.RecordFilename
        os.remove(data.RecordFolder+data.RecordFilename)
        RecordWindow.hide()
        self.hide()
        StartWindow.show()
        app.processEvents() #force ui update
        SwitchModes(False) #change back to normal mode and general profile if requested



    ###########################################################################
    ##
    ###########################################################################
    def ContinueRecording(self): ##(Class AskForKeepWindow)
        print "continue recording"
        self.hide()
        RecordWindow.show()
        return


    ###########################################################################
    ##
    ###########################################################################
    def closeEvent(self, event): ##(Class AskForKeepWindow)
        event.ignore() #do not close
        self.KeepRecord() #interprete closing as "stop and keep file"

















###########################################################################
##
###########################################################################
class DisplayStatusDaemon:
    def __init__(self):
        if(data.demo==False): #We are on a N900
            loop = DBusQtMainLoop(set_as_default=True)
            system_bus = dbus.SystemBus()
            system_bus.add_signal_receiver(self.DisplayStatus,
                                           path='/com/nokia/mce/signal',
                                        signal_name='display_status_ind',
                                        dbus_interface='com.nokia.mce.signal')

            if(data.debug==1): print "DisplayStatusDaemon started"




    ###########################################################################
    ##
    ###########################################################################
    def DisplayStatus(self, *params):
        global RecordWindow
#        print "1", len(params)
#        print params
        tmp=str(params)
        tmp=string.split(tmp, "'")
#        print tmp
        if(tmp[1]=="off"): data.DisplayIsOn=False
        elif(tmp[1]=="on"):
            data.DisplayIsOn=True
#            if(data.run==True):
            if(RecordWindow.isHidden()==False):
                if(data.debug==1): print "Update RecordWindow screen & graph"
#                data.ChangesInGraph=True

                GenerateStatisticData()
                GenerateGraphBitmap(RecordWindow, False)
                RecordWindow.Canvas.update()
                RecordWindow.UpdateScreen()








###########################################################################
##
###########################################################################
class MyGraphCanvas(QtGui.QWidget):
    def __init__(self, parent):
        QMainWindow.__init__(self, parent)
        self.setGeometry(0, 0, 770, data.GraphBitMapHeight)

#        print "Create canvas with:",  770,  data.Ysize,  "(Size will change depending on graph len)"


    ###########################################################################
    ##
    ###########################################################################
    def paintEvent(self, event):
        if(data.DisplayIsOn==False):
            if(data.debug==1): print "-> o Do not paint event (Graph), Display is off!"
            return
        else:
            if(data.debug==1): print "-> x paint event (Graph)"

        paint = QtGui.QPainter()
        paint.begin(self)
#        paint.setBackgroundMode(QtCore.Qt.TransparentMode)
        paint.drawPixmap(0, 0, data.GraphBitMap) #load graph from Bitmap
        paint.end()

        #Scroll graph to end, as there is the most interesting part
        if(data.ScrollToEnd==True):
            if(data.run==True): RecordWindow.ui.ScrollArea.ensureVisible(data.GraphBitMapLen+100, 0, 0 , 0)
            else:
                ViewWindow.ui.ScrollArea.ensureVisible(data.GraphBitMapLen+100, 0, 0 , 0)
            data.ScrollToEnd=False

###########################################################################
##
###########################################################################
class MyStatCanvas(QtGui.QWidget):
    def __init__(self, parent):
        QMainWindow.__init__(self, parent)
        self.setGeometry(406, 280, data.StatisticsBitMapLen,  data.StatisticsBitMapHeight)


    ###########################################################################
    ##
    ###########################################################################
    def paintEvent(self, event):
        if(data.DisplayIsOn==False):
            if(data.debug==1): print "-> o Do not paint event (Statistics), Display is off!"
            return
        else:
            if(data.debug==1): print "-> x paint event (Statistics)"

        paint = QtGui.QPainter()
        paint.begin(self)
        paint.setBackgroundMode(QtCore.Qt.TransparentMode)
        paint.drawPixmap(0, 0, data.StatisticsBitMap) #load statistics from Bitmap
        paint.end()










###########################################################################
##
###########################################################################
def GenerateGraphBitmap(self, ShowFiltered=False):
#    if(data.DisplayIsOn==False):
#        if(data.debug==1): print "-> o Do not Generate Graph Bitmap, Display is off!"
#        return
#    else:
    if(data.debug==1): print "Generate Graph Bitmap"

    Xscale=data.arrXscale[data.Xscale]
#        data.GraphBitMapLen=max((records.LastIndex+5)/Xscale + 2 ,  770)
    if(ShowFiltered==False): #recording
        data.GraphBitMapLen=max((records.LastIndex+5)/Xscale + 2 ,  735)
    else: #view window, show filtered graph
        data.GraphBitMapLen=max((records.LastIndex+5)/Xscale + 2 ,  649) #628 minimum width
    data.GraphBitMapHeight=data.Ysize+34

#    print "GraphBitMapLen:",  data.GraphBitMapLen,  ", GraphBitMapHeight:",  data.GraphBitMapHeight ,  ", Ysize is:", data.Ysize
    try:
        data.GraphBitMap = QtGui.QPixmap(data.GraphBitMapLen, data.GraphBitMapHeight) #create bitmap for graph
        DrawGraph(data.GraphBitMap,  QtGui, ShowFiltered) # draw graph
    except:
        print "Error while creating the graph bitmap"

#    ViewWindowCanvas.setGeometry(0, 0, data.GraphBitMapLen, data.GraphBitMapHeight)  #set size of canvas to graph
    self.Canvas.setGeometry(0, 0, data.GraphBitMapLen, data.GraphBitMapHeight)  #set size of canvas to graph
#    print "Set size of canvas to :",  data.GraphBitMapLen, "x",  data.GraphBitMapHeight

    self.ui.CanvasWidget.setMinimumSize(data.GraphBitMapLen, data.GraphBitMapHeight)  #set size of canvas widget to graph

    data.ScrollToEnd=True

    self.ui.CanvasWidget.hide()
    self.ui.CanvasWidget.show()
    self.UpdateScreen()






###########################################################################
##
###########################################################################
def GenerateStatisticsBitmap(self):
#    if(data.DisplayIsOn==False):
#        if(data.debug==1): print "-> o Do not Generate Statistics Bitmap, Display is off!"
#        return
#    else:
    if(data.debug==1): print "Generate Statistics Bitmap"

#    try: #TESTING
    data.StatisticsBitMap = QtGui.QPixmap(data.StatisticsBitMapLen,  data.StatisticsBitMapHeight) #create bitmap for graph
    DrawStatistic(data.StatisticsBitMap,  QtGui, QtCore) #draw statistics
#    except:
#        print "Error while creating the statistics bitmap"


#    self.ui.CanvasWidget.hide()
#    self.ui.CanvasWidget.show()





















###########################################################################
main() # Now we can start it.
