#!/usr/bin/env python
###########################################################################
## Name: 	       SleepAnalyser
## Description:  Sleep Analyser records your movement during your sleep. It is able to visualise it on a graph to show
##                      how much you move during your sleep. This can help to indicate how you sleep. It also has an alarm 
##                      function. You can set the alarm and a time window (ex. 30 minutes). The Alarm will then go off
##                      sometimes during the time window (as soon as you move more), but latest at the set alarm.
##                      Old records can be visualised and you can load records from your friends. SleepAnalyser also has a
##                      test function. It will record and visualise much faster. Also it will make a beep when ever your
##                      movement goes over the trigger level. 
## Copyright:     George Ruinelli, george@ruinelli.ch
## Licence:        GPL
###########################################################################

import sys
import random
import time
from time import strftime
import math
import array
import string

from PyQt4.QtCore import * 
from PyQt4.QtGui import * 

import ConfigParser
import os, os.path

import dbus
from dbus.mainloop.qt import DBusQtMainLoop

import records #record data
import data #own data



#own functions
from functions_general import *
from functions_file import *
from functions_record import *
from functions_alarm import *

#GUIs
from Ui_StartWindow import *
from Ui_RecordWindow import *
from Ui_ViewWindow import *
from Ui_ConfigWindow import *
from Ui_HelpWindow import *
from Ui_DateWindow import *
from Ui_AlarmTimeWindow import *
from Ui_AskForKeepWindow import *
from Ui_AdditionalInfoWindow import *




  
###########################################################################
## Main function
###########################################################################
def main():
    global app,  StartWindow,  RecordWindow,  ViewWindow ,  ConfigWindow,  DateWindow,  AlarmTimeWindow,  AskForKeepWindow,  AdditionalInfoWindow,  HelpWindow
    global config
   
    try:
        if(sys.argv[1]=="debug"): 
            print "Enable Debug mode"
            data.debug=1
    except:
        data.debug=0
        
    #if not on the N900, a acceleration sensor has to be simulated
    if(os.path.exists("/sys/class/i2c-adapter/i2c-3/3-001d/coord")): data.demo=False	#runs on real N900 hardware
    else: data.demo=True	#runs on other hardware/OS

    if(data.demo==True):
        print "Runs not on a N900 => Use randomizer for acceleration sensor data"
        data.configfolder=data.devconfigfolder
        data.logfolder=data.devlogfolder
    else: data.logfolder=data.defaultlogfolder

    app = QApplication(sys.argv)
    
    config=LoadConfig()
    
#    if(data.Ysize==120): #workaround for wrong defaults
#        data.Ysize=121
#        data.Yscale=1
#        data.Trigger=30  
#        config.set('main', 'Ysize', str(data.Ysize))      
#        config.set('main', 'Yscale', str(data.Yscale))      
#        print "changed default parameters for Yscale and Ysize"
    
    
    data.app_path=os.path.dirname(__file__)
    if(data.app_path==""): data.app_path="./"
    else: data.app_path=data.app_path+"/"
    
    
    
    #Load Stylesheet for buttons
    try:
        print "Load user defined style sheet"
        file = open(data.configfolder+"UIStyle.css", 'r')
        data.UIStyle=""
        for line in file: data.UIStyle=data.UIStyle+line+"\n"
        file.close()
    except:
        print "User defined style sheet not existing or not readable, load default one"
        file = open(data.app_path+"UIStyle.css", 'r')
        data.UIStyle=""
        for line in file: data.UIStyle=data.UIStyle+line+"\n"
        file.close()
        try:
            print "Create user defined style sheet in",  data.configfolder
            file= open(data.configfolder+"UIStyle.css", 'w')
            file.write(data.UIStyle)
            file.close()
        except:
            print "Could not create user defined style sheet"
        
    
    
    
    
    StartWindow = frmStartWindow()    
    StartWindow.show()    
    StartWindow.UpdateScreenStart()
    
    # Create subwindows
    RecordWindow = frmRecordWindow(StartWindow)
    ConfigWindow = frmConfigWindow(StartWindow)
    ViewWindow = frmViewWindow(StartWindow)
    HelpWindow = frmHelpWindow(StartWindow)
    AlarmTimeWindow = frmAlarmTimeWindow(StartWindow)
    DateWindow = frmDateWindow(ViewWindow)
    AdditionalInfoWindow = frmAdditionalInfoWindow(ViewWindow)
    AskForKeepWindow = frmAskForKeepWindow(RecordWindow)
    
    # Hide SubWindows at first
    RecordWindow.hide()
    ViewWindow.hide()
    ConfigWindow.hide()
    AlarmTimeWindow.hide()
    DateWindow.hide()    
    AdditionalInfoWindow.hide()    
    AskForKeepWindow.hide()    
    
    #get user name for in log files
    if(data.user==""):
        EditUserName()



    DisplayStatusDaemon()
    
    sys.exit(app.exec_())






###########################################################################
## 
###########################################################################
def EditUserName():
#      r=QInputDialog.getText(None,  "Add Note",  "Add a note to this record:\n(leave empty for no note)",  QLineEdit.Normal,  records.Note)
#        if(r[1]==True): #OK pressed
    r = QInputDialog.getText(None, "SleepAnalyser", "Please enter your name.\n"+ \
     "It will be used in the records file, this makes it easier\n" + \
     "when you want to share your sleep pattern with others.", 0,  data.user)
#    if(r != ""):
    if(r[1]==True): #OK pressed
        data.user=unicode(r[0])
        if(data.user==""): data.user="-"
        config.set('main', 'user', str(data.user.encode( "utf-8" )))  
        
    print "User name:",  data.user






###########################################################################
##  Gets called every second
###########################################################################
def MeasurementTimerLoop(): #(interval: sampling*sample_smoothing*value_smoothing)
    r=GetSensorData()     
    if(r==1): #new data got added to array
        ProcessData(False)   
        ProcessAlarm()        
        
        WriteIntoRecordFile()
        
#        GraphLen=records.Index/data.arrXscale[data.Xscale]
#        data.GraphStart=max(GraphLen-779, 0)
        
        data.ScrollToEnd=True
        if(data.DisplayIsOn==True): 
            GenerateStatisticData()   
            GenerateGraphBitmap(RecordWindow)    
    
        if(records.Index==records.MaxArrayDataLen-1): 
            print "Stopped, array is full"
            data.Status="Stopped, array is full!"
            QtGui.QMessageBox.warning(RecordWindow, 'Memory is full!', "You reached the end of the recording capacity!\nThe recording stops now.", QtGui.QMessageBox.Ok)
#            RecordWindow.StopMeasurement()
            StopMeasurement() #but keep alarm active!
    
    
    if(data.DisplayIsOn==True):
        if(data.debug==1): print "-> o Update screen"
        RecordWindow.UpdateScreen()      
    else:
        if(data.debug==1): print "-> x Do not update screen, Display is off!"
        
  
#    print data.AlarmIsActive,  data.StopAfterAlarm
    if(data.AlarmIsActive==True):
#        print "Alarm is active"
        if(data.StopAfterAlarm==True):
#            print "Stop measurement, alarm is active"
            data.Status="Alarm got activated, stopped recording"
            StopMeasurement() #but keep alarm active!
  


###########################################################################
## Start the measurement
###########################################################################
def InitMeasurement():
    global app

    data.sample_index=0
    if(data.QuickMode==False): data.Status="Recording..."
    else: data.Status="Recording in Test Mode..."

    records.User=data.user
    records.SWversion=data.version
    records.Sampling_interval=data.Sampling_interval
    records.Sample_smoothing=data.Sample_smoothing
    records.Value_smoothing=data.Value_smoothing
    records.Trigger=data.Trigger #save Trigger level on start
    records.statusID="OK"

    if(data.debug==1): PrintRecordData()
    
    #set scale and sampling interval
    if(data.QuickMode==False):
        data.interval=data.Sampling_interval #normal
        data.Xscale=1 #set to default zoom level
    else:  
        data.interval=10 #Fast mode for testing
        data.Xscale=0 #set to max zoom level
    
#    print "#################### USE MAXIMUM SPEED FOR DEBUGGING!!!!!!!!!!!"
#    data.interval=1 #Debugging
    
   

    data.run=True #change to recording
    app.timer_GetData = QtCore.QTimer()
    QtCore.QObject.connect(app.timer_GetData, QtCore.SIGNAL("timeout()"), MeasurementTimerLoop)     

    data.StartTime=time.localtime()
    data.StartTimeSec=Tuple2Seconds(data.StartTime)
   
#    if(data.debug==False): app.timer_GetData.start(data.interval)   
#    else: app.timer_GetData.start(1)    #run very fast for testing
    app.timer_GetData.start(data.interval)
    
    CreateRecordFile(data.interval)
    GetFirstEntry()
    WriteIntoRecordFile()
   
    GenerateGraphBitmap(RecordWindow)
#    RecordWindow.UpdateScreen()
    
    RecordWindow.ui.CanvasWidget.show()
    RecordWindow.UpdateScreen()
    
    

    if(data.debug==1): 
        print "Start timers (Interval:", data.interval ,"ms)..."
        print "Sampling interval:", data.interval, "ms"
        print "sample_smoothing:", data.Sample_smoothing, "x"
        print "value_smoothing:", data.Value_smoothing, "x"
        print "=> log interval:", data.interval*data.Value_smoothing, "ms"
        print "Derivation:", data.Derivation, "(substracts value", data.Derivation, "- value 0)"  
    
    
    
     #change to offline mode and silent profile if requested
    if(data.demo==False): #We are on a N900
        if(data.QuickMode==False):
            if(data.OfflineMode==True): #change to offline mode
                try:
                    cmd="dbus-send --system --dest=com.nokia.mce --print-reply --type=method_call /com/nokia/mce/request com.nokia.mce.request.get_device_mode"
                    ret=os.popen2(cmd) #get current mode
                    p=ret[1].readline() #not used
                    p=ret[1].readline()
                    tmp=string.split(p,  "\"")
                    data.PreviousMode=tmp[1]
                    if(data.PreviousMode=="normal"): #was normal, change to offline
                        cmd="dbus-send --system --dest=com.nokia.mce --type=method_call /com/nokia/mce/request com.nokia.mce.request.req_device_mode_change string:\"offline\""
                        os.popen(cmd)
                except:
                    pass
                    
            if(data.SilentProfile==True): #change to silent profile
                try:
                    cmd="dbus-send --type=method_call --print-reply --dest=com.nokia.profiled /com/nokia/profiled com.nokia.profiled.get_profile"
                    ret=os.popen2(cmd) #get current profile
                    p=ret[1].readline() #not used
                    p=ret[1].readline()
                    tmp=string.split(p,  "\"")
                    data.PreviousProfile=tmp[1]
                    if(data.PreviousProfile=="general"): #was general, change to silent
                        cmd="dbus-send --type=method_call --dest=com.nokia.profiled /com/nokia/profiled com.nokia.profiled.set_profile string:\"silent\" | echo \"\""
                        os.popen(cmd)
                        cmd="dbus-send --type=method_call --dest=org.freedesktop.Notifications /org/freedesktop/Notifications org.freedesktop.Notifications.SystemNoteInfoprint string:\"Silent Profile activated\""
                        os.popen(cmd)
                except:
                    pass   
    
    
    
    
###########################################################################
## 
###########################################################################
def StopMeasurement():
    global app
    
    RecordWindow.UpdateScreenStop()
    RecordWindow.Canvas.update()
    RecordWindow.UpdateScreen()
    
#    DeleteAlarm() #delete Alarm if existing
    
#    print "Stop Measurement",  data.run    
    if(data.run==True): #in recording
        if(data.debug==1): print "Stop Timer"
        app.timer_GetData.stop()
        StopLogfile() 
        UpdateFileHeader() #rewrite log file (only do when recording is stopped!)

    data.run=False

    #change back to normal mode and general profile if requested
    if(data.demo==False): #We are on a N900
        if(data.QuickMode==False): #normal mode
            if(data.OfflineMode==True and data.demo!=1): #change to offline mode
                try:
                    if(data.PreviousMode=="normal"): #was normal (before start measurement), so restore to normal
                        cmd="dbus-send --system --dest=com.nokia.mce --type=method_call /com/nokia/mce/request com.nokia.mce.request.req_device_mode_change string:\"normal\""
                        os.popen(cmd)
                except:
                    pass
            if(data.SilentProfile==True and data.demo!=1): #change to offline mode
                try:
                    if(data.PreviousProfile=="general"): #was general (before start measurement), so restore to general
                        cmd="dbus-send --type=method_call --dest=com.nokia.profiled /com/nokia/profiled com.nokia.profiled.set_profile string:\"general\" | echo \"\""
                        os.popen(cmd)  #set profile              
                        cmd="dbus-send --type=method_call --dest=org.freedesktop.Notifications /org/freedesktop/Notifications org.freedesktop.Notifications.SystemNoteInfoprint string:\"General Profile activated\""
                        os.popen(cmd)  #notification
                except:
                    pass






############################################################################
## Stretch graph to show more details
############################################################################
def Zoom(direction):
    if(direction==+1): #zoom in
        if(data.Xscale==0): return
        ViewWindow.ui.bZoomOut.setEnabled(True)
        data.Xscale=(data.Xscale-1) % data.arrXscaleLen
    else: # -1, zoom out
        if(data.Xscale==data.arrXscaleLen-1): return
        ViewWindow.ui.bZoomIn.setEnabled(True)
        data.Xscale=(data.Xscale+1) % data.arrXscaleLen




    
    
###########################################################################
## Raise or lower the Trigger in the graph + sleep phases
###########################################################################
def MoveTrigger(direction):
    if(direction==+1): #up
        data.Trigger=data.Trigger+1
        if(data.Trigger>data.maxTrigger): data.Trigger=data.maxTrigger      
    else: #-1, down
        data.Trigger=data.Trigger-1
        if(data.Trigger<0): data.Trigger=0
    if(data.run==True): records.Trigger=data.Trigger




def About(ui):
    QtGui.QMessageBox.about(ui, 'About SleepAnalyser', "Copyright 2010 by George Ruinelli with great support from the MAEMO community\n"+\
    "Version: "+str(data.version)+"-"+str(data.build)+"\n"+\
    "Contact: george@ruinelli.ch\nWebsite: https://garage.maemo.org/projects/sleepanalyser/\n"+\
    "\nFeedback and donations are very welcome!")




###########################################################################
##  Class for Main Window
###########################################################################
class frmStartWindow(QMainWindow):
    def __init__(self):
        QMainWindow.__init__(self)
        if(data.demo==False): self.setAttribute(Qt.WA_Maemo5StackedWindow) # This attribute makes the whole Stacked Window thing work
        self.ui = Ui_StartWindow()
        self.ui.setupUi(self)

        QtCore.QObject.connect(self.ui.bCollecting, QtCore.SIGNAL("clicked()"), self.StartMeasurementNormal)
        QtCore.QObject.connect(self.ui.bLog, QtCore.SIGNAL("clicked()"), self.ShowViewWindow)
        QtCore.QObject.connect(self.ui.bSetAlarm, QtCore.SIGNAL("clicked()"), self.SetAlarm)
        
        #Add styles to buttons
        self.ui.bCollecting.setStyleSheet(data.UIStyle)
        self.ui.bSetAlarm.setStyleSheet(data.UIStyle)
        self.ui.bLog.setStyleSheet(data.UIStyle)

        
        QtCore.QObject.connect(self.ui.actionTest_recording, QtCore.SIGNAL("triggered()"), self.StartMeasurementQuick)
        QtCore.QObject.connect(self.ui.actionSettings, QtCore.SIGNAL("triggered()"), self.ShowConfigWindow)
        QtCore.QObject.connect(self.ui.actionAbout, QtCore.SIGNAL("triggered()"), self.About)
        QtCore.QObject.connect(self.ui.actionHelp, QtCore.SIGNAL("triggered()"), self.ShowHelpWindow)

        
        if(data.app_path!="./"): #only needed if not started in app path
            self.ui.lblIcon.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"SleepAnalyser.png"))
            self.ui.bSetAlarm.setIcon(QIcon(data.app_path+"img/"+"SleepAnalyser.png"))
#            self.ui.bQuick.setIcon(QIcon(data.app_path+"img/"+"krec.png"))
            self.ui.bCollecting.setIcon(QIcon(data.app_path+"img/"+"krec.png"))
            self.ui.bLog.setIcon(QIcon(data.app_path+"img/"+"zoom.png"))
#            self.ui.bConfig.setIcon(QIcon(data.app_path+"img/"+"configure.png"))






    ###########################################################################
    ## Show the collectring window (her the data gets visualised during the logging)
    ###########################################################################
    def StartMeasurementNormal(self):
        if(data.debug==1): print "Measure in normal Mode"
        data.QuickMode=False #run in normal mode
        self.StartMeasurement()
        
        
        
        
        
    ###########################################################################
    ## Show the collectring window (her the data gets visualised during the logging)
    ###########################################################################
    def StartMeasurementQuick(self):
        if(data.debug==1): print "Measure in QuickMode"
        data.QuickMode=True #run in quick mode (for testing only)
        data.Xscale=0 #default zoom level
        self.StartMeasurement()
        
        
        
        
        
    ###########################################################################
    ## 
    ###########################################################################
    def StartMeasurement(self):
        RecordWindow.ui.lblStatus.hide()
        RecordWindow.show()
        app.processEvents() #force ui update  
        
        InitRecord() #clear all old data
        InitMeasurement()
        RecordWindow.UpdateScreenStart()
        RecordWindow.Canvas.show()
        
        
    ###########################################################################
    ## Show the log window (where the data from a file gets visualised)
    ###########################################################################
    def ShowViewWindow(self):        
        ViewWindow.ui.lblStart.setText("")
        ViewWindow.ui.lblEnd.setText("")
        ViewWindow.ui.lblNote.setText("")
        ViewWindow.Canvas.hide()
        ViewWindow.StatisticsCanvas.hide()
        ViewWindow.show()
        app.processEvents() #force ui update   
        
        #reset to default log folder
        if(data.demo==True): data.logfolder=data.devlogfolder #folder on devel PC
        else: data.logfolder=data.defaultlogfolder

        LoadFileList()
        data.arrLogFileIndex=len(data.arrLogFiles)-1 #no file selected, show latest
        
        if os.path.exists(data.logfolder+data.lastlogfile):
            data.logfilename=data.lastlogfile
        else:
            l=len(data.arrLogFiles)
            if(l>0):
                data.logfilename=data.arrLogFiles[l-1] #open last (youngest) log file
            else: #no logfiles in logfolder, show "open file" dialog
                tmp=ViewWindow.ShowFileDialog()
                if(not os.path.exists(tmp)): #only continue if valid log file selected
                    StartWindow.show()
                    ViewWindow.hide()
                    return #only continue if valid log file selected
                tmp2=tmp.split("/")
                l=len(tmp2)    
                data.logfilename=tmp2[l-1]
                data.logfolder=string.replace(tmp,  data.logfilename,  "") #get folder of selected file
                print "Set folder=",  data.logfolder,  ", file=",  data.logfilename
        
        
        ViewWindow.ShowLogFile()
        
        if(data.logfilename!=""):
            ViewWindow.show()
        else:
            print "No valid file selected"
        
        
        
        
        
        
    def ShowConfigWindow(self):
        ConfigWindow.UpdateScreen()
        ConfigWindow.show()

        
        
        
        
    def ShowHelpWindow(self):
        HelpWindow.show()
        #    cmd="dbus-send --system --type=method_call --dest=com.nokia.osso_browser /com/nokia/osso_browser/request com.nokia.osso_browser.load_url string:\"/opt/SleepAnalyser/Info.htm\""
#        cmd="dbus-send --type=method_call --dest=com.nokia.osso_browser /com/nokia/osso_browser com.nokia.osso_browser.load_url string:\"/opt/SleepAnalyser/Info.htm\""
#        ret=os.popen2(cmd) #get current mode






        
    def SetAlarm(self):
        AlarmTimeWindow.UpdateScreen()
        AlarmTimeWindow.show()




    def UpdateScreenStart(self):
        try:   
            if(data.debug==1): print "Application path:",  data.app_path
            file = open(data.app_path+"version", 'r')
            data.version = file.readline()
            data.version=data.version[:-1]
            data.build = file.readline()
            print "Application version:",  data.version,  "Build:",  data.build
        except:
            print "Version file not found"
#        self.ui.lblVersion.setText("Version: "+data.version+"-"+data.build)


    def About(self):
        About(self)
        
        

    def closeEvent(self, event):
        global config
        WriteConfig()

















###########################################################################
#  Class for Collecting Window
###########################################################################
class frmRecordWindow(QMainWindow):
    def __init__(self, parent=None):       
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor
        if(data.demo==False): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack
        
        self.ui = Ui_RecordWindow()
        self.ui.setupUi(self)
        
#        scroller = self.ui.ScrollArea.property("kineticScroller").toPyObject()
#        scroller.setEnabled(True)
        
        self.Canvas = MyGraphCanvas(self.ui.CanvasWidget)
        
        self.ui.CanvasWidget.setMinimumSize(QtCore.QSize(1000, data.GraphBitMapHeight)) #needed?
        
        
        
        
        QtCore.QObject.connect(self.ui.bStop, QtCore.SIGNAL("clicked()"), self.StopMeasurement)
        QtCore.QObject.connect(self.ui.bRaiseTrigger, QtCore.SIGNAL("clicked()"), self.RaiseTrigger)
        QtCore.QObject.connect(self.ui.bLowerTrigger, QtCore.SIGNAL("clicked()"), self.LowerTrigger)
        QtCore.QObject.connect(self.ui.bZoomIn, QtCore.SIGNAL("clicked()"), self.ZoomIn)
        QtCore.QObject.connect(self.ui.bZoomOut, QtCore.SIGNAL("clicked()"), self.ZoomOut)
        QtCore.QObject.connect(self.ui.bEditNote, QtCore.SIGNAL("clicked()"), self.EditNote)
        
        QtCore.QObject.connect(self.ui.actionAbout, QtCore.SIGNAL("triggered()"), self.About)

        #Add styles to buttons
        self.ui.bStop.setStyleSheet(data.UIStyle)
        self.ui.bRaiseTrigger.setStyleSheet(data.UIStyle)
        self.ui.bLowerTrigger.setStyleSheet(data.UIStyle)
        self.ui.bZoomIn.setStyleSheet(data.UIStyle)
        self.ui.bZoomOut.setStyleSheet(data.UIStyle)
        self.ui.bEditNote.setStyleSheet(data.UIStyle)
        
        if(data.app_path!="./"): #only needed if not started in app path
            self.ui.bZoomIn.setIcon(QIcon(data.app_path+"img/"+"zoom-in.png"))
            self.ui.bZoomOut.setIcon(QIcon(data.app_path+"img/"+"zoom-out.png"))
            self.ui.bRaiseTrigger.setIcon(QIcon(data.app_path+"img/"+"go-up.png"))
            self.ui.bLowerTrigger.setIcon(QIcon(data.app_path+"img/"+"go-down.png"))
            self.ui.bEditNote.setIcon(QIcon(data.app_path+"img/"+"kwrite.png"))
            
            
            
            

    def UpdateScreenStart(self):
#        t=strftime("%d. %b. %Y %H:%M:%S", data.StartTime)
        t=strftime("%a. %x %X", data.StartTime)
        self.ui.lblStarttime.setText("Start: "+t)
        self.ui.lblElapsedTime.setText("Elapsed time:")
        self.ui.lblIcon.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"SleepAnalyser.png"))
        self.ui.bStop.setIcon(QIcon(data.app_path+"img/"+"krec2.png"))
        self.ui.bStop.setText("Stop")

        if(records.AlarmIsSet==True): #alarm active
            m=str(data.AlarmMinute)
            if(len(m)==1): m="0"+m
#            self.ui.lblAlarmTime.setText(str(data.AlarmHour)+":"+m+" or up to "+str(data.AlarmTimeWindow)+" minutes before that")
            self.ui.lblAlarmTime.setText("Alarm time: "+strftime("%H:%M", records.AlarmTimeWindowStart_tuple)+" - "+strftime("%H:%M", records.AlarmTimeWindowEnd_tuple))
        else: #Do only recording, no alarm
            self.ui.lblAlarmTime.setText("Alarm time: (No Alarm set)")
                        
        self.ui.lblNote.setText("Note: ")
        self.UpdateScreen()
        
        
        
        
    def UpdateScreen(self):    
#        if(data.DisplayIsOn==False):
#            if(data.debug==1): print "-> x Do not update screen, Display is off!"
#            return
#        else:
#            if(data.debug==1): print "-> o Update screen"
        
        if(data.run==True): #recording
            t=time.mktime(time.localtime())
            time_difference=t-data.StartTimeSec
#            d=time.strftime("%X", time.gmtime(time_difference))
            d=time.strftime("%H:%M:%S", time.gmtime(time_difference))
            self.ui.lblElapsedTime.setText("Elapsed time: "+d + " ("+str(records.Index)+" entries)")
        else: self.UpdateScreenStop()
        
        self.ui.lblStatus2.setText("Status: "+data.Status)
            
        if(data.Xscale==0): self.ui.bZoomIn.setEnabled(False)
        else: self.ui.bZoomIn.setEnabled(True)
        if(data.Xscale==data.arrXscaleLen-1): self.ui.bZoomOut.setEnabled(False)
        else: self.ui.bZoomOut.setEnabled(True)

        if(data.Trigger==data.maxTrigger): self.ui.bRaiseTrigger.setEnabled(False)
        else: self.ui.bRaiseTrigger.setEnabled(True)
        if(data.Trigger==0): self.ui.bLowerTrigger.setEnabled(False)
        else: self.ui.bLowerTrigger.setEnabled(True)
        


        
    

    def UpdateScreenStop(self):
        self.ui.bStop.setText("Close")
        self.ui.lblStatus2.setText("Status: "+data.Status)
        
        
        
        
        
        
    def StopMeasurement(self):
            self.closeEvent(0)



    def ZoomIn(self):
        self.ui.lblStatus.setText("Zoom in...")    
        self.ui.lblStatus.show()
        app.processEvents() #force ui update   
        Zoom(+1)
        GenerateGraphBitmap(self)
        self.ui.lblStatus.hide()
        


    def ZoomOut(self):
        self.ui.lblStatus.setText("Zoom out...")    
        self.ui.lblStatus.show()
        app.processEvents() #force ui update   
        Zoom(-1)
        GenerateGraphBitmap(self)
        self.ui.lblStatus.hide()



    def RaiseTrigger(self):   
        self.ui.lblStatus.setText("Raise trigger level...")    
        self.ui.lblStatus.show()
        app.processEvents() #force ui update   
        MoveTrigger(+1)
        self.ui.bLowerTrigger.setEnabled(True)
        if(data.Trigger==data.maxTrigger): self.ui.bRaiseTrigger.setEnabled(False)
        else: self.ui.bRaiseTrigger.setEnabled(True)
         
        GenerateGraphBitmap(self)
        self.ui.lblStatus.hide()
        
        
        
    def LowerTrigger(self):     
        self.ui.lblStatus.setText("Lower trigger level...")    
        self.ui.lblStatus.show()
        app.processEvents() #force ui update   
        MoveTrigger(-1)
        self.ui.bRaiseTrigger.setEnabled(True)
        if(data.Trigger==0): self.ui.bLowerTrigger.setEnabled(False)
        else: self.ui.bLowerTrigger.setEnabled(True)
         
        GenerateGraphBitmap(self)
        self.ui.lblStatus.hide()



    def closeEvent(self, event):
        global AskForKeepWindow,  StartWindow    
#        close=True
        if(data.QuickMode==False ): #running in normal mode
            try:
                event.ignore()  #ignore closing
            except:
                pass
            AskForKeepWindow.UpdateScreen()
            AskForKeepWindow.show()
#        else: 
#            pass #close
        else: #quick mode, delete file anyway and close
            StopMeasurement()
            print "Delete logfile: "+data.logfilename
            os.remove(data.logfolder+data.logfilename)
            data.QuickMode=False
            StartWindow.show()
            self.hide()
            


    def EditNote(self):
        r=QInputDialog.getText(None,  "Add Note",  "Add a note to this record:\n(leave empty for no note)",  QLineEdit.Normal,  records.Note)
        if(r[1]==True): #OK pressed
            records.Note=unicode(r[0])
            self.ui.lblNote.setText("Note: "+records.Note)
            if(data.run==True): UpdateFileHeader() #rewrite log file (only do when recording is stopped!)



    def About(self):
        About(self)











###########################################################################
## Class for Log Window
###########################################################################
class frmViewWindow(QMainWindow):
    def __init__(self, parent=None):
#        global Canvas
        QMainWindow.__init__(self, parent)
        if(data.demo==False): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack
        
        self.ui = Ui_ViewWindow()
        self.ui.setupUi(self)
        
#        scroller = self.ui.ScrollArea.property("kineticScroller").toPyObject()
#        scroller.setEnabled(True)
        
        self.Canvas = MyGraphCanvas(self.ui.CanvasWidget)
        self.StatisticsCanvas = MyStatCanvas(self)
        
        self.ui.CanvasWidget.setMinimumSize(QtCore.QSize(1000, data.GraphBitMapHeight)) #needed?
        
        
        
        QtCore.QObject.connect(self.ui.bRaiseTrigger, QtCore.SIGNAL("clicked()"), self.RaiseTrigger)
        QtCore.QObject.connect(self.ui.bLowerTrigger, QtCore.SIGNAL("clicked()"), self.LowerTrigger)
        QtCore.QObject.connect(self.ui.bZoomIn, QtCore.SIGNAL("clicked()"), self.ZoomIn)
        QtCore.QObject.connect(self.ui.bZoomOut, QtCore.SIGNAL("clicked()"), self.ZoomOut)
        QtCore.QObject.connect(self.ui.bNext, QtCore.SIGNAL("clicked()"), self.NextLogFile)
        QtCore.QObject.connect(self.ui.bLast, QtCore.SIGNAL("clicked()"), self.LastLogFile)
        QtCore.QObject.connect(self.ui.bDelete, QtCore.SIGNAL("clicked()"), self.DeleteLogFile)
        QtCore.QObject.connect(self.ui.bOpen, QtCore.SIGNAL("clicked()"), self.OpenRecord)
        QtCore.QObject.connect(self.ui.bOpenDate, QtCore.SIGNAL("clicked()"), self.OpenDate)
        QtCore.QObject.connect(self.ui.bEditNote, QtCore.SIGNAL("clicked()"), self.EditNote)
#        QtCore.QObject.connect(self.ui.bLeft, QtCore.SIGNAL("clicked()"), self.goLeft)
#        QtCore.QObject.connect(self.ui.bRight, QtCore.SIGNAL("clicked()"), self.goRight)

        QtCore.QObject.connect(self.ui.actionAbout, QtCore.SIGNAL("triggered()"), self.About)
        QtCore.QObject.connect(self.ui.actionShowRecordData, QtCore.SIGNAL("triggered()"), self.ShowRecordData)
        QtCore.QObject.connect(self.ui.actionExport, QtCore.SIGNAL("triggered()"), self.Export)

        
        #Add styles to buttons
        self.ui.bRaiseTrigger.setStyleSheet(data.UIStyle)
        self.ui.bLowerTrigger.setStyleSheet(data.UIStyle)
        self.ui.bZoomIn.setStyleSheet(data.UIStyle)
        self.ui.bZoomOut.setStyleSheet(data.UIStyle)
        self.ui.bNext.setStyleSheet(data.UIStyle)
        self.ui.bLast.setStyleSheet(data.UIStyle)
        self.ui.bDelete.setStyleSheet(data.UIStyle)
        self.ui.bOpen.setStyleSheet(data.UIStyle)
        self.ui.bOpenDate.setStyleSheet(data.UIStyle)
        self.ui.bEditNote.setStyleSheet(data.UIStyle)
#        self.ui.bLeft.setStyleSheet(data.UIStyle)
#        self.ui.bRight.setStyleSheet(data.UIStyle)
        
        
        
        
        if(data.app_path!="./"): #only needed if not started in app path
            self.ui.bLast.setIcon(QIcon(data.app_path+"img/"+"arrow-left-double.png"))
            self.ui.bNext.setIcon(QIcon(data.app_path+"img/"+"arrow-right-double.png"))
            self.ui.bDelete.setIcon(QIcon(data.app_path+"img/"+"edittrash.png"))
            self.ui.bOpen.setIcon(QIcon(data.app_path+"img/"+"document-open-folder.png"))
            self.ui.bOpenDate.setIcon(QIcon(data.app_path+"img/"+"date.png"))
            self.ui.bZoomIn.setIcon(QIcon(data.app_path+"img/"+"zoom-in.png"))
            self.ui.bZoomOut.setIcon(QIcon(data.app_path+"img/"+"zoom-out.png"))
            self.ui.bRaiseTrigger.setIcon(QIcon(data.app_path+"img/"+"go-up.png"))
            self.ui.bLowerTrigger.setIcon(QIcon(data.app_path+"img/"+"go-down.png"))
#            self.ui.bLeft.setIcon(QIcon(data.app_path+"img/"+"go-previous.png"))
#            self.ui.bRight.setIcon(QIcon(data.app_path+"img/"+"go-next.png"))
            self.ui.bEditNote.setIcon(QIcon(data.app_path+"img/"+"kwrite.png"))





    def UpdateScreen(self):
#        shortfolder=string.replace(data.logfolder, data.homedir, "~/")
#        shortfolder=string.replace(shortfolder, "//", "/") #workaround
#        self.ui.lblLogfile.setText(data.logfilename+" (in "+shortfolder+")")
#        self.ui.lblLogfile.setText("Record: "+data.logfilename)

#        d0=strftime("%a. %d.%m.%Y  %H:%M:%S", records.StartTime_tuple)
        d0=strftime("%a. %x %X", records.StartTime_tuple)
#        EndTime_tuple=Seconds2Tuple(Tuple2Seconds(records.StartTime_tuple)+records.Index*records.Sampling_interval*records.Value_smoothing/1000)
#        if(strftime("%d.%m.%Y", records.StartTime_tuple)==strftime("%d.%m.%Y", records.EndTime_tuple)):
#            d1=strftime("%H:%M:%S", records.EndTime_tuple) #on same day, only show date once
#        else:
#            d1=strftime("%a. %d.%m.%Y  %H:%M:%S", EndTime_tuple) 
#            d1=strftime("%X", records.EndTime_tuple) #on same day, only show date once
#        else:
        d1=strftime("%a. %x %X", records.EndTime_tuple) 
#            
#        diff_sec=records.Index*records.Sampling_interval*records.Value_smoothing/1000
#        diff_sec=Tuple2Seconds(records.EndTime_tuple)-Tuple2Seconds(records.StartTime_tuple)
        diff_sec=records.EndTime_seconds-records.StartTime_seconds
#        print diff_sec,  Tuple2Seconds(records.EndTime_tuple), Tuple2Seconds(records.StartTime_tuple)
#        diff_tuple=Seconds2Tuple(diff_sec)
#        print diff_tuple,  records.StartTime_tuple,  records.EndTime_tuple
#        diff=strftime("%H:%M:%S", diff_tuple)
#        diff=time.strftime("%X", time.gmtime(diff_sec))
        diff=time.strftime("%H:%M:%S", time.gmtime(diff_sec))
        if(diff_sec>86400): d="1 day, " #longer than 24h
        else: d=""
#        self.ui.lblTimes.setText("Time: "+str(d0)+"  -  "+str(d1)+"  (Duration: "+d+str(diff)+")")
        self.ui.lblStart.setText(str(d0))
        self.ui.lblEnd.setText(str(d1)+" (Duration: "+d+str(diff)+")")

        self.ui.lblNote.setText("Note: "+records.Note)

        if(data.Xscale==0): self.ui.bZoomIn.setEnabled(False)
        else: self.ui.bZoomIn.setEnabled(True)
        if(data.Xscale==data.arrXscaleLen-1): self.ui.bZoomOut.setEnabled(False)
        else: self.ui.bZoomOut.setEnabled(True)

        if(data.Trigger==data.maxTrigger): self.ui.bRaiseTrigger.setEnabled(False)
        else: self.ui.bRaiseTrigger.setEnabled(True)
        if(data.Trigger==0): self.ui.bLowerTrigger.setEnabled(False)
        else: self.ui.bLowerTrigger.setEnabled(True)
        
        if(len(data.arrLogFiles)<2): #disable last/next buttons if only one file in folder
            self.ui.bNext.setEnabled(False)
            self.ui.bLast.setEnabled(False)
        else:
            self.ui.bNext.setEnabled(True)
            self.ui.bLast.setEnabled(True)

#        ScrollbarStartPercent,ScrollbarEndPercent= ScrollbarData()
#        ScrollbarStartPercent=round(ScrollbarStartPercent,  0)
#        ScrollbarEndPercent=round(ScrollbarEndPercent,  0)
##        print "UpdateScreen,  Scrollbar percent:",  ScrollbarStartPercent,  ScrollbarEndPercent
#        if(ScrollbarStartPercent>0): self.ui.bLeft.setEnabled(True)
#        else: self.ui.bLeft.setEnabled(False)
#        if(ScrollbarEndPercent<100): self.ui.bRight.setEnabled(True)
#        else: self.ui.bRight.setEnabled(False)
    
    
    
            
    ###########################################################################
    ## 
    ###########################################################################
    def ShowLogFile(self):
        
        self.ui.lblStart.setText("")
        self.ui.lblEnd.setText("")
        self.ui.lblNote.setText("")
        self.Canvas.hide()
        self.StatisticsCanvas.hide()
        self.ui.lblStatus.setText("Loading...")        
        self.ui.lblStatus.show()
        app.processEvents() #force ui update
        
        r=LoadRecord()    
        if(r==1): #file too long            
            QtGui.QMessageBox.warning(self, 'Record too long!', "The record is too long. The end will be cut off!", QtGui.QMessageBox.Ok)
        elif(r==2): #file does not exist
            self.OpenRecord()
            return
#        elif(r==3): #file is corrupted => draw empty graph
#            QtGui.QMessageBox.warning(self, 'File corrupded!', "The data in the record file\n"+data.logfilename+"\nis invalid!\nNo data loaded.", QtGui.QMessageBox.Ok)
#        elif(r==8): #file header is corrupted
#            QtGui.QMessageBox.warning(self, 'File header corrupted!', "The header data in the record file\n"+data.logfilename+"\nis invalid!\nNo data loaded.", QtGui.QMessageBox.Ok)
#        elif(r==4): #file is empty
#            QtGui.QMessageBox.warning(self, 'File empty!', "The data in the record file\n"+data.logfilename+"\ncontains no data!", QtGui.QMessageBox.Ok)
#        elif(r==9): #Record too old
#            QtGui.QMessageBox.warning(self, 'File version too old!', "Can not load record \n"+data.logfilename+"\nas it got created from SleepAnalyser version <1.6 and is not compatible with the current SleepAnalyser version.", QtGui.QMessageBox.Ok)
#        LoadFileList()
        ProcessData(True)
        GenerateStatisticData()
        data.Xscale=1 #default zoom level
#        data.Xscale=0
#        data.GraphStart=max(records.LastIndex-data.arrXscale[data.Xscale]*775, 0)
        
#        data.ChangesInGraph=True #Make sure it only gets refreshed once


        self.ui.lblStatus.hide()
        self.Canvas.show()
        self.StatisticsCanvas.show()
        GenerateGraphBitmap(self)
        GenerateStatisticsBitmap(self)
        
      
        
        

        
    ###########################################################################
    ## 
    ###########################################################################
    def NextLogFile(self):
        data.arrLogFileIndex=(data.arrLogFileIndex+1)  % len(data.arrLogFiles)
        data.logfilename=data.arrLogFiles[data.arrLogFileIndex]
        self.ShowLogFile()

    ###########################################################################
    ## 
    ###########################################################################
    def LastLogFile(self):
        data.arrLogFileIndex=(data.arrLogFileIndex-1)  % len(data.arrLogFiles)
        data.logfilename=data.arrLogFiles[data.arrLogFileIndex]
        self.ShowLogFile()


    ###########################################################################
    ## 
    ###########################################################################
    def ZoomIn(self):        
        self.ui.lblStatus.setText("Zoom in...")    
        self.ui.lblStatus.show()
        app.processEvents() #force ui update        
        Zoom(+1)
        self.ui.bZoomOut.setEnabled(True)
        if(data.Xscale==0): self.ui.bZoomIn.setEnabled(False)
        else: self.ui.bZoomIn.setEnabled(True)
        
#        print "records.LastIndex:",  records.LastIndex, data.arrXscale[data.Xscale]*775
#        data.GraphStart=max(records.LastIndex-data.arrXscale[data.Xscale]*775, 0)
#        print "GraphStart:",  data.GraphStart
        
#        data.ChangesInGraph=True #Make sure it only gets refreshed once
                
        GenerateGraphBitmap(self)
        
        self.ui.lblStatus.hide()
        
        
        
        
        
        
        

    ###########################################################################
    ## 
    ###########################################################################
    def ZoomOut(self):
        self.ui.lblStatus.setText("Zoom out...")    
        self.ui.lblStatus.show()
        app.processEvents() #force ui update
        Zoom(-1)
        self.ui.bZoomIn.setEnabled(True)
        if(data.Xscale==data.arrXscaleLen-1): self.ui.bZoomOut.setEnabled(False)
        else: self.ui.bZoomOut.setEnabled(True)
        
#        data.GraphStart=max(records.LastIndex-data.arrXscale[data.Xscale]*775, 0)
        
#        data.ChangesInGraph=True #Make sure it only gets refreshed once
        
        GenerateGraphBitmap(self)
        
        self.ui.lblStatus.hide()

        
    ###########################################################################
    ## 
    ###########################################################################
    def RaiseTrigger(self):   
        self.ui.lblStatus.setText("Raise trigger level...")    
        self.ui.lblStatus.show()
        MoveTrigger(+1)
        app.processEvents() #force ui update
        
        GenerateGraphBitmap(self)
        
        GenerateStatisticData()
        GenerateStatisticsBitmap(self)
#        self.update() #update statistic bars
        self.ui.lblStatus.hide()


    ###########################################################################
    ## 
    ###########################################################################
    def LowerTrigger(self):    
        self.ui.lblStatus.setText("Lower trigger level...")    
        self.ui.lblStatus.show() 
        MoveTrigger(-1)
        app.processEvents() #force ui update
        
        GenerateGraphBitmap(self)
        
        GenerateStatisticData()
        GenerateStatisticsBitmap(self)
#        self.update() #update statistic bars
        self.ui.lblStatus.hide()
    
        
#    ###########################################################################
#    ## 
#    ###########################################################################   
#    def goLeft(self):
#        m=600 #data.arrXscale[data.Xscale]*60
#        data.GraphStart=max(data.GraphStart-m, 0)
#        
#        
#
#        self.ui.ScrollArea.ensureVisible(10000, 10000, 0 , 0)
#        
#        
#        data.ChangesInGraph=True #Make sure it only gets refreshed once
#        self.update()
#        self.UpdateScreen()
#        
#    ###########################################################################
#    ## 
#    ###########################################################################
#    def goRight(self):
#        m=600   #data.arrXscale[data.Xscale]*60
##        print "records.LastIndex:",  records.LastIndex, data.arrXscale[data.Xscale]*775
#        MaxGraphStart=max(records.LastIndex-data.arrXscale[data.Xscale]*775, 0)
#        data.GraphStart=min(MaxGraphStart,  data.GraphStart+m)
##        print "goRight MaxGraphStart:",  MaxGraphStart
##        print "GraphStart:",  data.GraphStart
#
#
#        self.ui.ScrollArea.ensureVisible(10000, 10000, 0 , 0)
#
#        data.ChangesInGraph=True #Make sure it only gets refreshed once
#        self.update()
#        self.UpdateScreen()
        
        
    ###########################################################################
    ## 
    ###########################################################################
    def DeleteLogFile(self):
        self.ui.lblStatus.setText("Delete file...")        
        self.ui.lblStatus.show()
        reply = QtGui.QMessageBox.question(self, 'Delete Record?', "Are you sure you want to delete this record:\n"+data.logfilename, QtGui.QMessageBox.Yes, QtGui.QMessageBox.No)
        if reply == QtGui.QMessageBox.Yes:
            print "Delete logfile: "+data.logfilename
            os.remove(data.logfolder+data.logfilename)    

            index=data.arrLogFileIndex
            LoadFileList()            
            if(len(data.arrLogFiles)>0):
#                data.logfilename=data.arrLogFiles[l-1] #open last (youngest) log file
                data.arrLogFileIndex=index-1
                data.logfilename=data.arrLogFiles[data.arrLogFileIndex]
            else: #no logfiles in logfolder, show "open file" dialog
                tmp=self.ShowFileDialog()
                if(tmp==""): #only continue if valid log file selected
                    self.closeEvent("") #no file selected, close
                    return
                tmp2=tmp.split("/")
                l=len(tmp2)    
                data.logfilename=tmp2[l-1]
                data.logfolder=string.replace(tmp,  data.logfilename,  "") #get folder of selected file
                print "Set folder=",  data.logfolder,  ", file=",  data.logfilename
            self.ShowLogFile()
            self.UpdateScreen()
        self.ui.lblStatus.hide()
        self.StatisticsCanvas.show()  
            


    ###########################################################################
    ## 
    ###########################################################################
    def OpenRecord(self):
         
        self.ui.lblStart.setText("")
        self.ui.lblEnd.setText("")
        self.ui.lblNote.setText("")
        self.Canvas.hide()
        self.StatisticsCanvas.hide()
        self.ui.lblStatus.setText("Opening file dialog...")        
        self.ui.lblStatus.show()
        
        
        tmp=self.ShowFileDialog()
        if(not os.path.exists(tmp)): #only continue if valid log file selected
            return 
        tmp2=tmp.split("/")
        l=len(tmp2)    
        data.logfilename=tmp2[l-1]
        data.logfolder=string.replace(tmp,  data.logfilename,  "") #get folder of selected file
        print "Set folder=",  data.logfolder,  ", file=",  data.logfilename
        LoadFileList()
        self.ShowLogFile()
        self.ui.lblStatus.hide()
        self.StatisticsCanvas.show()  
    
    
    ###########################################################################
    ## 
    ###########################################################################
    def ShowFileDialog(self):        
        data.logfolder=string.replace(data.logfolder, "~",  data.homedir) #workaround
        if(data.debug==1): print "Open file dialog in:",  data.logfolder
        tmp= QFileDialog.getOpenFileName(None,"Open SleepAnalyser Log File", data.logfolder, "Log Files (*.csv)")
        print "Selected file:",  tmp
        return tmp
        
        
    ###########################################################################
    ## 
    ###########################################################################
    def OpenDate(self):
        DateWindow.show()
        DateWindow.DateChanged()


    ###########################################################################
    ## 
    ###########################################################################
    def closeEvent(self, event):
        ViewWindow.close()
        StartWindow.show()
        

    ###########################################################################
    ## 
    ###########################################################################
    def EditNote(self):
        r=QInputDialog.getText(None,  "Edit Note",  "Add a note to this record:\n(leave empty for no note)",  QLineEdit.Normal,  records.Note)
        if(r[1]==True): #OK pressed
            tmp=r[0]
            tmp=unicode(tmp)
            records.Note=tmp
            self.ui.lblNote.setText("Note: "+records.Note)
            UpdateFileHeader() #rewrite log file (only do when recording is stopped!)



    ###########################################################################
    ## 
    ###########################################################################
    def About(self):
        About(self)


    ###########################################################################
    ## 
    ###########################################################################
    def ShowRecordData(self):
        AdditionalInfoWindow.UpdateScreen()
        AdditionalInfoWindow.show()





    def Export(self):
        filename=string.replace(data.logfilename, ".csv",  "")
        tmp=QFileDialog.getSaveFileName(None,"Export graph as", data.defaultlogfolder+filename, "PNG-Image (*.png)") #;;JPEG-Image (*.jpg)
        if(tmp!=""):
            print "Export graph as :",  tmp
            r=data.GraphBitMap.save(tmp,  "png",  -1)
            if(r==True): print "Export was successful"
            else: QtGui.QMessageBox.warning(self, 'Could not export file', "Failed to save graph as file\n"+tmp, QtGui.QMessageBox.Ok)









###########################################################################
## Class for Help Window
###########################################################################
class frmHelpWindow(QMainWindow):
    def __init__(self, parent):
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor
        if(data.demo==False): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack
        
        self.ui = Ui_HelpWindow()
        self.ui.setupUi(self)
      
        file = open(data.app_path+"Info.htm", 'r')
        txt=""
        for line in file: txt=txt+line+"\n"
        file.close()
        self.ui.textBrowser.setHtml(txt)
        





###########################################################################
## Class for Configuration Window
###########################################################################
class frmConfigWindow(QMainWindow):
    def __init__(self, parent):
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor
        if(data.demo==False): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack
        
        self.ui = Ui_ConfigWindow()
        self.ui.setupUi(self)
        self.ui.lblIcon.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"SleepAnalyser.png"))
        
        QtCore.QObject.connect(self.ui.bEditUserName, QtCore.SIGNAL("clicked()"), self.EditUserName)
        QtCore.QObject.connect(self.ui.SlrDelayStart,SIGNAL("valueChanged(int)"),self.DelaySliderMoved)
        QtCore.QObject.connect(self.ui.SlrYscale,SIGNAL("valueChanged(int)"),self.YscaleSliderMoved)

        #Add styles to buttons
        self.ui.bEditUserName.setStyleSheet(data.UIStyle)        
        self.ui.SlrDelayStart.setStyleSheet(data.UIStyle)        
        self.ui.SlrYscale.setStyleSheet(data.UIStyle)        
        
        #Add styles to tick boxes
        self.ui.cOfflineMode.setStyleSheet(data.UIStyle)    
        self.ui.cStopAfterAlarm.setStyleSheet(data.UIStyle)   
        self.ui.cSilentProfile.setStyleSheet(data.UIStyle)   
        
        

    ###########################################################################
    ## 
    ###########################################################################
    def UpdateScreen(self):
        if(data.OfflineMode==True): self.ui.cOfflineMode.setChecked(True)
        else: self.ui.cOfflineMode.setChecked(False)    
        if(data.SilentProfile==True): self.ui.cSilentProfile.setChecked(True)
        else: self.ui.cSilentProfile.setChecked(False)      
        if(data.StopAfterAlarm==True): self.ui.cStopAfterAlarm.setChecked(True)
        else: self.ui.cStopAfterAlarm.setChecked(False)        
        self.ui.lblUser.setText("User name: "+data.user)
        self.ui.lblDelayStart.setText("Start delay: "+str(data.DelayStartTime)+" sec")
        self.ui.SlrDelayStart.setValue(int(data.DelayStartTime))
        self.ui.lblYscale.setText("Y scaling: "+str(data.Yscale))
        self.ui.SlrYscale.setValue(int(data.Yscale))


    ###########################################################################
    ## 
    ###########################################################################
    def closeEvent(self, event):        
        if(self.ui.cOfflineMode.isChecked()==True):          
                data.OfflineMode=True
        else: 
            data.OfflineMode=False
        config.set('main', 'OfflineMode', str(data.OfflineMode))   
        
        if(self.ui.cSilentProfile.isChecked()==True):          
                data.SilentProfile=True
        else: 
            data.SilentProfile=False
        config.set('main', 'SilentProfile', str(data.SilentProfile))   
        
        if(self.ui.cStopAfterAlarm.isChecked()==True):          
                data.StopAfterAlarm=True
        else: 
            data.StopAfterAlarm=False
        config.set('main', 'StopAfterAlarm', str(data.StopAfterAlarm))   
        
        config.set('main', 'user', str(data.user.encode( "utf-8" )))      
        config.set('main', 'DelayStartTime', str(data.DelayStartTime))      
        config.set('main', 'Yscale', str(data.Yscale))      
        
        ConfigWindow.close()
        StartWindow.show()


    ###########################################################################
    ## 
    ###########################################################################
    def EditUserName(self):
        EditUserName()
        self.ui.lblUser.setText("User name: "+data.user)
        

    ###########################################################################
    ## 
    ###########################################################################
    def DelaySliderMoved(self):
        x=self.ui.SlrDelayStart.value()
        data.DelayStartTime=x
        self.ui.lblDelayStart.setText("Start delay: "+str(data.DelayStartTime)+" sec")


    ###########################################################################
    ## 
    ###########################################################################
    def YscaleSliderMoved(self):
        x=self.ui.SlrYscale.value()
        data.Yscale=int(x)
        self.ui.lblYscale.setText("Y scaling: "+str(data.Yscale))






###########################################################################
## Class for Date Window
###########################################################################
class frmDateWindow(QMainWindow):
    def __init__(self, parent=None):
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor
        if(data.demo==False): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack
        
        self.ui = Ui_DateWindow()
        self.ui.setupUi(self)

        if(data.app_path!="./"): #only needed if not started in app path
            self.ui.bOpen.setIcon(QIcon(data.app_path+"img/"+"zoom.png"))

        QtCore.QObject.connect(self.ui.cal, QtCore.SIGNAL('selectionChanged()'), self.DateChanged)
        QtCore.QObject.connect(self.ui.bOpen, QtCore.SIGNAL("clicked()"), self.OpenRecord)
        
        self.ui.cal.showToday()




    ###########################################################################
    ## 
    ###########################################################################
    def DateChanged(self):
        date = self.ui.cal.selectedDate()
        date=str(date.toPyDate())

#        print "Load list of all csv files in folder:",  data.logfolder + " with date " + date
        output = os.popen("cd "+str(data.logfolder)+"; ls -C1 *"+date+"*.csv").read() # run ls and make the output look like a file I can read
        arr=output.split("\n")

        amount=len(arr)-1
        
        self.ui.lblResult.setText("Found "+str(amount)+" Records for this date.")
        if(amount>0): 
            self.ui.bOpen.setEnabled(True)
            data.LogFileSelectedDate=arr[0] #select first file of that date for opening
            
            for i in range(0,  len(data.arrLogFiles)-1):
                if(data.arrLogFiles[i]==data.LogFileSelectedDate):
#                    print "found:",  i
                    data.arrLogFileIndex=i
                    break
            
        else: self.ui.bOpen.setEnabled(False)


    ###########################################################################
    ## 
    ###########################################################################
    def OpenRecord(self):
        data.logfilename=data.LogFileSelectedDate
        print "Set folder=",  data.logfolder,  ", file=",  data.logfilename
        DateWindow.hide()
        ViewWindow.ShowLogFile()








###########################################################################
## Class for Date Window
###########################################################################
class frmAdditionalInfoWindow(QMainWindow):
    def __init__(self, parent=None):
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor
        if(data.demo==False): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack
        
        self.ui = Ui_AdditionalInfoWindow()
        self.ui.setupUi(self)


    ###########################################################################
    ## 
    ###########################################################################
    def UpdateScreen(self):
        shortfolder=string.replace(data.logfolder, data.homedir, "~/")
        shortfolder=string.replace(shortfolder, "//", "/") #workaround
        
        d0=strftime("%a. %d.%m.%Y  %H:%M:%S", records.StartTime_tuple)
#        EndTime_seconds=Tuple2Seconds(records.StartTime_tuple)+records.Index*records.Sampling_interval*records.Value_smoothing/1000
#        EndTime_tuple=Seconds2Tuple(records.EndTime_seconds)
        if(strftime("%d.%m.%Y", records.StartTime_tuple)==strftime("%d.%m.%Y", records.EndTime_tuple)):
            d1=strftime("%H:%M:%S", records.EndTime_tuple) #on same day, only show date once
        else:
            d1=strftime("%a. %d.%m.%Y  %H:%M:%S", records.EndTime_tuple) 
        
        diff_sec=records.Index*records.Sampling_interval*records.Value_smoothing/1000
#        diff=time.strftime("%X", time.gmtime(diff_sec))
        diff=time.strftime("%H:%M:%S", time.gmtime(diff_sec))
        
        if(diff_sec>86400): diff="1 day, "+diff #longer than 24h
        
        if(records.AlarmIsSet==True):
            a0=strftime("%a. %d.%m.%Y  %H:%M:%S", records.AlarmTimeWindowStart_tuple)
            if(strftime("%d.%m.%Y", records.AlarmTimeWindowStart_tuple)==strftime("%d.%m.%Y", records.AlarmTimeWindowEnd_tuple)):
                a1=strftime("%H:%M:%S", records.AlarmTimeWindowEnd_tuple) #on same day, only show date once
            else:
                a1=strftime("%a. %d.%m.%Y  %H:%M:%S", records.AlarmTimeWindowEnd_tuple) 
            alarm="<b>Alarm Time Window:</b> "+a0+" - "+a1 + "<br>"
            if(records.AlarmTime_seconds!=-1): #alarm was set
                if(records.EndTime_seconds>=records.AlarmTime_seconds): #alarm got set off during recording
                    alarm=alarm+"<b>Alarm:</b> Alarm went off on: "+strftime("%a. %d.%m.%Y  %H:%M:%S", records.AlarmTime_tuple)+"<br>"
                    if(records.AlarmTimeWindowEnd_seconds<=records.AlarmTime_seconds):
                        alarm=alarm+"<b>Alarm reason:</b> End of alarm time window<br>"
                    else:
                        alarm=alarm+"<b>Alarm reason:</b> Movements over trigger level<br>"
                else:
                    alarm=alarm+"<b>Alarm:</b> Alarm never went off<br>"
            else:
                alarm=alarm+"<b>Alarm:</b> Alarm never went off<br>"
        else:
            alarm="<b>Alarm:</b> No alarm was set<br>"
        
        txt= "<b>File:</b> "+str(shortfolder)+str(data.logfilename+"<br>") + \
        "<b>User:</b> "+records.User +"<br>" + \
        "<b>Time:</b> "+d0+" - "+d1+"<br>" + \
        "<b>Duration:</b> "+ diff+"<br>" + \
        "<b>Record entries:</b> "+str(records.Index)+" (1 per "+str(int(records.Sampling_interval*records.Value_smoothing/1000))+" seconds)<br>" +\
        "<b>Record version:</b> "+str(records.SWversion) + "<br>" + \
        "<b>Note:</b> "+records.Note+"<br>" +\
        alarm +\
        "<b>Trigger level:</b> "+str(records.Trigger) + "<br>" + \
        "<b>Condition of record:</b> "+str(records.status)+"<br><br>"
        
            
#        t_total=(records.EndTime_seconds-records.StartTime_seconds)
#        print records.EndTime_seconds,  records.StartTime_seconds,  t_total
#        if(t_total==0):  t_total=1 #division by zerro prevention
#        pa=records.arrStatData[0][0]*100/t_total
#        pb=records.arrStatData[1][0]*100/t_total
#        pc=records.arrStatData[2][0]*100/t_total
##        pd=records.arrStatData[3][0]*100/t_total
#        pl=100-pa-pb-pc
         
        txt=txt+"<b>Statistics:</b><table>\n" + \
                "<tr><td><b>Sleep phase: </b></td>   <td><b>Total time: </b></td>                                     <td><b>Longest time:</b></td>                                    <td><b>Count: </b></td>                     <td colspan=2><b>Time percent:</b></td>\n" + \
                "<tr><td><b>Awake:</b></td>         <td>"+strftime("%X", time.gmtime(records.arrStatData[3][0]))+"</td>    <td>-</td>    <td>-</td>    <td>"+str(records.arrStatData[3][3])+"%</td><td><img src=\""+data.app_path+"img/"+"State4.png"+"\" width="+str(records.arrStatData[3][3]*2)+" height=25</td></tr>\n" + \
                "<tr><td><b>Light:</b></td>         <td>"+strftime("%X", time.gmtime(records.arrStatData[2][0]))+"</td>    <td>"+strftime("%X", time.gmtime(records.arrStatData[2][2]))+"</td>    <td>"+str(records.arrStatData[2][1])+"</td>    <td>"+str(records.arrStatData[2][3])+"%</td><td><img src=\""+data.app_path+"img/"+"State3.png"+"\" width="+str(records.arrStatData[2][3]*2)+" height=25</td></tr>\n" + \
                "<tr><td><b>Middle:</b></td>        <td>"+strftime("%X", time.gmtime(records.arrStatData[1][0]))+"</td>    <td>"+strftime("%X", time.gmtime(records.arrStatData[1][2]))+"</td>    <td>"+str(records.arrStatData[1][1])+"</td>    <td>"+str(records.arrStatData[1][3])+"%</td><td><img src=\""+data.app_path+"img/"+"State2.png"+"\" width="+str(records.arrStatData[1][3]*2)+" height=25</td></tr>\n" + \
                "<tr><td><b>Deep:</b></td>          <td>"+strftime("%X", time.gmtime(records.arrStatData[0][0]))+"</td>    <td>"+strftime("%X", time.gmtime(records.arrStatData[0][2]))+"</td>    <td>"+str(records.arrStatData[0][1])+"</td>    <td>"+str(records.arrStatData[0][3])+"%</td><td><img src=\""+data.app_path+"img/"+"State1.png"+"\" width="+str(records.arrStatData[0][3]*2)+" height=25</td></tr>\n" + \
              "</table><br><br>\n"
        
        txt=txt+"<b>Sampling interval:</b> "+str(records.Sampling_interval) + " ms<br>" + \
        "<b>Sampling smoothing:</b> "+str(records.Sample_smoothing) + "<br>" + \
        "<b>Value smoothing:</b> "+str(records.Value_smoothing)

        self.ui.textBrowser.setHtml("<html><body>"+txt+"</html></body>")


    ###########################################################################
    ## 
    ###########################################################################
    def Close(self):        
        self.close()








###########################################################################
## Class for Setting the alarm time Window
###########################################################################
class frmAlarmTimeWindow(QMainWindow):
    def __init__(self, parent=None):        
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor
        
        if(data.demo==False): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack
#        self.setWindowFlags(Qt.SplashScreen)   
        self.ui = Ui_AlarmTimeWindow()
        self.ui.setupUi(self)
        self.ui.lblIcon.setPixmap(QtGui.QPixmap(data.app_path+"img/"+"SleepAnalyser.png"))

        QtCore.QObject.connect(self.ui.bSetAlarm, QtCore.SIGNAL("clicked()"), self.SetAlarm)
        
        #Add styles to buttons
        self.ui.bSetAlarm.setStyleSheet(data.UIStyle)
        
                
        if(data.app_path!="./"): #only needed if not started in app path
            self.ui.bSetAlarm.setIcon(QIcon(data.app_path+"img/"+"SleepAnalyser.png"))

    

    ###########################################################################
    ## 
    ###########################################################################
    def UpdateScreen(self):
        if(data.AlarmHour==-1 or data.AlarmMinute==-1):
            data.AlarmHour=7
            data.AlarmMinute=0
        if(data.debug==1): print "Alarm time:",  data.AlarmHour, ":",  data.AlarmMinute
        self.ui.lstAlarmHour.setCurrentRow(int(data.AlarmHour))
        self.ui.lstAlarmMinute.setCurrentRow(int(data.AlarmMinute)/5)
        self.ui.lstAlarmTimeWindow.setCurrentRow(int(data.AlarmTimeWindow)/5-1)
        s="The Alarm will go off sometimes during the set time window (as soon as your movement is over the trigger level), but latest at the set alarm time."
        self.ui.lblNote.setText(s)



    ###########################################################################
    ## 
    ###########################################################################
    def SetAlarm(self):
        
        InitRecord() #clear all old data
        
#        print "Set Alarm"
        data.QuickMode=False #run in normal mode        
        
#        RecordWindowCanvas = RecordWindowCanvas(RecordWindow)
                
        try:
            data.AlarmHour=int(self.ui.lstAlarmHour.currentItem().text())
            data.AlarmMinute=int(self.ui.lstAlarmMinute.currentItem().text())
            data.AlarmTimeWindow=int(self.ui.lstAlarmTimeWindow.currentItem().text())
            config.set('main', 'AlarmHour', str(data.AlarmHour))  
            config.set('main', 'AlarmMinute', str(data.AlarmMinute))  
            config.set('main', 'AlarmTimeWindow', str(data.AlarmTimeWindow))  
            
            now=int(strftime("%H", time.localtime()))*3600+int(strftime("%M", time.localtime()))*60
            al=int(data.AlarmHour)*3600+int(data.AlarmMinute)*60
            alarmdiff=al-now
            if(alarmdiff<0):  #set to tomorrow if time already is elapsed today
                alarmdiff=alarmdiff+86400

            records.AlarmTime_seconds=int((records.StartTime_seconds+alarmdiff)/60)*60 #set seconds to zero
            records.AlarmTime_tuple=Seconds2Tuple(records.AlarmTime_seconds)           
            
            records.AlarmTimeWindowEnd_tuple=records.AlarmTime_tuple
            records.AlarmTimeWindowEnd_seconds=records.AlarmTime_seconds
            
            records.AlarmTimeWindow_minutes=int(self.ui.lstAlarmTimeWindow.currentItem().text())
            
            records.AlarmTimeWindowStart_seconds=records.AlarmTimeWindowEnd_seconds-records.AlarmTimeWindow_minutes*60
            records.AlarmTimeWindowStart_tuple=Seconds2Tuple(records.AlarmTimeWindowStart_seconds)            
            
#            print records.AlarmTime_tuple,  records.AlarmTimeWindow_minutes
#            print records.AlarmTimeWindowStart_tuple
#            print records.AlarmTimeWindowEnd_tuple
        
        except: #If nothing selected (workaround) 
            print "Selected time not valid!"           
#            print sys.exc_info()  
            StartWindow.show()
            AlarmTimeWindow.hide()
            return

        DeleteAlarm() #delete old alarm if existing
        records.AlarmIsSet=True
        SetAlarm() #set alarm for the last possible time
        InitMeasurement()
        
        RecordWindow.show()
        AlarmTimeWindow.hide()
        RecordWindow.UpdateScreenStart()
#        RecordWindowCanvas.show()








###########################################################################
## Class for Setting the alarm time Window
###########################################################################
class frmAskForKeepWindow(QMainWindow):
    def __init__(self, parent=None):        
        QMainWindow.__init__(self, parent) # Notice that you must give a parent window as parameter to the constuctor
        
        if(data.demo==False): self.setAttribute(Qt.WA_Maemo5StackedWindow) # Also set the Stacked Window parameter for every subwindow in the stack
        
        self.ui = Ui_AskForKeepWindow()
        self.ui.setupUi(self)

        QtCore.QObject.connect(self.ui.bstop_keepRecord, QtCore.SIGNAL("clicked()"), self.KeepRecord)
        QtCore.QObject.connect(self.ui.bstop_removeRecord, QtCore.SIGNAL("clicked()"), self.RemoveRecord)
        QtCore.QObject.connect(self.ui.bcontinue, QtCore.SIGNAL("clicked()"), self.ContinueRecording)

        #Add styles to buttons
        self.ui.bstop_keepRecord.setStyleSheet(data.UIStyle)
        self.ui.bstop_removeRecord.setStyleSheet(data.UIStyle)
        self.ui.bcontinue.setStyleSheet(data.UIStyle)
        
        if(data.app_path!="./"): #only needed if not started in app path
            self.ui.bstop_keepRecord.setIcon(QIcon(data.app_path+"img/"+"zoom.png"))
            self.ui.bstop_removeRecord.setIcon(QIcon(data.app_path+"img/"+"edittrash.png"))
            self.ui.bcontinue.setIcon(QIcon(data.app_path+"img/"+"krec.png"))


    ###########################################################################
    ## 
    ###########################################################################
    def UpdateScreen(self):
#        s="Do you really want to stop recording?"
#        if(records.AlarmIsSet==True): s=s+"\nThis will also remove the set alarm!"
#        s=s+"\n\nAnd do you want to keep the record?"
#        self.ui.lblNote.setText(s)
        pass
        

    ###########################################################################
    ## 
    ###########################################################################
    def KeepRecord(self):
        print "Stop recording and keep record"
        global ViewWindow
        StopMeasurement()
        DeleteAlarm() #delete Alarm if existing
        records.AlarmIsSet=False
        data.lastlogfile=data.logfilename #set to corrent file for viewwindow
        config.set('main', 'lastlogfile', str(data.lastlogfile))              
        RecordWindow.hide()
        print "Show record in ViewWindow:",  data.lastlogfile
        StartWindow.ShowViewWindow()
        self.hide()



    ###########################################################################
    ## 
    ###########################################################################
    def RemoveRecord(self): #removed record
        print "Stop recording and remove record"
        global StartWindow,  RecordWindow
        StopMeasurement()
        DeleteAlarm() #delete Alarm if existing
        records.AlarmIsSet=False
        print "Delete logfile: "+data.logfilename
        os.remove(data.logfolder+data.logfilename)
        RecordWindow.hide()
        StartWindow.show()
        self.hide()



    ###########################################################################
    ## 
    ###########################################################################
    def ContinueRecording(self):        
        print "continue recording"
        self.hide()
        return


    ###########################################################################
    ## 
    ###########################################################################
    def closeEvent(self, event):        
        event.ignore() #do not close
        self.KeepRecord() #interprete closing as "stop and keep file"

















###########################################################################
## 
###########################################################################
class DisplayStatusDaemon:
    def __init__(self):
        if(data.demo==False): #We are on a N900
            loop = DBusQtMainLoop(set_as_default=True)
            system_bus = dbus.SystemBus()
            system_bus.add_signal_receiver(self.DisplayStatus,
                                           path='/com/nokia/mce/signal',
                                        signal_name='display_status_ind',
                                        dbus_interface='com.nokia.mce.signal')
                                        
            if(data.debug==1): print "DisplayStatusDaemon started"




    ###########################################################################
    ## 
    ###########################################################################
    def DisplayStatus(self, *params):
        global RecordWindow
#        print "1", len(params)
#        print params
        tmp=str(params)
        tmp=string.split(tmp, "'")
#        print tmp
        if(tmp[1]=="off"): data.DisplayIsOn=False
        elif(tmp[1]=="on"): 
            data.DisplayIsOn=True
            if(data.run==True): 
                if(data.debug==1): print "Update screen & graph"
#                data.ChangesInGraph=True

                GenerateStatisticData()   
                GenerateGraphBitmap(RecordWindow)    
                RecordWindow.Canvas.update()
                RecordWindow.UpdateScreen()








###########################################################################
## 
###########################################################################
class MyGraphCanvas(QtGui.QWidget):
    def __init__(self, parent):
        QMainWindow.__init__(self, parent)    
        self.setGeometry(0, 0, 770, data.GraphBitMapHeight)
        
#        print "Create canvas with:",  770,  data.Ysize,  "(Size will change depending on graph len)"


    ###########################################################################
    ## 
    ###########################################################################
    def paintEvent(self, event):
        if(data.DisplayIsOn==False):
            if(data.debug==1): print "-> o Do not paint event (Graph), Display is off!"
            return
        else:
            if(data.debug==1): print "-> x paint event (Graph)"

        
        if(data.ScrollToEnd==True): 
            if(data.run==True): RecordWindow.ui.ScrollArea.ensureVisible(data.GraphBitMapLen+100, 0, 0 , 0)
            else: ViewWindow.ui.ScrollArea.ensureVisible(data.GraphBitMapLen+100, 0, 0 , 0)
            data.ScrollToEnd=False

#        DrawGraph(self,  QtGui)

        paint = QtGui.QPainter()
        paint.begin(self)        
        paint.drawPixmap(0, 0, data.GraphBitMap) #load graph from Bitmap
        paint.end()

    
    
    
    
    
    
###########################################################################
## 
###########################################################################
class MyStatCanvas(QtGui.QWidget):
    def __init__(self, parent):
        QMainWindow.__init__(self, parent)    
        self.setGeometry(402, 280, data.StatisticsBitMapLen,  data.StatisticsBitMapHeight)
        

    ###########################################################################
    ## 
    ###########################################################################
    def paintEvent(self, event):
        if(data.DisplayIsOn==False):
            if(data.debug==1): print "-> o Do not paint event (Statistics), Display is off!"
            return
        else:
            if(data.debug==1): print "-> x paint event (Statistics)"

        paint = QtGui.QPainter()
        paint.begin(self)        
        paint.drawPixmap(0, 0, data.StatisticsBitMap) #load statistics from Bitmap
        paint.end()

    
    
    
    
    
    
    
    
    
###########################################################################
## 
###########################################################################        
def GenerateGraphBitmap(self):    
#    if(data.DisplayIsOn==False):
#        if(data.debug==1): print "-> o Do not Generate Graph Bitmap, Display is off!"
#        return
#    else:
    if(data.debug==1): print "Generate Graph Bitmap"

    Xscale=data.arrXscale[data.Xscale]
#        data.GraphBitMapLen=max((records.LastIndex+5)/Xscale + 2 ,  770)
    data.GraphBitMapLen=max((records.LastIndex+5)/Xscale + 2 ,  735)
    data.GraphBitMapHeight=data.Ysize+34
    
#    print "GraphBitMapLen:",  data.GraphBitMapLen,  ", GraphBitMapHeight:",  data.GraphBitMapHeight ,  ", Ysize is:", data.Ysize  
    try:
        data.GraphBitMap = QtGui.QPixmap(data.GraphBitMapLen, data.GraphBitMapHeight) #create bitmap for graph    
        DrawGraph(data.GraphBitMap,  QtGui) # draw graph
    except:
        print "Error while creating the graph bitmap"
    
#    ViewWindowCanvas.setGeometry(0, 0, data.GraphBitMapLen, data.GraphBitMapHeight)  #set size of canvas to graph 
    self.Canvas.setGeometry(0, 0, data.GraphBitMapLen, data.GraphBitMapHeight)  #set size of canvas to graph 
#    print "Set size of canvas to :",  data.GraphBitMapLen, "x",  data.GraphBitMapHeight
    
    self.ui.CanvasWidget.setMinimumSize(data.GraphBitMapLen, data.GraphBitMapHeight)  #set size of canvas widget to graph

    data.ScrollToEnd=True
    
    self.ui.CanvasWidget.hide()
    self.ui.CanvasWidget.show()
    self.UpdateScreen()
    
    
    
    
    
    
###########################################################################
## 
###########################################################################        
def GenerateStatisticsBitmap(self):    
#    if(data.DisplayIsOn==False):
#        if(data.debug==1): print "-> o Do not Generate Statistics Bitmap, Display is off!"
#        return
#    else:
    if(data.debug==1): print "Generate Statistics Bitmap"

    try:
        data.StatisticsBitMap = QtGui.QPixmap(data.StatisticsBitMapLen,  data.StatisticsBitMapHeight) #create bitmap for graph   
        DrawStatistic(data.StatisticsBitMap,  QtGui) #draw statistics
    except:
        print "Error while creating the statistics bitmap"
    
    
#    self.ui.CanvasWidget.hide()
#    self.ui.CanvasWidget.show()
    
    
    
###########################################################################
main() # Now we can start it.
