#include <QDebug>
#include <QMaemo5InformationBox>
#include <QStandardItem>
#include <QDir>
#include <QDebug>
#include "expensedetailsdialog.h"
#include "scheduleddetailsdialog.h"
#include "budgetdialog.h"
#include "conceptsdialog.h"
#include "conceptdetailsdialog.h"
#include "expenseslist.h"
#include "controller.h"
#include "scheduledlist.h"
#include "confirmationdialog.h"
#include "summarywindow.h"
#include "datechooserdialog.h"

Controller::Controller()
{
        QString configFile(QDir::home().path());
        configFile.append(QDir::separator()).append(DATA_FOLDER);
        configFile.append(QDir::separator()).append(CONFIG_FILENAME);

        settings = new QSettings (configFile, QSettings::NativeFormat);

        concepts = new QStandardItemModel(0,0);
        loadConcepts();

        scheduled = new QStandardItemModel(0,7);
        loadScheduled();

        month = new MonthData(QDate::currentDate());
        loadMonth();

        mainWindow.setMonthData(month);

        connect(&mainWindow, SIGNAL(previousMonth()), this, SLOT(previousMonth()));
        connect(&mainWindow, SIGNAL(nextMonth()), this, SLOT(nextMonth()));
        connect(&mainWindow, SIGNAL(addExpenseSelected()), this, SLOT(newExpenseSelected()));
        connect(&mainWindow, SIGNAL(budgetSelected()), this, SLOT(budgetSelected()));
        connect(&mainWindow, SIGNAL(conceptsSelected()), this, SLOT(conceptsSelected()));
        connect(&mainWindow, SIGNAL(expensesSelected()), this, SLOT(expensesSelected()));
        connect(&mainWindow, SIGNAL(scheduledSelected()), this, SLOT(scheduledSelected()));
        connect(&mainWindow, SIGNAL(summarySelected()), this, SLOT(summarySelected()));
}

Controller::~Controller()
{
        delete month;
        delete concepts;
        delete settings;
}

void Controller::runApp()
{
        mainWindow.show();
}

void Controller::previousMonth()
{
        MonthData *old = month;
        month = new MonthData(old->previous());
        loadMonth();
        mainWindow.setMonthData(month);
        delete old;
}

void Controller::nextMonth()
{
        if (month->next() <= QDate::currentDate()) {
                MonthData *old = month;
                month = new MonthData(old->next());
                loadMonth();
                mainWindow.setMonthData(month);
                delete old;
        } else {
                QMaemo5InformationBox::information(&mainWindow, tr("Already at current month"));
        }
}

/* ------------------ add expense dialog functions ---------------------*/

void Controller::newExpenseSelected()
{
        ExpenseDetailsDialog dialog(month->date(), concepts, (QWidget*)QObject::sender());

        connect(&dialog, SIGNAL(newExpenseConfirmed(Expense*)), this, SLOT(newExpenseConfirmed(Expense*)));

        dialog.exec();
}

void Controller::newExpenseConfirmed(Expense *e)
{
        if (db.create(month->year(), month->month(), e)) {
                loadMonth(true);
        }
        delete e;
}

void Controller::expensesSelected()
{
        ExpensesList *list = new ExpensesList (month->expenses(), &mainWindow);

        connect(list, SIGNAL(newExpenseSelected()), this, SLOT(newExpenseSelected()));
        connect(list, SIGNAL(expenseDetailsSelected(QModelIndex)), this, SLOT(expenseDetailsSelected(QModelIndex)));

        list->show();

}

void Controller::expenseDetailsSelected(QModelIndex index)
{
        ExpenseDetailsDialog dialog(month->date(), concepts, month->expenses(), index.row(), (QWidget*)QObject::sender());

        connect(&dialog, SIGNAL(deleteExpenseConfirmed(Expense*)),
                this, SLOT(deleteExpenseConfirmed(Expense*)));
        connect(&dialog, SIGNAL(updateExpenseConfirmed(Expense*)),
                this, SLOT(updateExpenseConfirmed(Expense*)));

        dialog.exec();
}

void Controller::deleteExpenseConfirmed(Expense *e)
{
        if (db.remove(e)) {
                loadMonth(true);
        } else {
                QMaemo5InformationBox::information(&mainWindow, tr("Error deleting expense"));
        }

        delete e;
}

void Controller::updateExpenseConfirmed(Expense *e)
{
        if (db.update(e)) {
                loadMonth(true);
        } else {
                QMaemo5InformationBox::information(&mainWindow, tr("Error updating expense"));
        }

        delete e;
}

/* ------------------ budget dialog functions ---------------------*/

void Controller::budgetSelected()
{
        BudgetDialog dialog(month->budget(), &mainWindow);

        connect(&dialog, SIGNAL(changeBudget(double, bool)), this, SLOT(changeBudgetConfirmed(double, bool)));

        dialog.exec();
}

void Controller::changeBudgetConfirmed (double budget, bool setDefault)
{
        if (db.updateBudget(month->year(), month->month(), budget)) {
                month->setBudget(budget);
        }

        if (setDefault) {
                settings->setValue (DEFAULT_BUDGET_KEY, budget);
        }
}

/* ------------------ concept management dialog functions ---------------------*/

void Controller::conceptsSelected()
{
        ConceptsDialog dialog(concepts, &mainWindow);

        connect(&dialog, SIGNAL(newConceptSelected()), this, SLOT(newConceptSelected()));
        connect(&dialog, SIGNAL(conceptDetailsSelected(QModelIndex)), this, SLOT(conceptDetailsSelected(QModelIndex)));

        dialog.exec();
}

void Controller::newConceptSelected()
{
        ConceptDetailsDialog dialog(&mainWindow);

        connect(&dialog, SIGNAL(newConceptConfirmed(Concept*)), this, SLOT(newConceptConfirmed(Concept*)));

        dialog.exec();
}

void Controller::conceptDetailsSelected(const QModelIndex &index)
{
        ConceptDetailsDialog dialog(concepts, index.row(), &mainWindow);

        connect(&dialog, SIGNAL(deleteConceptConfirmed(Concept*)),
                this, SLOT(deleteConceptConfirmed(Concept*)));
        connect(&dialog, SIGNAL(updateConceptConfirmed(Concept*)),
                this, SLOT(updateConceptConfirmed(Concept*)));

        dialog.exec();
}

void Controller::newConceptConfirmed(Concept *c)
{
        if (!db.exists(c)) {
                if (db.create(c)) {
                        loadConcepts();
                } else {
                        QMaemo5InformationBox::information(&mainWindow, tr("Error creating concept"));
                }
        } else {
                QMaemo5InformationBox::information(&mainWindow, tr("Concept already exists"));
        }
        delete c;
}

void Controller::deleteConceptConfirmed(Concept *c)
{
        QString message(tr("When deleting a concept, all the expenses associated to it "
                           "will be assigned an empty concept instead.\n"
                           "Are you sure you want to delete the concept?"));
        if (ConfirmationDialog::askConfirmation(message)) {
                if (db.remove(c)) {
                        loadConcepts();
                        loadScheduled();
                        loadMonth(true);
                } else {
                        QMaemo5InformationBox::information(&mainWindow, tr("Error deleting concept"));
                }
        }
        delete c;
}

void Controller::updateConceptConfirmed(Concept *c)
{
        if (db.update(c)) {
                loadConcepts();
        } else {
                QMaemo5InformationBox::information(&mainWindow, tr("Error updating concept"));
        }

        delete c;
}

/* ------------------ scheduled expenses functions ---------------------*/

void Controller::scheduledSelected()
{
        ScheduledList *list = new ScheduledList (scheduled, &mainWindow);

        connect(list, SIGNAL(newScheduledSelected()), this, SLOT(newScheduledSelected()));
        connect(list, SIGNAL(scheduledDetailsSelected(QModelIndex)), this, SLOT(scheduledDetailsSelected(QModelIndex)));

        list->show();
}

void Controller::newScheduledSelected()
{
        ScheduledDetailsDialog dialog(concepts, (QWidget*)QObject::sender());

        connect(&dialog, SIGNAL(newScheduledConfirmed(Expense*)), this, SLOT(newScheduledConfirmed(Expense*)));

        dialog.exec();
}

void Controller::scheduledDetailsSelected(QModelIndex index)
{
        ScheduledDetailsDialog dialog(concepts, scheduled, index.row(), (QWidget*)QObject::sender());

        connect(&dialog, SIGNAL(deleteScheduledConfirmed(Expense*)),
                this, SLOT(deleteScheduledConfirmed(Expense*)));
        connect(&dialog, SIGNAL(updateScheduledConfirmed(Expense*)),
                this, SLOT(updateScheduledConfirmed(Expense*)));

        dialog.exec();
}

void Controller::newScheduledConfirmed(Expense *e)
{
        if (db.createScheduled(e)) {
                loadScheduled();
                if (e->frecuency() & (1 << QDate::currentDate().month() -1)) {
                        /* current month fits into the just created scheduled expense. Ask the user
                           if it should be applied to current month */
                        QString message(tr("Current month fits in the scheduled expense description.\n"
                                           "Do you want the expense to be created in it?"));
                        if (ConfirmationDialog::askConfirmation(message)) {
                                Expense *newExpense = new Expense (0, e->day(), e->amount(), e->concept(), e->description());
                                db.create(QDate::currentDate().year(), QDate::currentDate().month(), newExpense);
                                loadMonth(true);
                                delete newExpense;
                        }
                }
        } else {
                QMaemo5InformationBox::information(&mainWindow, tr("Error creating scheduled expense"));
        }

        delete e;
}

void Controller::deleteScheduledConfirmed(Expense *e)
{
        if (db.removeScheduled(e)) {
                loadScheduled();
        } else {
                QMaemo5InformationBox::information(&mainWindow, tr("Error deleting scheduled expense"));
        }

        delete e;
}

void Controller::updateScheduledConfirmed(Expense *e)
{
        /* store the previous frecuency value from the updated scheduled */
        QList<QStandardItem*> list = scheduled->findItems(QString("%1").arg(e->id()),
                                                          Qt::MatchFixedString,
                                                          0);
        int previousFrecuency = scheduled->item(list[0]->index().row(), 5)->text().toInt();

        if (db.updateScheduled(e)) {
                loadScheduled();
                if (e->frecuency() & (1 << QDate::currentDate().month() - 1)
                    && !(previousFrecuency & (1 << QDate::currentDate().month() - 1))) {
                        /* current month has been added to the scheduled expense in the modification. Ask the user
                           if it should be applied to current month */
                        QString message(tr("Current month fits in the scheduled expense description after the modification.\n"
                                           "Do you want the expense to be created in it?"));
                        if (ConfirmationDialog::askConfirmation(message)) {
                                Expense *newExpense = new Expense (0, e->day(), e->amount(), e->concept(), e->description());
                                db.create(QDate::currentDate().year(), QDate::currentDate().month(), newExpense);
                                loadMonth(true);
                                delete newExpense;
                        }
                }
        } else {
                QMaemo5InformationBox::information(&mainWindow, tr("Error updating scheduled expense"));
        }

        delete e;
}

/*---------------- query  funcions -------------*/

void Controller::summarySelected()
{
        int minYear, minMonth, maxYear, maxMonth;

        if (db.getDataInterval(minYear, minMonth, maxYear, maxMonth)) {
                summaryWindow = new SummaryWindow(minYear, minMonth, maxYear, maxMonth, &mainWindow);

                connect (summaryWindow, SIGNAL(periodChanged(int, int, int, int)),
                         this, SLOT(summaryPeriodChanged(int, int, int, int)));

                summaryWindow->show();
        } else {
                QMaemo5InformationBox::information(&mainWindow, tr("Error loading data interval"));
        }
}

void Controller::summaryPeriodChanged(int startYear, int startMonth, int endYear, int endMonth)
{
        SummaryData *data = new SummaryData(startYear, startMonth, endYear, endMonth);
        if (db.load(data)) {
                summaryWindow->setSummaryData(data);
        } else {
                QMaemo5InformationBox::information(&mainWindow, tr("Error loading interval data"));
                delete data;
        }
}

/*---------------- util funcions -------------*/

void Controller::loadMonth(bool reload)
{
        if (reload) {
                month->clear();
                db.load(month, true);
        } else {
                if (db.exists(month)) {
                        db.load (month, false);
                } else {
                        month->setBudget (settings->value(DEFAULT_BUDGET_KEY, DEFAULT_BUDGET).toDouble());
                        db.create(month);
                        /* create scheduled expenses in new month, but only if it is the current one */
                        if (month->isCurrent()) {
                                int i;
                                for (i = 0; i < scheduled->rowCount(); i++) {
                                        int frec = scheduled->item(i, 5)->text().toInt();
                                        if (frec & (1 << month->month() - 1)) {
                                                /* check if date is valid before creating the expense because of leap years */
                                                if (QDate::isValid(month->year(),
                                                                   month->month(),
                                                                   scheduled->item(i,1)->text().toInt())) {
                                                        Expense *e = new Expense(0,
                                                                                 scheduled->item(i, 1)->text().toInt(),
                                                                                 scheduled->item(i, 2)->text().toDouble(),
                                                                                 scheduled->item(i, 3)->text(),
                                                                                 scheduled->item(i, 4)->text());
                                                        db.create(month->year(), month->month(), e);
                                                        delete e;
                                                }
                                        }
                                }
                                db.load(month, true);
                        }
                }
        }
}

void Controller::loadConcepts()
{
        concepts->clear();
        db.loadConcepts(concepts);
}

void Controller::loadScheduled()
{
        scheduled->removeRows(0, scheduled->rowCount());
        db.loadScheduled(scheduled);
}
