#include "dbmanager.h"
#include "global.h"
#include <QtDebug>
#include <QSqlQuery>
#include <QVariant>
#include <QStandardItem>


DBManager::DBManager()
{
        initDB();
}

DBManager::~DBManager()
{
        closeDB();
}

void DBManager::initDB()
{
        bool create;

        QString dirPath(QDir::home().path());
        dirPath.append(QDir::separator()).append(DATA_FOLDER);

        QString filePath(dirPath);
        filePath.append(QDir::separator()).append(DB_FILENAME);

        create = !QDir::home().exists(filePath);

        if (!QDir::home().exists(dirPath)) {
                QDir::home().mkdir(dirPath);
        }

        db = QSqlDatabase::addDatabase("QSQLITE");
        db.setDatabaseName(filePath);
        openDB();

        if (create) {
                qDebug() << "\nDatabase empty. Creating tables";
                if (!createTables()) {
                        qCritical() << "\nError Initializing database! Exiting...";
                        exit(0);
                }
        }
}

bool DBManager::createTables()
{
        bool ret;
        QSqlQuery query;

        ret = query.exec("create table month (year integer not null, "
                         "month integer not null, "
                         "budget real not null, "
                         "primary key (year,month))");
        if (!ret) {
                qDebug() << "\nError creating table month" << query.lastError();
                return false;
        }

        ret = query.exec("create table concept ("
                         "name text primary key, "
                         "price real not null)");
        if (!ret) {
                qDebug() << "\nError creating table concept" << query.lastError();
                return false;
        }

        ret = query.exec("create table expense (id integer primary key autoincrement,"
                         "year integer, "
                         "month integer, "
                         "day integer, "
                         "amount real not null, "
                         "concept text references concept(name), "
                         "description text, "
                         "foreign key (year,month) references month(year,month))");

        if (!ret) {
                qDebug() << "\nError creating table expense" << query.lastError();
        }

        ret = query.exec("create table scheduled (id integer primary key autoincrement,"
                         "day integer, "
                         "amount real not null, "
                         "concept text references concept(name), "
                         "description text, "
                         "frecuency integer)");

        if (!ret) {
                qDebug() << "\nError creating table scheduled" << query.lastError();
        }

        return ret;
}

/*----------- MonthData operations -------------*/

bool DBManager::create(MonthData *month)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec (QString("insert into month (year, month, budget) "
                                  "values (%1, %2, %3)")
                          .arg(month->year())
                          .arg(month->month())
                          .arg(month->budget()));

        if (!ret) {
                qDebug() << "\nError adding month to database";
        }
        return ret;
}

bool DBManager::load(MonthData *month, bool expensesOnly)
{
        bool ret;
        QSqlQuery query;

        if (!expensesOnly) {
                ret = query.exec(QString ("select budget from month where year=%1 and month=%2")
                                 .arg(month->year())
                                 .arg(month->month()));
                if (!ret) {
                        qDebug() << "\nError loading month budget from database" << query.lastError();
                        return ret;
                }

                if (query.next()) {
                        month->setBudget(query.value(0).toDouble());
                } else {
                        qDebug() << "\nError: Selected month doesn't exist";
                        return false;
                }
        }

        ret = query.exec(QString ("select id, day, amount, concept, description from expense "
                                  "where year=%1 and month=%2 order by day")
                         .arg(month->year())
                         .arg(month->month()));

        if (!ret) {
                qDebug() << "\nError loading month expenses from database" << query.lastError();
                return ret;
        }

        while (query.next()) {
                month->addExpense(new Expense(query.value(0).toInt(),
                                              query.value(1).toInt(),
                                              query.value(2).toDouble(),
                                              query.value(3).toString(),
                                              query.value(4).toString()));
        }

        return true;
}

bool DBManager::exists(MonthData *month)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("select * from month where year=%1 and month=%2")
                         .arg(month->year())
                         .arg(month->month()));
        if (!ret) {
                qDebug() << "\nError accessing database" << query.lastError();
                return ret;
        }

        return query.next();
}

/*----------- Expense operations -------------*/

bool DBManager::create(int year, int month, Expense *e)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("insert into expense (id, year, month, day, amount, concept, description) "
                                  "values (NULL, %1, %2, %3, %4, '%5', '%6')")
                         .arg(year)
                         .arg(month)
                         .arg(e->day())
                         .arg(e->amount())
                         .arg(e->concept())
                         .arg(e->description()));

        if (!ret) {
                qDebug() << "\nError adding expense to the database" << query.lastError();
        }

        return ret;
}

bool DBManager::update(Expense *e)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("update expense set day=%1, amount=%2, concept='%3',"
                                  "description='%4' where id=%5")
                         .arg(e->day())
                         .arg(e->amount())
                         .arg(e->concept())
                         .arg(e->description())
                         .arg(e->id()));

        if (!ret) {
                qDebug() << "\nError updating expense in database" << query.lastError();
        }
        return ret;
}

bool DBManager::remove(Expense *e)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("delete from expense where id=%1")
                         .arg(e->id()));

        if (!ret) {
                qDebug() << "\nError deleting expense from database" << query.lastError();
        }

        return ret;
}

/*----------- Concept operations -------------*/

bool DBManager::create(Concept *c)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("insert into concept (name,price) "
                                  "values ('%1', %2)")
                         .arg(c->name())
                         .arg(c->price()));

        if (!ret) {
                qDebug() << "\nError adding concept to the database" << query.lastError();
        }
        return ret;
}

bool DBManager::loadConcepts(QStandardItemModel *concepts)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("select name,price from concept order by name collate nocase"));

        if (!ret) {
                qDebug() << "\nError loading concepts from database" << query.lastError();
                return ret;
        }

        while (query.next()) {
                QList<QStandardItem*> row;
                QStandardItem *item;
                item = new QStandardItem (query.value(0).toString());
                item->setTextAlignment(Qt::AlignCenter);
                item->setEditable(false);
                row.append(item);
                item = new QStandardItem (query.value(1).toString());
                item->setTextAlignment(Qt::AlignCenter);
                item->setEditable(false);
                row.append(item);
                concepts->appendRow(row);
        }

        return true;
}

bool DBManager::update(Concept *c)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("update concept set price=%1 where name='%2'")
                         .arg(c->price())
                         .arg(c->name()));

        if (!ret) {
                qDebug() << "\nError updating concept in database" << query.lastError();
        }
        return ret;
}

bool DBManager::remove(Concept *c)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("delete from concept where name='%1'")
                         .arg(c->name()));

        if (!ret) {
                qDebug() << "\nError deleting concept from database" << query.lastError();
                return ret;
        }

        ret = query.exec(QString ("update expense set concept='' where concept='%1'")
                         .arg(c->name()));

        if (!ret) {
                qDebug() << "\nError disassociating expenses from removed concept" << query.lastError();
        }

        ret = query.exec(QString ("update scheduled set concept='' where concept='%1'")
                         .arg(c->name()));

        if (!ret) {
                qDebug() << "\nError disassociating scheduled expenses from removed concept" << query.lastError();
        }

        return ret;
}

bool DBManager::exists(Concept *c)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("select * from concept where name='%1'")
                         .arg(c->name()));
        if (!ret) {
                qDebug() << "\nError accessing database" << query.lastError();
                return ret;
        }

        return query.next();
}

/*----------- Budget operations -------------*/

bool DBManager::updateBudget(int year, int month, double budget)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("update month set budget=%1 where year=%2 and month=%3")
                         .arg(budget)
                         .arg(year)
                         .arg(month));
        if (!ret) {
                qDebug() << "\nError saving month budget to database" << query.lastError();
                return ret;
        }

        return true;
}

/*----------- Scheduled expense operations -------------*/

bool DBManager::createScheduled(Expense *e)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("insert into scheduled (id, day, amount, concept, description, frecuency) "
                                  "values (NULL, %1, %2, '%3', '%4', %5)")
                         .arg(e->day())
                         .arg(e->amount())
                         .arg(e->concept())
                         .arg(e->description())
                         .arg(e->frecuency()));

        if (!ret) {
                qDebug() << "\nError adding scheduled expense to the database" << query.lastError();
        }

        return ret;
}

bool DBManager::removeScheduled(Expense *e)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("delete from scheduled where id=%1")
                         .arg(e->id()));

        if (!ret) {
                qDebug() << "\nError deleting scheduled expense from database" << query.lastError();
        }

        return ret;
}

bool DBManager::loadScheduled(QStandardItemModel *scheduled)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("select id,day,amount,concept,description,frecuency "
                                  "from scheduled order by day"));

        if (!ret) {
                qDebug() << "\nError loading scheduled expenses from database" << query.lastError();
                return ret;
        }

        while (query.next()) {
                QList<QStandardItem*> row;
                QStandardItem *item;
                item = new QStandardItem (query.value(0).toString());
                item->setTextAlignment(Qt::AlignCenter);
                item->setEditable(false);
                row.append(item);
                item = new QStandardItem (query.value(1).toString());
                item->setTextAlignment(Qt::AlignCenter);
                item->setEditable(false);
                row.append(item);
                item = new QStandardItem (query.value(2).toString());
                item->setTextAlignment(Qt::AlignCenter);
                item->setEditable(false);
                row.append(item);
                item = new QStandardItem (query.value(3).toString());
                item->setTextAlignment(Qt::AlignLeft | Qt::AlignVCenter);
                item->setEditable(false);
                row.append(item);
                item = new QStandardItem (query.value(4).toString());
                item->setTextAlignment(Qt::AlignLeft | Qt::AlignVCenter);
                item->setEditable(false);
                row.append(item);
                item = new QStandardItem (query.value(5).toString());
                item->setTextAlignment(Qt::AlignCenter);
                item->setEditable(false);
                row.append(item);
                item = new QStandardItem (Expense::frecuencyAsString(query.value(5).toInt()));
                item->setTextAlignment(Qt::AlignLeft | Qt::AlignVCenter);
                item->setEditable(false);
                row.append(item);
                scheduled->appendRow(row);
        }

        return true;
}

bool DBManager::updateScheduled(Expense *e)
{
        bool ret;
        QSqlQuery query;

        ret = query.exec(QString ("update scheduled set day=%1, amount=%2, concept='%3', "
                                  "description='%4', frecuency=%5 where id=%6")
                         .arg(e->day())
                         .arg(e->amount())
                         .arg(e->concept())
                         .arg(e->description())
                         .arg(e->frecuency())
                         .arg(e->id()));

        if (!ret) {
                qDebug() << "\nError updating scheduled in database" << query.lastError();
        }
        return ret;
}

bool DBManager::openDB()
{
        return db.open();
}

void DBManager::closeDB()
{
        db.close();
}
