#include "filelist.h"
#include "videofile.h"
#include "Settings.h"
#include <QDir>
#include <iostream>
#include <QtAlgorithms>
#include <QProgressDialog>
#include "filesearchdialog.h"
#include <QCoreApplication>

using namespace std;

FileList::FileList( QObject* parent ) :
    QObject( parent )
{
    loadFileList();

    m_knownExtensions << "avi" << "mpg" << "mpeg" << "mkv"
                      << "ogm" << "asf" << "wmv" << "divx"
                      << "mov" << "mp4" << "vob" << "rm"
                      << "flv" << "ogv" << "rmvb";

    m_sortCriteria = ESortByResumePos;
    m_sortOrder = Qt::DescendingOrder;

    sortFileList( m_sortCriteria, m_sortOrder );
}

FileList::~FileList()
{
    saveFileList();
}

void FileList::searchFiles( const QList<QString>& searchFolders, FileSearchDialog& progressDialog )
{
    QFileInfoList files;

    for ( int i = 0; i < searchFolders.count(); i++ ) {

        progressDialog.setValue(i);

        QDir dir( searchFolders[i] );
        getFilesInDir( dir, files, progressDialog );
    }

    updateFileList( files );

    sortFileList( m_sortCriteria, m_sortOrder );
}

const QList<VideoFile*>& FileList::files()
{
    return m_files;
}

VideoFile* FileList::file( int idx ) {
    return m_files[idx];
}

// ------------------------------------------------------------------------------
// Sorts the files on given criteria and order.
// ------------------------------------------------------------------------------
//
void FileList::sortFileList( SortCriteria criteria, Qt::SortOrder order )
{
    m_sortCriteria = criteria;
    m_sortOrder = order;

    if ( criteria == ESortByName ) {
        if ( order == Qt::AscendingOrder ) {
            qSort( m_files.begin(), m_files.end(), VideoFile::nameAscCmp );
        } else {
            qSort( m_files.begin(), m_files.end(), VideoFile::nameDesCmp );
        }
    } else if ( criteria == ESortBySize ) {
        if ( order == Qt::AscendingOrder ) {
            qSort( m_files.begin(), m_files.end(), VideoFile::sizeAscCmp );
        } else {
            qSort( m_files.begin(), m_files.end(), VideoFile::sizeDesCmp );
        }
    } else if ( criteria == ESortByResumePos ) {
        if ( order == Qt::AscendingOrder ) {
            qSort( m_files.begin(), m_files.end(), VideoFile::resumeAscCmp );
        } else {
            qSort( m_files.begin(), m_files.end(), VideoFile::resumeDesCmp );
        }
    }
}

void FileList::updateFileList( const QFileInfoList& files )
{
    for ( int i = 0; i < files.count(); i++ ) {
        bool append = true; // check first if m_list already contains this file
        for ( int j = 0; j < m_files.count(); j++ ) {
            if ( m_files[j]->file() == files[i] ) {
                append = false;
                break;
            }
        }
        if ( append ) {
            m_files.append( new VideoFile( files[i], this ) );
        }
    }

    // check if there are files in m_files that are not in files and remove them
    for ( int i = m_files.count() - 1; i >= 0; i-- ) {
        bool found = false;
        for ( int j = 0; j < files.count(); j++ ) {
            if ( files[j] == m_files[i]->file() ) {
                found = true;
                break;
            }
        }
        if ( !found ) {
            m_files.removeAt( i );
        }
    }
}

// ------------------------------------------------------------------------------
// Searches files recursively in given dir.
// ------------------------------------------------------------------------------
//
void FileList::getFilesInDir( const QDir& dir, QFileInfoList& fileList, FileSearchDialog& progress )
{
    QCoreApplication::instance()->processEvents();
    QFileInfoList tempList = dir.entryInfoList();

    for ( int i = 0; i < tempList.count(); i++ ) {
        if ( progress.wasCanceled() )
        {
            return;
        }

        if ( tempList[i].isFile() && m_knownExtensions.contains( tempList[i].suffix(), Qt::CaseInsensitive ) ) {
            fileList.append( tempList[i] );
        } else if (
            tempList[i].isDir() &&
            tempList[i].fileName() != "." &&
            tempList[i].fileName() != ".." &&
            !tempList[i].fileName().isEmpty() )
        {
            getFilesInDir( QDir( tempList[i].absoluteFilePath() ), fileList, progress );
        }
    }
}

void FileList::loadFileList()
{
    QSettings settings;
    int size =  settings.beginReadArray( PREFIX_FILES );

    for ( int i = 0; i < size; i++ ) {
        settings.setArrayIndex( i );

        QFileInfo fileinfo(  settings.value( KEY_FILENAME ).toString() );
        unsigned int resumePos = settings.value( KEY_FILEPOS ).toUInt();
        m_files.append( new VideoFile( fileinfo, resumePos, this ) );
    }
    settings.endArray();
}

void FileList::saveFileList()
{
    QSettings settings;
    settings.remove( PREFIX_FILES );
    settings.beginWriteArray( PREFIX_FILES );
    for ( int i = 0; i < m_files.count(); i++ ) {
    	if ( m_files[i]->resumePos() > 0 ) // only store files that have a resumepos
    	{
            settings.setArrayIndex( i );
            settings.setValue( KEY_FILENAME, m_files[i]->file().absoluteFilePath() );
            settings.setValue( KEY_FILEPOS, m_files[i]->resumePos() );
    	}
    }
    settings.endArray();
}

