#include <QFile>
#include <QTime>
#include <QDebug>

#include "settings.h"

QMap<QString, QStringList> SettingsWindow::_settingsMap;

SettingsWindow::SettingsWindow(QMainWindow *parent) : QMainWindow(parent), BINPATH("/opt/showtime/bin/"), ICONPATH("/opt/showtime/icons/")
{
}

void SettingsWindow::loadSettings()
{
	// clear previous settings
	_settingsMap.clear();

	QFile file(QString(BINPATH) + ".appsetting");
	file.open(QIODevice::WriteOnly | QIODevice::ReadOnly);

	QTextStream t(&file);

	while( !t.atEnd() )
	{
		QStringList line = t.readLine().split('=');
		
		QString param = line[0];
		QString value = line[1];

		// favourites are in the format UKFAV(n)=bbc1 where n=1,2,3...n
		// the region is also stored REGION=UK
		if ( param.size() != 0 && value.size() != 0)
		{
			// if value is CH1#Ch2#CH3 the split breaks it to "CH1","CH2", "CH3" as a string list	
			// for display purposes
			_settingsMap[param] = value.split('#', QString::SkipEmptyParts);
		}
	}
}

void SettingsWindow::updateSettings()
{
	QString filename = QString(BINPATH) + ".appsetting";

	QFile::remove(filename);
	QFile file(filename);

	file.open(QIODevice::WriteOnly | QIODevice::ReadOnly);

	QTextStream t(&file);

	foreach(QString key, _settingsMap.keys())
	{
		// dont write in empty values
		if ( _settingsMap[key].isEmpty() )
			continue;

		// favourites are to be stored in the format UKFAV=bbc1#ch4#sky1
		t << key << "=";
		foreach(QString value, _settingsMap[key])
		{
			if ( key.contains("FAV") || key.contains("SEARCH") )
				t << value << "#";
			else
				t << value;
		}

		t << endl;
	}
	
	file.close();
}

///////////////////############### GETTERS ##################//////////////////////
QString SettingsWindow::getRegion()
{
	if(_settingsMap.find("REGION") != _settingsMap.end() )
		return _settingsMap["REGION"].back();
	else
		return "UK"; // default region
}

QString SettingsWindow::getRemindType()
{
	if(_settingsMap.find("REMINDER") != _settingsMap.end() )
		return _settingsMap["REMINDER"].back();
	else
		return "ALARM"; // default region
}

QString SettingsWindow::getOrientation()
{
	if(_settingsMap.find("ORIENTATION") != _settingsMap.end() )
		return _settingsMap["ORIENTATION"].back();
	else
		return "AUTO"; // default region
}

QString SettingsWindow::getIconType()
{
	if(_settingsMap.find("ICONS") != _settingsMap.end() )
		return _settingsMap["ICONS"].back();
	else
		return "SET1"; // default region
}

const QStringList SettingsWindow::getFavourites(QString region) const
{
	if(_settingsMap.find(region+"FAV") != _settingsMap.end() )
		return _settingsMap[region+"FAV"];
	else
		return QStringList(); // default no favourites
}

QStringList SettingsWindow::getFavourites(QString region) 
{
	if(_settingsMap.find(region+"FAV") != _settingsMap.end() )
		return _settingsMap[region+"FAV"];
	else
		return QStringList(); // default no favourites
}

QStringList SettingsWindow::getSearchTitles(QString region, QString channel) 
{
	if(_settingsMap.find(region+channel+"SEARCH") != _settingsMap.end() )
		return _settingsMap[region+channel+"SEARCH"];
	else
		return QStringList(); // default no search titles
}

QString SettingsWindow::getShowPastInd()
{
	if(_settingsMap.find("SHOWPAST") != _settingsMap.end() )
		return _settingsMap["SHOWPAST"].back();
	else
		return "Y"; // default show the past listings also
}

QString SettingsWindow::getPrimeTimeStart()
{
	if(_settingsMap.find("START_PRIMETIME") != _settingsMap.end() )
		return _settingsMap["START_PRIMETIME"].back();
	else
		return "00:00";
}

QString SettingsWindow::getPrimeTimeEnd()
{
	if(_settingsMap.find("END_PRIMETIME") != _settingsMap.end() )
		return _settingsMap["END_PRIMETIME"].back();
	else
		return "23:59";
}

const QString SettingsWindow::getIconPath() 
{
	return QString(ICONPATH + getIconType() + "/");
}

const QString SettingsWindow::getBinPath() 
{
	return QString(BINPATH);
}

QMap<QString, QStringList>& SettingsWindow::getSettingsMap()
{
	return _settingsMap;
}

///////////////////############### SETTERS ##################//////////////////////
void SettingsWindow::setRegion(QString region)
{
	_settingsMap["REGION"] = QStringList(region);
}

void SettingsWindow::setRemindType(QString remindType)
{
	_settingsMap["REMINDER"] = QStringList(remindType);
}

void SettingsWindow::setOrientation(QString orientation)
{
	_settingsMap["ORIENTATION"] = QStringList(orientation);
}

void SettingsWindow::setIconType(QString iconset)
{
	_settingsMap["ICONS"] = QStringList(iconset);
}

void SettingsWindow::setFavourites(QString region, QString channel) 
{
	_settingsMap[region+"FAV"] << channel;
}

void SettingsWindow::removeFromFavourite(QString region, QString channel) 
{
	_settingsMap[region+"FAV"].removeAll(channel);
}

void SettingsWindow::setSearchTitles(QString region, QString channel, QString searchTitle) 
{
	_settingsMap[region+channel+"SEARCH"] << searchTitle;
}

void SettingsWindow::removeFromSearchTitles(QString region, QString channel, QString searchTitle)
{
	_settingsMap[region+channel+"SEARCH"].removeAll(searchTitle);
}

void SettingsWindow::setShowPastInd(QString indicator)
{
	_settingsMap["SHOWPAST"] = QStringList(indicator);
}

void SettingsWindow::setPrimeTimeStart(QString time)
{
	_settingsMap["START_PRIMETIME"] = QStringList(time);
}

void SettingsWindow::setPrimeTimeEnd(QString time)
{
	_settingsMap["END_PRIMETIME"] = QStringList(time);
}

SettingsWindow::~SettingsWindow()
{
}
